<?php
if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;
$table_name = $wpdb->prefix . 'vtupress_internal_pins';

// AUTO-MIGRATION: Check and Add 'value' column if missing
$col_check = $wpdb->get_results("SHOW COLUMNS FROM $table_name LIKE 'value'");
if (empty($col_check)) {
    $wpdb->query("ALTER TABLE $table_name ADD COLUMN value VARCHAR(50) DEFAULT '' AFTER plan_id");
}

// AUTO-MIGRATION: Check and Add 'price' column if missing
$col_check_price = $wpdb->get_results("SHOW COLUMNS FROM $table_name LIKE 'price'");
if (empty($col_check_price)) {
    $wpdb->query("ALTER TABLE $table_name ADD COLUMN price DECIMAL(10,2) DEFAULT 0.00 AFTER value");
}


$current_country = Vtupress_Country_Context::get();
$all_countries = Vtupress_Country_Manager::all();
$cdata = $all_countries[$current_country] ?? [];
$voucher_services = vtupress_get_voucher_services();

// Prepare Providers Data for JS
$providers_data = [
    'recharge_card' => $cdata['recharge_card'] ?? [],
    'data_card' => $cdata['data_card'] ?? [],
    'education' => $cdata['education'] ?? []
];


// Handle Form Submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    
    // 1. Upload PINs
    if (isset($_POST['action']) && $_POST['action'] === 'upload_pins') {
        if (!wp_verify_nonce($_POST['_wpnonce'], 'upload_pins_action')) {
            die('Security check failed');
        }

        $service = sanitize_text_field($_POST['service']);
        $plan = sanitize_text_field($_POST['plan']);
        $value = sanitize_text_field($_POST['value']);
        $price = floatval($_POST['price']); // New Price Field
        $type = sanitize_text_field($_POST['type']); // New Type Field
        $pins_data = sanitize_textarea_field($_POST['pins_data']);
        $country = sanitize_text_field($_POST['country'] ?? $country);

        $lines = explode("\n", $pins_data);
        $count = 0;
        $ignored = 0;
        
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line))
                continue;

            // Format: PIN,SERIAL (optional)
            $parts = explode(',', $line);
            $pin = trim($parts[0]);
            $serial = isset($parts[1]) ? trim($parts[1]) : '';

            // DUPLICATE CHECK
            $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table_name WHERE pin = %s AND set_country = '$country'", $pin));
            if ($exists) {
                $ignored++;
                continue;
            }

            $wpdb->insert($table_name, [
                'service_id' => $service,
                'plan_id' => $plan,
                'value' => $value,
                'price' => $price,
                'type' => $type,
                'pin' => $pin,
                'serial' => $serial,
                'status' => 'unused',
                'set_country' => $country
            ]);
            $count++;
        }
        
        $msg = "Successfully uploaded $count PINs.";
        if ($ignored > 0) {
            $msg .= " Ignored $ignored duplicate PINs.";
        }
        echo "<div class='alert alert-success'>$msg</div>";
    } 
    
    // 2. Bulk Delete
    elseif (isset($_POST['bulk_action']) && $_POST['bulk_action'] === 'delete_selected') {
        check_admin_referer('bulk_pin_action');
        if (!empty($_POST['pin_ids']) && is_array($_POST['pin_ids'])) {
            $ids = array_map('intval', $_POST['pin_ids']);
            $ids_sql = implode(',', $ids);
            $wpdb->query("DELETE FROM $table_name WHERE id IN ($ids_sql)");
            echo "<div class='alert alert-success'>Deleted " . count($ids) . " PINs.</div>";
        }
    }
    
    // 3. Bulk Update Price
    elseif (isset($_POST['bulk_action']) && $_POST['bulk_action'] === 'update_price') {
        check_admin_referer('bulk_pin_action');
        if (!empty($_POST['pin_ids']) && is_array($_POST['pin_ids']) && isset($_POST['new_bulk_price'])) {
            $ids = array_map('intval', $_POST['pin_ids']);
            $price = floatval($_POST['new_bulk_price']);
            $ids_sql = implode(',', $ids);
            $wpdb->query($wpdb->prepare("UPDATE $table_name SET price = %f WHERE id IN ($ids_sql)", $price));
            echo "<div class='alert alert-success'>Updated price for " . count($ids) . " PINs.</div>";
        }
    }

    // 4. Single Delete
    elseif (isset($_POST['delete_single_id'])) {
         check_admin_referer('bulk_pin_action');
         $pin_id = intval($_POST['delete_single_id']);
         $wpdb->delete($table_name, ['id' => $pin_id]);
         echo "<div class='alert alert-success'>PIN Deleted.</div>";
    }
}

// Pagination & Filtering
$limit = 20;
$page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
$offset = ($page - 1) * $limit;

$total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE set_country = '$country'");
$pins = $wpdb->get_results("SELECT * FROM $table_name  WHERE set_country = '$country' ORDER BY id DESC LIMIT $limit OFFSET $offset");
$total_pages = ceil($total_items / $limit);

?>

<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-body">
                <h4 class="card-title">PIN Manager</h4>
                <p class="card-description">Upload and Manage Manual Service PINs (Current Country:
                    <strong><?php echo strtoupper($current_country); ?></strong>)
                </p>

                <!-- Upload Form -->
                <button class="btn btn-primary mb-3" type="button" data-bs-toggle="collapse"
                    data-bs-target="#uploadForm" aria-expanded="false" aria-controls="uploadForm">
                    <i class="mdi mdi-upload"></i> Upload New PINs
                </button>

                <div class="collapse mb-4" id="uploadForm">
                    <div class="card card-body border">
                        <form method="post">
                            <?php wp_nonce_field('upload_pins_action'); ?>
                            <input type="hidden" name="action" value="upload_pins">
                            <input type="hidden" name="country" value="<?php echo esc_attr($current_country); ?>">

                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Service</label>
                                    <select name="service" id="service-select" class="form-select" required>
                                        <option value="">-- Select Service --</option>
                                        <?php foreach ($voucher_services as $vid => $vdata): ?>
                                            <option value="<?php echo esc_attr($vid); ?>">
                                                <?php echo esc_html($vdata['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Service Type</label>
                                    <select name="type" id="type-select" class="form-select" required>
                                        <option value="">-- Select Type --</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Provider / Exam Body</label>
                                    <select name="plan" id="plan-select" class="form-select" required>
                                        <option value="">-- Select Provider --</option>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">Value / Denomination</label>
                                    <input type="text" name="value" class="form-control"
                                        placeholder="100, 200, 1GB, WAEC..." required>
                                    <small class="text-muted">E.g., 100 for N100 card, 1GB for Data.</small>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Price (₦)</label>
                                    <input type="number" step="0.01" name="price" class="form-control"
                                        placeholder="0.00" required>
                                    <small class="text-muted">The cost deducted from user wallet.</small>
                                </div>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Upload CSV (Optional)</label>
                                <div class="input-group">
                                    <input type="file" id="pinCsvFile" class="form-control" accept=".csv, .txt">
                                    <button class="btn btn-secondary" type="button" id="processCsvBtn">Fill from
                                        CSV</button>
                                </div>
                                <small class="text-muted">Touching "Fill from CSV" will populate the box below. CSV
                                    Format: <code>PIN, SERIAL</code> (First 2 columns).</small>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">PINs Data</label>
                                <textarea name="pins_data" id="pins_data_area" class="form-control" rows="5"
                                    placeholder="Enter one PIN per line.&#10;Format: PIN,SERIAL (Serial is optional)&#10;Example:&#10;123456789012,SN001&#10;987654321098"
                                    required></textarea>
                            </div>

                            <script>
                                document.getElementById('processCsvBtn').addEventListener('click', function () {
                                    var fileInput = document.getElementById('pinCsvFile');
                                    var textArea = document.getElementById('pins_data_area');

                                    if (fileInput.files.length === 0) {
                                        alert('Please select a CSV file first.');
                                        return;
                                    }

                                    var file = fileInput.files[0];
                                    var reader = new FileReader();

                                    reader.onload = function (e) {
                                        var text = e.target.result;
                                        var lines = text.split(/\r\n|\n/);
                                        var output = [];

                                        lines.forEach(function (line) {
                                            if (!line.trim()) return;
                                            var parts = line.split(',');
                                            var pin = parts[0] ? parts[0].trim() : '';
                                            var serial = parts[1] ? parts[1].trim() : '';

                                            // Handle potential header row (skip if "pin" or "serial" literal)
                                            if (pin.toLowerCase().includes('pin') && (serial.toLowerCase().includes('serial') || !serial)) {
                                                return;
                                            }

                                            if (pin) {
                                                output.push(pin + (serial ? ',' + serial : ''));
                                            }
                                        });

                                        if (output.length > 0) {
                                            var currentVal = textArea.value.trim();
                                            textArea.value = (currentVal ? currentVal + '\n' : '') + output.join('\n');
                                            alert('Populated ' + output.length + ' PINs from CSV.');
                                        } else {
                                            alert('No valid PINs found in CSV.');
                                        }
                                    };

                                    reader.onerror = function () {
                                        alert('Error reading file.');
                                    };

                                    reader.readAsText(file);
                                });
                            </script>

                            <button type="submit" class="btn btn-success">Upload Results</button>
                        </form>
                    </div>
                </div>

                <!-- Bulk Actions & Table -->
                <form method="post" id="bulk-pin-form">
                    <?php wp_nonce_field('bulk_pin_action'); ?>
                    <input type="hidden" name="bulk_action" id="bulk_action_input" value="">

                    <!-- Bulk Controls -->
                    <div id="bulk-controls" class="d-flex align-items-center mb-3 flex-wrap gap-2" style="display:none !important;">
                        <button type="button" class="btn btn-danger btn-sm text-white" onclick="submitBulk('delete_selected')">
                            <i class="mdi mdi-delete"></i> Delete Selected
                        </button>
                        
                        <div class="input-group input-group-sm" style="max-width: 250px; margin-left: 10px;">
                            <span class="input-group-text">₦</span>
                            <input type="number" step="0.01" name="new_bulk_price" class="form-control" placeholder="New Price">
                            <button class="btn btn-info text-white" type="button" onclick="submitBulk('update_price')">
                                Update Price
                            </button>
                        </div>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="select-all-pins"></th>
                                    <th>ID</th>
                                    <th>Service</th>
                                    <th>Provider</th>
                                    <th>Type</th>
                                    <th>Value</th>
                                    <th>Price</th>
                                    <th>PIN</th>
                                    <th>Serial</th>
                                    <th>Status</th>
                                    <th>Country</th>
                                    <th>Date</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($pins): ?>
                                    <?php foreach ($pins as $pin): ?>
                                        <tr>
                                            <td>
                                                <input type="checkbox" name="pin_ids[]" value="<?php echo $pin->id; ?>" class="pin-checkbox">
                                            </td>
                                            <td><?php echo $pin->id; ?></td>
                                            <td><?php echo strtoupper(str_replace('_', ' ', $pin->service_id)); ?></td>
                                            <td><?php echo strtoupper($pin->plan_id); ?></td>
                                            <td><?php echo strtoupper($pin->type); ?></td>
                                            <td><?php echo esc_html($pin->value); ?></td>
                                            <td>₦<?php echo number_format($pin->price, 2); ?></td>
                                            <td><?php echo esc_html($pin->pin); ?></td>
                                            <td><?php echo esc_html($pin->serial); ?></td>
                                            <td>
                                                <span
                                                    class="badge <?php echo $pin->status == 'unused' ? 'badge-success' : 'badge-danger'; ?>">
                                                    <?php echo ucfirst($pin->status); ?>
                                                </span>
                                            </td>
                                            <td><?php echo strtoupper($pin->set_country); ?></td>
                                            <td><?php echo $pin->created_at; ?></td>
                                            <td>
                                                <button type="submit" name="delete_single_id" value="<?php echo $pin->id; ?>" 
                                                        class="btn btn-danger btn-sm p-1" 
                                                        onclick="return confirm('Are you sure you want to delete this PIN?')">
                                                    <i class="mdi mdi-delete"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="13" class="text-center">No PINs found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
                
                <script>
                    function toggleBulkActions() {
                        var checked = document.querySelectorAll('.pin-checkbox:checked').length;
                        var controls = document.getElementById('bulk-controls');
                        if (checked > 0) {
                            controls.style.display = 'flex'; // Use flex to maintain layout
                            controls.style.setProperty('display', 'flex', 'important');
                        } else {
                            controls.style.display = 'none';
                            controls.style.setProperty('display', 'none', 'important');
                        }
                    }

                    // Select All Logic
                    document.getElementById('select-all-pins').addEventListener('change', function() {
                        var checkboxes = document.querySelectorAll('.pin-checkbox');
                        for(var checkbox of checkboxes) {
                            checkbox.checked = this.checked;
                        }
                        toggleBulkActions();
                    });

                    // Individual Checkbox Logic
                    var checkboxes = document.querySelectorAll('.pin-checkbox');
                    checkboxes.forEach(function(checkbox) {
                        checkbox.addEventListener('change', toggleBulkActions);
                    });

                    // Bulk Submit Logic
                    function submitBulk(action) {
                        var checked = document.querySelectorAll('.pin-checkbox:checked').length;
                        if(checked === 0) {
                            alert('Please select at least one PIN.');
                            return;
                        }
                        
                        if(action === 'delete_selected' && !confirm('Are you sure you want to DELETE ' + checked + ' pins?')) {
                            return;
                        }
                        
                        document.getElementById('bulk_action_input').value = action;
                        document.getElementById('bulk-pin-form').submit();
                    }
                </script>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <nav class="mt-4">
                        <ul class="pagination">
                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                    <a class="page-link"
                                        href="?page=vtupanel&adminpage=pin_manager&paged=<?php echo $i; ?>"><?php echo $i; ?></a>
                                </li>
                            <?php endfor; ?>
                        </ul>
                    </nav>
                <?php endif; ?>

            </div>
        </div>
    </div>
</div>

<script>
    var providersData = <?php echo json_encode($providers_data); ?>;
    var servicesData = <?php echo json_encode($voucher_services); ?>;

    jQuery('#service-select').change(function () {
        var service = jQuery(this).val();
        var planSelect = jQuery('#plan-select');
        var typeSelect = jQuery('#type-select');

        // Reset
        planSelect.empty().append('<option value="">-- Select Provider --</option>');
        typeSelect.empty().append('<option value="">-- Select Type --</option>');

        // Populate Types
        if (service && servicesData[service] && servicesData[service].types) {
            servicesData[service].types.forEach(function (t) {
                typeSelect.append('<option value="' + t + '">' + t.toUpperCase() + '</option>');
            });
            // Select first if only one
            if (servicesData[service].types.length === 1) {
                typeSelect.val(servicesData[service].types[0]);
            }
        }

        // Populate Providers (Logic remains Country-based for now as per prev implementation)
        // Adjust logical mapping: user wants service dropdown to be dynamic. 
        // providersData currently maps 'recharge_card' -> [...]. If new service is added, it needs to be in providersData?
        // Ideally providersData should also be dynamic, but for now we follow existing object structure.

        if (service && providersData[service]) {
            providersData[service].forEach(function (prov) {
                var label = prov.toUpperCase();
                planSelect.append('<option value="' + prov + '">' + label + '</option>');
            });
        }
    });
</script>