<?php
if (!defined('ABSPATH')) {
    die("Not Allowed");
}


class Vtupress_Option
{
    protected static function prefix(): string
    {
        $country = Vtupress::$static_country;

        return $country . '_';
    }

    public static function get(string $key, $default = null, $specific_country = false): mixed
    {
        $key = $specific_country ? self::prefix() . $key : $key;

        return is_multisite()
            ? get_site_option($key, $default)
            : get_option($key, $default);
    }

    public static function update(string $key, $value, $specific_country = false)
    {
        $key = $specific_country ? self::prefix() . $key : $key;

        return is_multisite()
            ? update_site_option($key, $value)
            : update_option($key, $value);
    }

    public static function add(string $key, $value, $specific_country = false)
    {
        $key = $specific_country ? self::prefix() . $key : $key;

        return is_multisite()
            ? add_site_option($key, $value)
            : add_option($key, $value);
    }
}

class Vtupress_System_State
{
    public static function isConfigured(): bool
    {
        $enabled = Vtupress_option::get('vtupress_enabled_countries', []);
        return is_array($enabled) && count($enabled) > 0;
    }
}


class Vtupress_Country_Context
{
    public static ?string $license_for = null;
    public static ?string $license_country = null;

    protected static function boot(): void
    {
        if (self::$license_for === null) {
            self::$license_for = Vtupress_Option::get(
                'vtupress_custom_licensefor'
            );
        }

        if (self::$license_country === null) {
            self::$license_country = Vtupress_Option::get(
                'vtupress_custom_licensecountry'
            );
        }
    }

    public static function get(int $api_user_id = null): string
    {

        if (Vtupress_option::get('vtupress_custom_active', 'no') != "yes")
            return 'nigeria';

        self::boot(); // 👈 ensure values are loaded
        static $resolved = null;

        // Only cache when NOT resolving for a specific API user
        if ($resolved !== null && $api_user_id === null) {
            return $resolved;
        }

        $can_check_user = function_exists('is_user_logged_in');
        $users = new Users();

        $licensefor = self::$license_for;
        if ($licensefor != 'all') {
            return Vtupress_Country_Manager::validate(
                $licensefor,
                false
            );
        }

        /*
         * 1️⃣ API user context (explicit, highest priority)
         */
        if ($api_user_id) {
            $vt_user = $users->getUser($api_user_id, false, true);

            if ($vt_user && !empty($vt_user->set_country)) {
                return Vtupress_Country_Manager::validate(
                    $vt_user->set_country,
                    false
                );
            }
        }

        /*
         * 2️⃣ Logged-in user context
         */
        if ($can_check_user && is_user_logged_in() && !is_admin()) {
            $vt_user = $users->getUser(get_current_user_id(), false, true);

            if ($vt_user && !empty($vt_user->set_country)) {
                return $resolved = Vtupress_Country_Manager::validate(
                    $vt_user->set_country,
                    false
                );
            }
        }

        /*
         * 3️⃣ Business / system context (admin, public, cron, fallback)
         */
        $business_country = Vtupress_Option::get(
            'vtupress_business_country',
            self::$license_country
        );


        $resolved = Vtupress_Country_Manager::validate(
            $business_country,
            false
        );

        if (is_wp_error($resolved)) {
            wp_die($resolved->get_error_message());
        }

        return $resolved;
    }

}





class Vtupress_Country_Manager
{
    public static function all(): array
    {
        static $all = null;
        if ($all !== null)
            return $all;

        $raw = file_get_contents(VTUPRESS_PATH . '/config/country.json');
        $all = json_decode($raw, true) ?: [];

        return apply_filters('vtupress_country_array', $all);
    }

    public static function enabled(): array
    {

        return Vtupress_option::get('vtupress_enabled_countries', [Vtupress_Country_Context::$license_country]);
    }

    public static function validate($country, bool $strict = true)
    {

        if (!is_string($country)) {
            return false;
        }
        $all = self::all();
        $enabled = self::enabled();

        // Country must exist (always)
        if (!isset($all[$country])) {
            return new \WP_Error('invalid_country', "Invalid country: {$country}");
        }

        // During bootstrap, do NOT enforce enabled list
        if (!Vtupress_System_State::isConfigured()) {
            return $country;
        }

        // Runtime strict validation
        if ($strict && !in_array($country, $enabled, true)) {
            return new \WP_Error('invalid_country', "Country not enabled: {$country}");
        }

        return $country;
    }

    /**
     * Automatically ensure "reseller" and "customer" levels exist for a country.
     */
    public static function initialize_defaults(string $country)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'vtupress_levels';

        $defaults = ['reseller', 'customer'];

        foreach ($defaults as $name) {
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table_name WHERE name = %s AND set_country = %s",
                $name,
                $country
            ));

            if (!$exists) {
                $wpdb->insert($table_name, [
                    'name' => $name,
                    'status' => 'active',
                    'set_country' => $country
                ]);
            }
        }
    }

    public static function currency(string $country): string
    {
        $all = self::all();
        // Return symbol if exists, else fallback to '₦' or config default
        return $all[$country]['symbol'] ?? '₦';
    }
}


/*

######## -1
$enabled = Vtupress_option::get('vtupress_enabled_countries', []);

if (!in_array($current_country, $enabled)) {
    return new WP_Error('country_disabled', 'Country not enabled');
}


####### -- 2
$country = Vtupress_Country_Context::get();

if (
    !in_array('all', $sv->countries) &&
    !in_array($country, $sv->countries)
) {
    return null;
}

$sv->current_country = $country;


if (!in_array($country, $enabled_countries)) {
    return new WP_Error('invalid_country');
}



$wpdb->get_results(
    $wpdb->prepare(
        "SELECT * FROM $table WHERE service_name=%s AND set_country=%s",
        $service,
        $current_country
    )
);

try {
    $country = Vtupress_Country_Context::get();
} catch (Exception $e) {
    return wp_send_json_error([
        'message' => $e->getMessage()
    ]);
}


add_action('admin_notices', function () {
    try {
        Vtupress_Country_Context::get();
    } catch (Exception $e) {
        echo '<div class="notice notice-error"><p>' . esc_html($e->getMessage()) . '</p></div>';
    }
});


*/




function generateApiKey($length = 32)
{
    // Generate random bytes
    $bytes = random_bytes($length);
    // Convert to hexadecimal string
    return bin2hex($bytes);
}

function vtupress_is_plugin_active(string $plugin): bool
{
    if (is_plugin_active($plugin)) {
        return true;
    }

    // Multisite: network-activated
    if (is_multisite() && is_plugin_active_for_network($plugin)) {
        return true;
    }

    return false;
}


function vtupress_plugin_activation_status(string $plugin): string
{
    if (is_plugin_active_for_network($plugin)) {
        return 'network';
    }

    if (is_plugin_active($plugin)) {
        return 'site';
    }

    return 'inactive';
}

function vtupress_require_plugin(
    string $plugin,
    string $plugin_name
): bool {
    if (vtupress_is_plugin_active($plugin)) {
        return true;
    }

    add_action('admin_notices', function () use ($plugin_name) {
        echo '<div class="notice notice-error"><p>';
        echo '<strong>Vtupress Error:</strong> ';
        echo esc_html($plugin_name) . ' is required for this feature.';
        echo '</p></div>';
    });

    return false;
}

/*

if (!function_exists('is_plugin_active')) {
    require_once ABSPATH . 'wp-admin/includes/plugin.php';
}


if (vtupress_is_plugin_active('vtupress/vtupress.php')) {
    // Safe to run
}

$status = vtupress_plugin_activation_status('vtupress/vtupress.php');

if ($status === 'network') {
    // Network-wide logic
} elseif ($status === 'site') {
    // Site-only logic
}




if (!vtupress_is_plugin_active('required/required.php')) {
    add_action('admin_init', 'vtupress_deactivate_self');
    return;
}


*/



include_once VTUPRESS_PATH . '/includes/functions.php';
include_once VTUPRESS_PATH . '/functions.php';
include_once ABSPATH . 'wp-admin/includes/plugin.php';
include_once ABSPATH . 'wp-admin/includes/upgrade.php';
include_once __DIR__ . "/gateway_class.php";
include_once __DIR__ . "/services_class.php";
include_once __DIR__ . "/menu_walker.php";
include_once __DIR__ . '/user_class.php';
include_once __DIR__ . '/admin_class.php';

function get_vp_current_user_id(){
    $user_id = get_current_user_id();
    if($user_id){
        $users = new Users();
        $user = $users->getUser($user_id,false,true);
        return $user->id ?? 0;
    }

    return 0;
}

class Vtupress
{

    public $admin_ui;
    public $currentuserdata;
    public $admin_instance;

    public $country;

    public static $static_country;

    //constructor
    public function __construct()
    {
        // if (Vtupress_option::get('vtupress_custom_active', 'no') == "yes"):
        $this->admin_ui = Vtupress_Option::get("vtupress_admin_ui", "admin");
        $this->init();
        // endif;
    }
    public function init()
    {

        self::check_conflicts();

        // Load all gateways to register their AJAX hooks
        if (class_exists('Vtupress_Gateway_Container')) {
            Vtupress_Gateway_Container::all();
        }

        add_action('init', [$this, 'boot_country'], 1);

        add_action('parse_request', 'vtupress_custom_api_router');

        add_action('wp_ajax_vtupress_verify_service', ['Vtupress_services', 'ajax_verify']);
        add_action('wp_ajax_nopriv_vtupress_verify_service', ['Vtupress_services', 'ajax_verify']);

        add_action('wp_ajax_vtupress_register_user', [$this, 'ajax_register_user']);
        add_action('wp_ajax_nopriv_vtupress_register_user', [$this, 'ajax_register_user']);
        add_action('wp_ajax_vtupress_login_user', [$this, 'ajax_login_user']);
        add_action('wp_ajax_nopriv_vtupress_login_user', [$this, 'ajax_login_user']);
        add_action('wp_ajax_vtupress_activate_user', [$this, 'ajax_activate_user']);
        add_action('wp_ajax_nopriv_vtupress_activate_user', [$this, 'ajax_activate_user']);
        add_filter('redirect_canonical', function ($redirect, $request) {
            if (get_query_var('vtupress-vend')) {
                return false;
            }
            return $redirect;
        }, 10, 2);

        // Register Automated Savings Cron
        if (!wp_next_scheduled('vtupress_process_automated_savings')) {
            wp_schedule_event(time(), 'hourly', 'vtupress_process_automated_savings');
        }
        add_action('vtupress_process_automated_savings', ['Vtupress_Savings', 'process_automated_savings_cron']);

        // Profile & API Hooks
        add_action('wp_ajax_vtupress_update_profile', [$this, 'handle_profile_update']);
        add_action('wp_ajax_vtupress_generate_apikey', [$this, 'handle_apikey_generation']);
        add_action('wp_ajax_vtupress_generate_virtual_account', [$this, 'ajax_generate_virtual_account']);

        if (!function_exists('vtupress_custom_api_router')) {
            function vtupress_custom_api_router($wp)
            {

                // $request = trim($_SERVER['REQUEST_URI'], '/');
                $request = trim($wp->request, '/');


                // Check if request starts with api/v1/
                if (preg_match('#^api/([^/]+)/?$#', $request, $matches)) {

                    $_POST = $_POST ?? [];
                    $_POST['action'] = sanitize_key($matches[1]);

                    $file = VTUPRESS_PATH . '/api/index.php';

                    if (file_exists($file)) {
                        include $file;
                        exit;
                    }

                    wp_send_json(['error' => 'Invalid API endpoint'], 404);
                } elseif (preg_match('#^payment-webhook/?$#', $request, $matches)) {
                    // Delegate to Gateway Container to find the matching gateway
                    Vtupress_Gateway_Container::handle_webhooks();
                    exit;
                } elseif (preg_match('#^internal-pin-api/?$#', $request, $matches)) {
                    include_once __DIR__ . "/internal_api.php";
                    vtupress_internal_pin_api();
                    exit;
                }
            }
        }

        add_action("init", [$this, "currentuserdata"]);

        $this->shortcodes();
        $all_uis = vtupress_admin_ui();
        if (array_key_exists($this->admin_ui, $all_uis)) {
            $this->admin_instance = new $all_uis[$this->admin_ui]();
            add_action("admin_menu", [$this, "addMenu"]);
        }

        add_action('init', function () {
            $page = self::vtupress_find_page_by_shortcode('vpaccount');

            if ($page) {
                $slug = $page->post_name;
                self::vtupress_add_init_with_slug($slug);
            }

        });


        // Hook when the static front page changes
        add_action('update_option_page_on_front', function ($old_value, $value) {
            if ($old_value != $value) {
                $page = self::vtupress_find_page_by_shortcode('vpaccount');

                if ($page) {
                    $slug = $page->post_name;
                    self::vtupress_add_init_with_slug($slug);
                }
                flush_rewrite_rules();
            }
        }, 10, 2);

        // Hook when the "Your homepage displays" setting changes
        add_action('update_option_show_on_front', function ($old_value, $value) {
            if ($old_value != $value) {
                $page = self::vtupress_find_page_by_shortcode('vpaccount');

                if ($page) {
                    $slug = $page->post_name;
                    self::vtupress_add_init_with_slug($slug);
                }
                flush_rewrite_rules();
            }
        }, 10, 2);


        add_filter('page_template', function ($page_template) {
            $page = self::vtupress_find_page_by_shortcode('vpaccount');

            if ($page) {
                $slug = $page->post_name;
                if (is_page($slug)) {
                    $page_template = VTUPRESS_PATH . '/template.php';
                }
            }
            return $page_template;
        });

        add_filter('query_vars', function ($vars) {
            $vars[] = 'vtupress-vend';
            $vars[] = 'vtupress-page2';
            $vars[] = 'vtupress-page3';
            return $vars;
        });


        if (vtupress_is_plugin_active("vtupress-templates/vtupress-templates.php")) {
            include_once ABSPATH . "wp-content/plugins/vtupress-templates/template_class.php";
            add_action('wp_enqueue_scripts', function () {
                $current_template = Vtupress_Option::get("vtupress_template", "msorg");
                $template = Vtupress_Template_Container::get($current_template);
                if (is_object($template)) {
                    $template->enqueue_this_template_scripts();
                }
            });
        }

        add_action('init', function () {

            register_nav_menus([
                'vtupress-main' => __('Before Vtupress Services', 'vtupress'),
                'vtupress-footer' => __('After Vtupress services', 'vtupress'),
            ]);

        });

        // Add the custom field to menu items in the admin menu editor
        add_action('wp_nav_menu_item_custom_fields', function ($item_id, $item, $depth, $args) {

            // Get menu being edited from $_GET['menu'] in admin
            $menu_id = isset($_GET['menu']) ? (int) $_GET['menu'] : 0;
            if (!$menu_id) {
                return; // safety check
            }

            // Get menu IDs for your registered locations
            $locations = get_nav_menu_locations();
            $vtupress_menu_ids = [
                $locations['vtupress-main'] ?? 0,
                $locations['vtupress-footer'] ?? 0,
            ];

            // Exit if this menu is not one of the Vtupress menus
            if (!in_array($menu_id, $vtupress_menu_ids)) {
                return;
            }
            $value = get_post_meta($item_id, '_vtupress_load', true);
            ?>
            <p class="field-vtupress description description-wide">
                <label for="edit-menu-item-vtupress-<?php echo $item_id; ?>">
                    <input type="checkbox" id="edit-menu-item-vtupress-<?php echo $item_id; ?>" name="menu-item-vtupress[
<?php echo $item_id; ?>]" value="1" <?php checked($value, '1'); ?> />
                    <?php _e('Load in Vtupress', 'vtupress'); ?>
                </label>
            </p>
            <?php
        }, 10, 4);

        add_action('wp_update_nav_menu_item', function ($menu_id, $menu_item_id) {
            if (isset($_POST['menu-item-vtupress'][$menu_item_id])) {
                update_post_meta($menu_item_id, '_vtupress_load', '1');
            } else {
                delete_post_meta($menu_item_id, '_vtupress_load');
            }
        }, 10, 2);



    }

    public function boot_country()
    {
        $this->country = Vtupress_Country_Context::get();
        self::$static_country = $this->country;
    }

    public function get_option($key, $default = false)
    {
        return Vtupress_Option::get("vtupress_{$key}", $default);
    }

    public function update($key, $value)
    {
        return Vtupress_option::update("vtupress_{$key}", $value);
    }

    public function vtupress_resolve_url(string $url): string
    {
        // Full URL? Return as-is
        if (preg_match('#^https?://#i', $url)) {
            return $url;
        }

        // Leading slash: resolve from site root
        if (str_starts_with($url, '/')) {
            return home_url($url);
        }

        $page = self::vtupress_find_page_by_shortcode('vpaccount');

        if ($page) {
            $slug = $page->post_name;
            return home_url($slug . '/' . ltrim($url, '/'));
        }

        return '';

    }

    public static function vtupress_add_init_with_slug($slug)
    {
        if (!$slug)
            return;

        // Check if this page is set as the homepage
        $frontpage_id = get_option('page_on_front');
        $page = get_page_by_path($slug);
        $is_homepage = ($frontpage_id && $page && $frontpage_id == $page->ID);

        // Add standard rules with slug prefix (/dashboard/page)
        add_rewrite_rule("^$slug/([^/]+)/([^/]+)/([^/]+)/?$", "index.php?pagename=$slug&vtupress-vend=\$matches[1]&vtupress-page2=\$matches[2]&vtupress-page3=\$matches[3]", 'top');
        add_rewrite_rule("^$slug/([^/]+)/([^/]+)/?$", "index.php?pagename=$slug&vtupress-vend=\$matches[1]&vtupress-page2=\$matches[2]", 'top');
        add_rewrite_rule("^$slug/([^/]+)/?$", "index.php?pagename=$slug&vtupress-vend=\$matches[1]", 'top');
        add_rewrite_rule("^$slug/?$", "index.php?pagename=$slug&vtupress-vend=dashboard", 'top');

        // If this page is the homepage, add root-level rules (//page)
        if ($is_homepage) {
            add_rewrite_rule("^([^/]+)/([^/]+)/([^/]+)/?$", "index.php?page_id=$frontpage_id&vtupress-vend=\$matches[1]&vtupress-page2=\$matches[2]&vtupress-page3=\$matches[3]", 'top');
            add_rewrite_rule("^([^/]+)/([^/]+)/?$", "index.php?page_id=$frontpage_id&vtupress-vend=\$matches[1]&vtupress-page2=\$matches[2]", 'top');
            add_rewrite_rule("^([^/]+)/?$", "index.php?page_id=$frontpage_id&vtupress-vend=\$matches[1]", 'top');
        }

    }

    public static function vtupress_find_page_by_shortcode($shortcode = 'vpaccount')
    {
        $pages = get_posts([
            'post_type' => 'page',
            'posts_per_page' => -1,
            'post_status' => 'publish',
        ]);

        foreach ($pages as $page) {
            if (has_shortcode($page->post_content, $shortcode)) {
                return $page; // Return the first match
            }
        }

        return null; // Not found
    }

    public function mail($to, $header = 'Header', $body = 'Body', $footer = '')
    {
        $site_name = get_bloginfo('name');
        $date = date('Y');

        $message = "
        <div style='background-color: #f8f9fa; padding: 50px 20px; font-family: \"Segoe UI\", Roboto, Helvetica, Arial, sans-serif;'>
            <div style='max-width: 600px; margin: 0 auto; background: #ffffff; border-radius: 12px; box-shadow: 0 8px 30px rgba(0,0,0,0.08); overflow: hidden;'>
                
                <!-- Header -->
                <div style='padding: 30px 40px; background: #ffffff; text-align: center; border-bottom: 1px solid #f0f0f0;'>
                    <h2 style='margin: 0; color: #333; font-size: 24px; font-weight: 700;'>$header</h2>
                </div>

                <!-- Body -->
                <div style='padding: 40px; color: #555; font-size: 16px; line-height: 1.6;'>
                    $body
                </div>

                <!-- Footer -->
                <div style='background-color: #fcfcfc; padding: 25px 40px; text-align: center; color: #888; font-size: 14px; border-top: 1px solid #f0f0f0;'>
                    <div style='margin-bottom: 15px;'>$footer</div>
                    <div style='font-size: 12px; color: #aaa;'>&copy; $date $site_name</div>
                </div>

            </div>
        </div>";

        $headers = array('Content-Type: text/html; charset=UTF-8');
        wp_mail($to, $header, $message, $headers);
    }


    public static function activate()
    {

        include_once __DIR__ . '/database.php';
        create_userDatabase();
        // self::vtupress_sortDbs();
        self::vtupress_roles();
        $page = self::vtupress_find_page_by_shortcode('vpaccount');

        if ($page) {
            $slug = $page->post_name;
            self::vtupress_add_init_with_slug($slug);
        }

        if ($page) {
            flush_rewrite_rules();
        }
    }

    public static function check_conflicts()
    {
        $plugins_to_check = [
            'vtupress/vtupress.php' => 'VTUPress Core',
            'vtupress_woocommerce/vtupress_woocommerce.php' => 'VTUPress WooCommerce',
            'vtupress-bot/plugin.php' => 'VTUPress Bot',
            'vpussd/vpussd.php' => 'VP USSD',
            'vpsms/vpsms.php' => 'VP SMS',
            'vprest/vprest.php' => 'VP API',
            'vpmlm/vpmlm.php' => 'VP MLM',
            'vpepin/vpepin.php' => 'VP E-Pin',
            'vpdatas/vpdatas.php' => 'VP Data',
            'vpcards/vpcards.php' => 'VP Cards',
            'opay/opay.php' => 'Opay Integration',
            'msorg_template/msorg_template.php' => 'Msorg Template',
            'bi-firewall/bi-guardian.php' => 'BI Firewall',
            'bcmv/bcmv.php' => 'BCMV'
        ];

        include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        $results = [];

        foreach ($plugins_to_check as $path => $name) {
            $is_active = vtupress_is_plugin_active($path);
            if ($is_active) {
                global $thereason;
                $thereason = $name;
                add_action('admin_notices', function () {
                    global $thereason;
                    echo '<div class="notice notice-error"><p>';
                    echo "<strong>Vtupress Disabled:</strong> Please deactivate $thereason and other vtupress addons";
                    echo '</p></div>';
                });

                deactivate_plugins('vtupress/plugin.php');

            }

        }


        // deactivate_plugins(plugin_basename(__FILE__));


    }

    public function addMenu()
    {
        add_menu_page("Vtu Press", "Vtupress", "vtupress_access_vtupress", "vtupanel", [$this, "render_admin_page"], "dashicons-calculator");
    }

    public function render_admin_page()
    {
        $this->admin_instance->render_admin_page();
    }

    public function ajax_generate_virtual_account()
    {
        $current_user_id = get_current_user_id();
        $target_user_id = intval($_POST['user_id'] ?? 0);

        // Allow if admin OR if user is generating for themselves
        if (!current_user_can('manage_options') && $current_user_id !== $target_user_id) {
            wp_send_json_error(['message' => 'Permission denied']);
        }
        
        // If frontend request doesn't send user_id, use current user
        if ($target_user_id === 0 && !current_user_can('manage_options')) {
            $target_user_id = $current_user_id;
        }

        // Optional: Add nonce check if you pass one in the JS
        // check_ajax_referer('vtupress_admin_nonce', 'nonce');

        $gateway_id = sanitize_text_field($_POST['gateway'] ?? '');
        $user_id = $target_user_id;

        if (!$gateway_id || !$user_id) {
            wp_send_json_error(['message' => 'Missing parameters']);
        }

        $gateway = Vtupress_Gateway_Container::get($gateway_id);

        if (!$gateway) {
            wp_send_json_error(['message' => 'Invalid gateway']);
        }

        if (method_exists($gateway, 'create_virtual_account')) {
            $result = $gateway->create_virtual_account($user_id);
            if ($result['status'] === 'success') {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result);
            }
        } else {
            wp_send_json_error(['message' => 'Method not implemented for this gateway']);
        }
    }

    public function ajax_login_user()
    {
        $this->boot_country();

        if (!check_ajax_referer('vtupress_user_nonce', '_nonce', false)) {
            wp_send_json(['status' => '101', 'message' => 'Security check failed']);
        }

        $username = isset($_POST['username']) ? sanitize_text_field($_POST['username']) : '';
        $password = isset($_POST['password']) ? $_POST['password'] : '';

        if (empty($username) || empty($password)) {
            wp_send_json(['status' => '101', 'message' => 'All fields are required']);
        }

        if (preg_match("/@/", $username)) {
            if (!email_exists($username)) {
                wp_send_json(['status' => '101', 'message' => 'Wrong Email']);
            } else {
                $wp_user = get_user_by('email', $username);
            }
        } else {
            $wp_user = get_user_by('login', $username);
        }

        if (empty($wp_user)) {
            wp_send_json(['status' => '101', 'message' => 'User not found']);
        }


        $username = $wp_user->user_login;
        //check if user exist
        if (!username_exists($username)) {
            wp_send_json(['status' => '101', 'message' => 'Wrong Username']);
        }

        //check if that user email is verified
        $users = new Users();
        $user = $users->getUser($wp_user->ID, true, true);
        if (!$user) {
            wp_send_json(['status' => '101', 'message' => 'Account not in sync. contact admin']);
        }

        $user_id = $user["id"];

        //user exist -- load settings
        global $wpdb;
        $settings_table = $wpdb->prefix . 'vtupress_general_settings';
        $settings = $wpdb->get_row($wpdb->prepare("SELECT * FROM $settings_table WHERE set_country = %s", $user["set_country"]), ARRAY_A);

        if (empty($settings)) {
            wp_send_json(['status' => '101', 'message' => 'Not settings for your country']);
        }

        //if enabled verify email
        if ($settings["enable_email_verification"] == "yes" && $user["email_verified"] != "yes") {
            // SEND EMAIL
            $code = $user["reset_code"] ?? rand(11111, 99999);
            $users->update($user_id, ["reset_code" => $code]);
            $header = 'Account Verification';
            $body = "Your activation code is <strong>$code</strong>";
            $this->mail($wp_user->user_email, $header, $body);

            wp_send_json(['status' => '101', 'message' => 'Please verify account. An email is already sent to you. You can check your spam folder']);
        }

        $login_data = array();
        $login_data['user_login'] = $username;
        $login_data['user_password'] = $password;
        $login_data['remember'] = true;

        $user_verify = wp_signon($login_data, false);

        if (is_wp_error($user_verify)) {
            wp_send_json(['status' => '101', 'message' => 'Wrong Credentials']);
        } else {
            wp_send_json([
                'status' => '100',
                'message' => 'welcome',
                'ID' => $user_verify->ID,
                'name' => $user_verify->user_login
            ]);
        }
    }

    public function ajax_register_user()
    {
        if (!check_ajax_referer('vtupress_user_nonce', '_nonce', false)) {
            wp_send_json(['status' => '101', 'message' => 'Security check failed']);
        }

        $user = isset($_POST["username"]) ? trim(sanitize_user($_POST["username"])) : '';
        $email = isset($_POST["email"]) ? trim(sanitize_email($_POST["email"])) : '';
        $pass = isset($_POST["pswd"]) ? $_POST["pswd"] : '';
        $phone = isset($_POST["phone"]) ? trim($_POST["phone"]) : '';
        $ref = isset($_POST["referral"]) ? trim($_POST["referral"]) : '';
        $fun = isset($_POST["fun"]) ? trim(sanitize_text_field($_POST["fun"])) : '';
        $lun = isset($_POST["lun"]) ? trim(sanitize_text_field($_POST["lun"])) : '';
        $pin = isset($_POST["pin"]) ? trim($_POST["pin"]) : '';
        $country = isset($_POST["country"]) ? trim(sanitize_text_field($_POST["country"])) : false;

        if (!$country) {
            wp_send_json(['status' => '101', 'message' => 'No country specified']);
        }

        //check if country is enabled
        if (!in_array($country, Vtupress_Country_Manager::enabled())) {
            wp_send_json(['status' => '101', 'message' => 'Country not enabled for registration']);
        }

        //check if registration is permitted for this country
        global $wpdb;
        $settings_table = $wpdb->prefix . 'vtupress_general_settings';
        $settings = $wpdb->get_row($wpdb->prepare("SELECT * FROM $settings_table WHERE set_country = %s", $country), ARRAY_A);

        if (empty($settings)) {
            wp_send_json(['status' => '101', 'message' => 'Not settings for your country']);
        }

        //  wp_send_json(['status' => '101', 'message' => $settings["enable_registration"]]);

        if (!$settings["enable_registration"] ?? false) {
            wp_send_json(['status' => '101', 'message' => 'Registration not allowed for your country']);
        }

        if (!is_email($email)) {
            wp_send_json(['status' => '101', 'message' => 'Incorrect Email']);
        }

        // if (!preg_match("/^[0-9]{10,15}$/", $phone)) {
        //     wp_send_json(['status' => '101', 'message' => 'Enter Valid Phone Numbers']);
        // } // Adjusted regex for broader support or length

        $verify_ref = preg_match("/^[0-9]+/", $ref);
        if (!$verify_ref) {
            wp_send_json(['status' => '101', 'message' => 'A Valid Referral ID']);
        }

        $ref_id = $verify_ref;

        if (strlen($fun) < 3 || strlen($lun) < 3) {
            wp_send_json(['status' => '101', 'message' => 'First And Last Name Must Be Of At Least (3) Letters Only Without Spaces']);
        }

        if (preg_match("/^0\d+$/", $pin)) {
            wp_send_json(['status' => '101', 'message' => 'Pin Must Not Start With Zero']);
        }
        if (preg_match("/[^0-9]/", $pin) || strlen($pin) != 4) {
            wp_send_json(['status' => '101', 'message' => 'Pin Must Be 4 Digits']);
        }

        if (username_exists($user)) {
            wp_send_json(['status' => '101', 'message' => 'Sorry, that username already exists!']);
        } elseif (email_exists($email)) {
            wp_send_json(['status' => '101', 'message' => 'Sorry, that email already exists!']);
        }


        global $wpdb;
        $vtupress_users_table = $wpdb->prefix . "vtupress_users";

        $exists = $wpdb->get_row($wpdb->prepare("SELECT * FROM $vtupress_users_table WHERE phone = %s", $phone));
        if ($exists) {
            wp_send_json(['status' => '101', 'message' => 'Phone number already exists!']);
        }


        $userdata = array(
            'user_login' => $user,
            'user_email' => $email,
            'user_pass' => $pass
        );

        $userid = wp_insert_user($userdata);

        if (is_wp_error($userid)) {
            wp_send_json(['status' => '101', 'message' => $userid->get_error_message()]);
        }

        $referrer_exists = $wpdb->get_row($wpdb->prepare("SELECT * FROM $vtupress_users_table WHERE id = %d", intval($ref)));
        if ($referrer_exists && $ref != $userid) {
            $ref_id = $ref;
        }

        $apikey = uniqid("vtu-", false);

        global $wpdb;
        $vtupress_users_table = $wpdb->prefix . "vtupress_users";
        $new_user_data = [
            'userId' => $userid,
            'first_name' => $fun,
            'last_name' => $lun,
            'email_verified' => 'false',
            'plan' => 'customer',
            'apikey' => $apikey,
            'phone' => $phone,
            'pin' => $pin,
            'balance' => 0,
            'referral_rewards' => 0,
            'referrer_id' => $ref_id,
            'referrers' => '[]',
            'indirect_referrers' => '[]',
            'other_referrers' => '[]',
            'about' => 'New User',
            'beneficiaries' => '',
            'fix_version' => '1.0',
            'all_plans' => 'customer',
            'run_code' => rand(1000, 9999),
            'set_country' => $country
        ];

        $wpdb->insert($vtupress_users_table, $new_user_data);
        $new_vtu_id = $wpdb->insert_id;

        if (!$new_vtu_id) {
            wp_send_json(['status' => '101', 'message' => 'Error creating account']);
        }

        // MLM Logic
        if ($ref_id != "1") {
            // LEVEL 1
            $upline_1 = $wpdb->get_row($wpdb->prepare("SELECT id, referrers, referrer_id FROM $vtupress_users_table WHERE id = %d", $ref_id));
            if ($upline_1) {
                $l1_refs = json_decode($upline_1->referrers, true) ?: [];
                $l1_refs[] = $new_vtu_id;
                $wpdb->update($vtupress_users_table, ['referrers' => json_encode($l1_refs)], ['id' => $upline_1->id]);

                // LEVEL 2
                $upline_2_id = $upline_1->referrer_id;
                if ($upline_2_id && $upline_2_id != "1") {
                    $upline_2 = $wpdb->get_row($wpdb->prepare("SELECT id, indirect_referrers, referrer_id FROM $vtupress_users_table WHERE id = %d", $upline_2_id));
                    if ($upline_2) {
                        $l2_refs = json_decode($upline_2->indirect_referrers, true) ?: [];
                        $l2_refs[] = $new_vtu_id;
                        $wpdb->update($vtupress_users_table, ['indirect_referrers' => json_encode($l2_refs)], ['id' => $upline_2->id]);

                        // LEVEL 3
                        $upline_3_id = $upline_2->referrer_id;
                        if ($upline_3_id && $upline_3_id != "1") {
                            $upline_3 = $wpdb->get_row($wpdb->prepare("SELECT id, other_referrers FROM $vtupress_users_table WHERE id = %d", $upline_3_id));
                            if ($upline_3) {
                                $l3_refs = json_decode($upline_3->other_referrers, true) ?: [];
                                $l3_refs[] = $new_vtu_id;
                                $wpdb->update($vtupress_users_table, ['other_referrers' => json_encode($l3_refs)], ['id' => $upline_3->id]);
                            }
                        }
                    }
                }
            }
        }

        $users = new Users();

        // Email Verification
        //if enabled verify email
        $success = ['status' => '100'];

        if ($settings["enable_email_verification"] == "yes") {
            // SEND EMAIL
            $code = rand(11111, 99999);
            $users->update($new_vtu_id, ["reset_code" => $code]);
            $header = 'Account Verification';
            $body = "Your activation code is <strong>$code</strong>";
            $this->mail($email, $header, $body);

            $success = array_merge($success, ['message' => 'Account registered, please verify your account with us. you can check your spam folder for activation code']);
        }

        do_action('user_register', $userid, $userdata);
        wp_send_json($success);
    }

    public function ajax_activate_user()
    {
        if (!check_ajax_referer('vtupress_user_nonce', '_nonce', false)) {
            wp_send_json(['status' => '101', 'message' => 'Security check failed']);
        }

        $user = isset($_POST["user"]) ? trim(sanitize_text_field($_POST["user"])) : '';
        $code = isset($_POST["verify"]) ? trim(sanitize_text_field($_POST["verify"])) : '';

        if (empty($user) || empty($code)) {
            wp_send_json(['status' => '101', 'message' => 'Please provide username and activation code']);
        }

        $userinfo = get_user_by('login', $user);
        if (!$userinfo) {
            wp_send_json(['status' => '101', 'message' => 'User not found']);
        }

        $user_id = $userinfo->ID;
        $users = new Users();
        $vtuser = $users->getUser($user_id, true, true);

        if (!$vtuser) {
            wp_send_json(['status' => '101', 'message' => 'VTU Account not found']);
        }

        

        if ($vtuser['email_verified'] == 'yes' || $vtuser['email_verified'] == 'verified') {
            wp_send_json(['status' => '101', 'message' => 'Account Already Verified']);
        }

        // Check code against reset_code (new) or email_verified (legacy)
        $stored_code = $vtuser['reset_code'] ?? '';
        $legacy_code = ($vtuser['email_verified'] != 'false' && $vtuser['email_verified'] != 'verified') ? $vtuser['email_verified'] : '';

        if (($stored_code && $code == $stored_code) || ($legacy_code && $code == $legacy_code)) {
            // Success
            $users->update($user_id, ['email_verified' => 'yes', 'reset_code' => '']);
            global $wpdb;
            $wpdb->update($wpdb->prefix . 'vtupress_users', ['email_verified' => 'yes'], ['id' => $vtuser['id']]); // Double check for old column usage if needed

            wp_send_json(['status' => '100', 'message' => 'Activation Successful', 'name' => $userinfo->user_login]);
        } else {
            wp_send_json(['status' => '101', 'message' => 'Invalid Activation Code']);
        }
    }


    public function currentuserdata()
    {
        $this->currentuserdata = array();

        if (is_user_logged_in()) {
            $currentuserid = get_current_user_id();
            $users = new Users();
            $user = $users->getUser($currentuserid, true, true);
            if (empty($user)) {
                $this->currentuserdata = array();
            } else {
                $this->currentuserdata = $user;
            }
        }
        return $this->currentuserdata;
    }

    public function shortcodes()
    {
        add_shortcode("vtupress", function ($atts = "") {
            $defaults = array_change_key_case($this->currentuserdata, CASE_LOWER);

            $attr = array_change_key_case((array) $atts, CASE_LOWER);
            $defaults["commission"] = intval($defaults["referral_commission"]) +
                intval($defaults["indirect_referral_commission"]) +
                intval($defaults["other_referral_commission"]) +
                intval($defaults["transactions_commission"]) +
                intval($defaults["referee_transaction"]) +
                intval($defaults["indirect_referee_transaction"]) +
                intval($defaults["other_referee_transaction"]);
            if (array_key_exists($attr["get"], $defaults)) {
                return $defaults[$attr["get"]];
            } else {
                return "key_not_found";
            }


        });

        add_shortcode("vpaccount", function () {
            ob_start();

            if (!vtupress_is_plugin_active("vtupress-templates/vtupress-templates.php")) {
                return '<div class="vtupress-error">Template plugin not active</div>';
            }

            include_once __DIR__ . '/user_class.php';
            include_once ABSPATH . "wp-content/plugins/vtupress-templates/template_class.php";
            $current_template = Vtupress_Option::get("vtupress_template", "msorg");

            $template = Vtupress_Template_Container::get($current_template);
            if ($template == null) {
                return '<div class="vtupress-error">Invalid Template</div>';
            }

            $template->render_user_dashboard();

            return ob_get_clean();
        });

    }

    public static function vtupress_roles()
    {

        //VTUPRESS ADMIN
        add_role(
            'vtupress_admin',
            "VTUPRESS ADMIN",
            [
                'read' => true,
                'vtupress_access_addons' => true,
                'vtupress_access_license' => true,
                'vtupress_access_levels' => true,
                'vtupress_access_kyc' => true,
                'vtupress_access_security' => true,
                'vtupress_access_gateway' => true,
                'vtupress_access_importer' => true,
                'vtupress_access_settings' => true,
                'vtupress_access_general' => true,
                'vtupress_access_payment' => true,
                'vtupress_access_history' => true,
                'vtupress_delete_history' => true,
                'vtupress_access_users' => true,
                'vtupress_access_users_action' => true,
                'vtupress_access_withdrawal' => true,
                'vtupress_access_mlm' => true,
                'vtupress_access_vtupress' => true,
                'vtupress_clear_history' => true
            ]
        );


        //VTUPRESS SALES MANAGER

        add_role(
            'vtupress_sales',
            "VTUPRESS SALES MANAGER",
            [
                'read' => true,
                'vtupress_access_addons' => false,
                'vtupress_access_license' => false,
                'vtupress_access_levels' => false,
                'vtupress_access_kyc' => false,
                'vtupress_access_security' => false,
                'vtupress_access_gateway' => false,
                'vtupress_access_importer' => false,
                'vtupress_access_settings' => true,
                'vtupress_access_general' => false,
                'vtupress_access_payment' => false,
                'vtupress_access_history' => true,
                'vtupress_delete_history' => true,
                'vtupress_access_users' => false,
                'vtupress_access_users_action' => false,
                'vtupress_access_withdrawal' => true,
                'vtupress_access_mlm' => false,
                'vtupress_access_vtupress' => true,
                'vtupress_clear_history' => true
            ]
        );

        //VTUPRESS USER MANAGER
        add_role(
            'vtupress_user',
            "VTUPRESS USER MANAGER",
            [
                'read' => true,
                'vtupress_access_addons' => false,
                'vtupress_access_license' => false,
                'vtupress_access_levels' => false,
                'vtupress_access_kyc' => false,
                'vtupress_access_security' => false,
                'vtupress_access_gateway' => false,
                'vtupress_access_importer' => false,
                'vtupress_access_settings' => true,
                'vtupress_access_general' => false,
                'vtupress_access_payment' => false,
                'vtupress_access_history' => false,
                'vtupress_delete_history' => false,
                'vtupress_access_users' => true,
                'vtupress_access_users_action' => true,
                'vtupress_access_withdrawal' => false,
                'vtupress_access_mlm' => false,
                'vtupress_access_vtupress' => true,
                'vtupress_clear_history' => false
            ]
        );

        // Gets the simple_role role object.
        $admin_role = get_role('vtupress_admin');

        // Add a new capability.
        $admin_role->add_cap('vtupress_access_vtupress', true);

        // Gets the simple_role role object.
        $user_role = get_role('vtupress_user');

        // Add a new capability.
        $user_role->add_cap('vtupress_access_vtupress', true);

        // Gets the simple_role role object.
        $sale_role = get_role('vtupress_sales');

        // Add a new capability.
        $sale_role->add_cap('vtupress_access_vtupress', true);


        $user = get_user_by('ID', 1);
        if (isset($user) && !empty($user)) {
            $user->add_role('vtupress_admin');
        }

    }

    public function handle_profile_update()
    {
        check_ajax_referer('vtupress_user_nonce', 'nonce'); // Expect frontend to send this

        if (!is_user_logged_in()) {
            wp_send_json(['status' => 'error', 'message' => 'Not logged in']);
        }

        $user_id = get_current_user_id();
        $users = new Users();
        $current_data = $users->getUser($user_id,false,true);
        $wp_user = get_userdata($user_id);

        global $wpdb;
        $vp_users_table = $wpdb->prefix . 'vtupress_users';

        $response = ['status' => 'success', 'message' => 'Profile Updated Successfully'];

        // 1. Handle Photo Upload
        if (!empty($_FILES['photo_upload']['name'])) {
            require_once(ABSPATH . 'wp-admin/includes/image.php');
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            require_once(ABSPATH . 'wp-admin/includes/media.php');

            $attachment_id = media_handle_upload('photo_upload', 0);

            if (is_wp_error($attachment_id)) {
                wp_send_json(['status' => 'error', 'message' => 'Photo Upload Error: ' . $attachment_id->get_error_message()]);
            } else {
                $photo_url = wp_get_attachment_url($attachment_id);
                // Update vtupress_users photo column
                $wpdb->update($vp_users_table, ['photo' => $photo_url], ['id' => $current_data->id]);
            }
        }

        // 2. Handle Name (Only if empty)
        $first_name = sanitize_text_field($_POST['first_name'] ?? '');
        $last_name = sanitize_text_field($_POST['last_name'] ?? '');

        $name_updates = [];
        if (empty($current_data->first_name) && !empty($first_name)) {
            $name_updates['first_name'] = $first_name;
        }
        if (empty($current_data->last_name) && !empty($last_name)) {
            $name_updates['last_name'] = $last_name;
        }

        if (!empty($name_updates)) {
            $wpdb->update($vp_users_table, $name_updates, ['id' => $current_data->id]);
        }


        // 3. Handle Email (WP User)
        $email = sanitize_email($_POST['email'] ?? '');
        if (!empty($email) && $email !== $wp_user->user_email) {
            if (email_exists($email)) {
                wp_send_json(['status' => 'error', 'message' => 'Email already in use']);
            }
            // Update WP User
            $args = ['ID' => $user_id, 'user_email' => $email];
            $ur = wp_update_user($args);
            if (is_wp_error($ur)) {
                wp_send_json(['status' => 'error', 'message' => 'Email Update Failed: ' . $ur->get_error_message()]);
            }
        }

        // 4. Handle Password (WP User)
        $password = $_POST['password'] ?? '';
        if (!empty($password)) {
            wp_set_password($password, $user_id);
            // Note: Relogin needed usually.
            // Auto login?
            $creds = array(
                'user_login' => $wp_user->user_login,
                'user_password' => $password,
                'remember' => true
            );
            $user = wp_signon($creds, false);
        }

        // 5. Handle Phone (VTUPress User)
        $phone = sanitize_text_field($_POST['phone'] ?? '');
        if (!empty($phone) && $phone !== $current_data->phone) {
            // Check duplicate
            $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $vp_users_table WHERE phone = %s AND id != %d", $phone, $current_data->id));
            if ($exists) {
                wp_send_json(['status' => 'error', 'message' => 'Phone number already in use']);
            }
            $wpdb->update($vp_users_table, ['phone' => $phone], ['id' => $current_data->id]);
        }

        // 6. Handle PIN
        $pin = sanitize_text_field($_POST['pin'] ?? '');
        if (!empty($pin)) {
            $wpdb->update($vp_users_table, ['pin' => $pin], ['id' => $current_data->id]);
        }

        // 7. Handle About
        $about = sanitize_textarea_field($_POST['about'] ?? '');
        if (isset($_POST['about'])) {
            $wpdb->update($vp_users_table, ['about' => $about], ['id' => $current_data->id]);
        }

        wp_send_json($response);
    }

    public function handle_apikey_generation()
    {
        check_ajax_referer('vtupress_user_nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json(['status' => 'error', 'message' => 'Not logged in']);
        }

        $user_id = get_current_user_id();
        $users = new Users();
        $vt_user = $users->getUser($user_id,false,true);

        // Generate secure 32 char hex key (16 bytes)
        $new_key = bin2hex(random_bytes(16));

        global $wpdb;
        $table = $wpdb->prefix . 'vtupress_users';
        $wpdb->update($table, ['apikey' => $new_key], ['id' => $vt_user->id]);

        wp_send_json(['status' => 'success', 'apikey' => $new_key]);
    }




}