<?php

function vtupress_get_voucher_services()
{
    global $wpdb;

    // Ensure core class is loaded if not already
    if (!class_exists('Vtupress_services')) {
        // Adjust path if needed, usually included by plugin main file
        // If not, we might be in trouble, but let's assume it is or try to include it.
        // require_once(VTUPRESS_PATH . '/includes/services_class.php'); // Use constant if user defined it
    }

    // Default classes we stick to
    $default_classes = ['RechargeCard', 'DataCard', 'ExamPin'];
    $services = apply_filters('vtupress_service_registry', $default_classes);
    $services = array_unique($services);

    $vouchers = [];

    foreach ($services as $class) {
        if (!class_exists($class))
            continue;

        // Instantiate to check properties
        // Pass required args: db, service_type=false, user_id=null
        try {
            $instance = new $class($wpdb, false, null);

            // Logic to fetch overrides from settings
            $settings = $instance->get_settings(false);
            $types = !empty($settings['service_types']) ? $settings['service_types'] : $instance->service_types;

            // Ensure types is array
            if (!is_array($types)) {
                $types = [$types];
            }

        } catch (Throwable $e) {
            continue; // Skip if instantiation fails
        }

        if (property_exists($instance, 'is_voucher') && $instance->is_voucher) {
            $vouchers[$instance->id] = [
                'name' => $instance->name,
                'types' => $types
            ];
        }
    }
    return $vouchers;
}

add_action('wp_ajax_vtupress_resolve_bank_account', 'vtupress_resolve_bank_account');
function vtupress_resolve_bank_account()
{
    // Security check (ideally nonce, but staying consistent with minimal checks for now)
    $bank_code = sanitize_text_field($_POST['bank_code'] ?? '');
    $account_number = sanitize_text_field($_POST['account_number'] ?? '');
    $gateway_id = sanitize_text_field($_POST['gateway'] ?? 'manual');

    if (empty($bank_code) || empty($account_number)) {
        wp_send_json_error(['message' => 'Missing bank or account number']);
    }

    $gateways = vtupress_gateway_registry();
    if (!isset($gateways[$gateway_id]) || !class_exists($gateways[$gateway_id])) {
        wp_send_json_error(['message' => 'Invalid gateway selected']);
    }

    global $wpdb;
    $class = $gateways[$gateway_id];
    $gateway = new $class($wpdb);

    if (!method_exists($gateway, 'resolve_account')) {
        wp_send_json_error(['message' => 'Gateway does not support account resolution']);
    }

    $result = $gateway->resolve_account($bank_code, $account_number);

    if ($result['status'] === 'success') {
        wp_send_json_success($result);
    } else {
        wp_send_json_error($result);
    }
}

add_action('wp_ajax_vtupress_transfer_funds', 'vtupress_transfer_funds');
function vtupress_transfer_funds()
{
    if (!is_user_logged_in()) {
        wp_send_json_error(['message' => 'Please login to continue']);
    }

    $amount = floatval($_POST['amount'] ?? 0);
    $bank_code = sanitize_text_field($_POST['bank_code'] ?? '');
    $account_number = sanitize_text_field($_POST['account_number'] ?? '');
    $account_name = sanitize_text_field($_POST['account_name'] ?? '');
    $gateway_id = sanitize_text_field($_POST['gateway'] ?? 'manual');
    $wp_user_id = get_current_user_id();

    $users = new Users();
    $vp_user = $users->getUser($wp_user_id, false, true);

    if (!$vp_user) {
        wp_send_json_error(['message' => 'User account not found']);
    }
    $user_id = $vp_user->id;

    // 1. Transaction Lock
    global $wpdb;
    $vend_lock_table = $wpdb->prefix . 'vtupress_vend_lock';
    $inserted = $wpdb->query($wpdb->prepare(
        "INSERT IGNORE INTO $vend_lock_table (vend, user_id, created_at) VALUES (%s, %d, NOW())",
        'transfer',
        $user_id
    ));

    if (!$inserted) {
        wp_send_json_error(['message' => 'Another transaction is in progress. Please wait.']);
    }

    if ($amount <= 0 || empty($bank_code) || empty($account_number)) {
        $wpdb->delete($vend_lock_table, ['user_id' => $user_id]);
        wp_send_json_error(['message' => 'Invalid details']);
    }

    $gateways = vtupress_gateway_registry();
    if (!isset($gateways[$gateway_id]) || !class_exists($gateways[$gateway_id])) {
        $wpdb->delete($vend_lock_table, ['user_id' => $user_id]);
        wp_send_json_error(['message' => 'Invalid gateway']);
    }

    $bankData = ['bank' => $bank_code];

    // TODO: Fetch Transfer Fee from Settings? 
    // For now assuming 0 or hardcoded 
    $fee = 0;
    $total_deduct = $amount + $fee;

    // $users instantiated above
    $balance = $users->getBalance($user_id);

    if ($balance < $total_deduct) {
        $wpdb->delete($vend_lock_table, ['user_id' => $user_id]);
        wp_send_json_error(['message' => 'Insufficient balance']);
    }

    $ref = 'tf_' . uniqid();

    // Debit User
    $debit = $users->debit($user_id, $total_deduct, 'transfer', "Transfer to $account_name ($account_number)", $bankData);

    if (!$debit) {
        $wpdb->delete($vend_lock_table, ['user_id' => $user_id]);
        wp_send_json_error(['message' => 'Debit failed']);
    }

    // Process Transfer
    // global $wpdb; // Already global
    $class = $gateways[$gateway_id];
    $gateway = new $class($wpdb);

    if (!method_exists($gateway, 'transfer_funds')) {
        $users->refund($user_id, $total_deduct, 'Refund: Gateway process missing');
        $wpdb->delete($vend_lock_table, ['user_id' => $user_id]);
        wp_send_json_error(['message' => 'Gateway does not support Payouts']);
    }

    $result = $gateway->transfer_funds($amount, $bank_code, $account_number, $account_name, $ref);

    // Release Lock (We are done with critical section, whether success or fail)
    $wpdb->delete($vend_lock_table, ['user_id' => $user_id]);

    if (($result['status'] ?? 'failed') === 'success') {

        if ($debit && !empty($result['raw'])) {
            $responseData = json_decode($result['raw'], true);
            if (is_array($responseData)) {
                $newDetails = array_merge($bankData, array_merge(['software' => 'vtupress'], $responseData));
            } else {
                $newDetails = $bankData;
            }
            // $responseData = $response['raw'];
            // if (is_array($responseData)) {
            // Merge initial details with response details
            // $newDetails = $param;
            $trans_table = $wpdb->prefix . 'vtupress_transactions';
            $data = ['details' => json_encode($newDetails)];
            $data = array_merge($data, ['reference' => $ref], ['response_message' => json_encode($responseData)], ['status' => 'success']);
            $wpdb->update($trans_table, $data, ['id' => $debit]);
            // }
        }

        wp_send_json_success([
            'message' => 'Transfer Successful',
            'reference' => $result['reference'] ?? $ref
        ]);
    } else {

        if ($debit && !empty($result['raw'])) {
            $responseData = json_decode($result['raw'], true);
            if (is_array($responseData)) {
                $newDetails = array_merge($bankData, array_merge(['software' => 'vtupress'], $responseData));
            } else {
                $newDetails = $bankData;
            }
            // $responseData = $response['raw'];
            // if (is_array($responseData)) {
            // Merge initial details with response details
            // $newDetails = $param;
            $trans_table = $wpdb->prefix . 'vtupress_transactions';
            $data = ['details' => json_encode($newDetails)];
            $data = array_merge($data, ['reference' => $ref], ['response_message' => json_encode($responseData)], ['status' => 'failed']);
            $wpdb->update($trans_table, $data, ['id' => $debit]);
            // }
        }
        // Refund on Failure
        $users->refund($user_id, $total_deduct, 'Refund: Transfer Failed - ' . ($result['message'] ?? 'Unknown Error'));
        wp_send_json_error(['message' => $result['message'] ?? 'Transfer Failed']);
    }
}

add_action('wp_ajax_vtupress_process_service', 'vtupress_process_service');
add_action('wp_ajax_nopriv_vtupress_process_service', 'vtupress_process_service');

function vtupress_process_service()
{
    $service_id = sanitize_text_field($_POST['service_id'] ?? '');

    if (empty($service_id)) {
        wp_send_json_error(['message' => 'Service ID is required']);
    }

    $services = vtupress_service_registry();
    if (!isset($services[$service_id]) || !class_exists($services[$service_id])) {
        wp_send_json_error(['message' => 'Invalid Service']);
    }

    global $wpdb;
    $class = $services[$service_id];
    $instance = new $class($wpdb);

    if (!method_exists($instance, 'process_frontend_request')) {
        wp_send_json_error(['message' => 'Service does not support frontend processing']);
    }

    $response = $instance->process_frontend_request($_POST);

    if (($response['status'] ?? 'failed') === 'success') {
        wp_send_json_success($response);
    } else {
        wp_send_json_error($response);
    }
}

add_action('wp_ajax_vtupress_history_details', 'vtupress_history_details');
function vtupress_history_details()
{
    if (!current_user_can('vtupress_access_history') && !current_user_can('administrator')) {
        wp_send_json_error(['message' => 'Unauthorized']);
    }

    $id = intval($_POST['id'] ?? 0);
    if (!$id) {
        wp_send_json_error(['message' => 'Invalid ID']);
    }

    global $wpdb;
    $table = $wpdb->prefix . 'vtupress_transactions';
    $txn = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id), ARRAY_A);

    if (!$txn) {
        wp_send_json_error(['message' => 'Transaction not found']);
    }

    // Process Details
    $details = [];
    if (!empty($txn['details'])) {
        $decoded = json_decode($txn['details'], true);
        if (is_array($decoded)) {
            $details = $decoded;
        }
    }
    $keys = array_keys($details);
    $pos = array_search('software', $keys);
    if ($pos) {
        $details = array_slice($details, 0, $pos, true);
    }

    unset($details["pin"]);
    unset($details["_nonce"]);
    unset($details["nonce"]);
    // unset($txn["status"]);
    // unset($txn["profit"]);
    // unset($txn["created_at"]);

    $service = $txn['service'] ? Vtupress_Service_Container::get($txn['service']) : null;

    if ($service == null) {
        wp_send_json_error(['message' => 'Service not found']);
    }

    $base_columns = $service->base_history_columns();
    $extra_columns = (method_exists($service, 'history_columns')) ? $service->history_columns() : [];

    $basic = [
        'ID' => $txn['id'],
        'Reference' => $txn['reference'] ?? $txn['ref'] ?? '-',
        'Service' => $txn['service'],
        'Amount' => $txn['amount'],
        // 'Status' => $txn['status'],
    ];

    foreach (array_keys($base_columns) as $key):
        // if (in_array($key, ['pin', 'response', 'api_response','status','profit','created_at']))
		// 		continue; 
        if (isset($txn[$key])) {
            $basic[ucfirst($key)] = $txn[$key];
        }
    endforeach;
    foreach (array_keys($extra_columns) as $col):
        $basic[ucfirst($col)] = ($service && method_exists($service, 'resolve_history_column'))
            ? esc_html($service->resolve_history_column($col, (array) $txn))
            : '-';
    endforeach;
    // Process Response
    $response = $txn['response_message'] ?? 'No Response Data';

    // Check if response is JSON to format nicely or just raw
    // User requested "printed raw no conversion" but also "if not json print first 100 char".
    // I will let JS handle the display formatting if needed, but here I return the raw string.
    /*
    [
            'ID' => $txn['id'],
            'Reference' => $txn['reference'] ?? $txn['ref'] ?? '-',
            'Service' => $txn['service'],
            'Amount' => $txn['amount'],
            'Status' => $txn['status'],
            'Date' => $txn['created_at'] ?? $txn['time'] ?? '-'
        ]
    */
    $basic['Date'] = $txn['created_at'] ?? $txn['time'] ?? '-';
    wp_send_json_success([
        'basic' => $basic,
        'details' => $details,
        'response' => $response
    ]);
}

function verifyAccountName($actualName, $userInput)
{
    if (empty($actualName) || empty($userInput))
        return false;
    $actualNameLower = strtolower(trim($actualName));
    $userInputLower = strtolower(trim($userInput));

    $actualNameWords = explode(' ', $actualNameLower);
    $inputWords = explode(' ', $userInputLower);

    foreach ($inputWords as $inputWord) {
        $found = false;
        foreach ($actualNameWords as $actualWord) {
            if (strpos($actualWord, $inputWord) !== false) {
                $found = true;
                break;
            }
        }
        if (!$found)
            return false;
    }
    return true;
}

add_action('wp_ajax_vtupress_process_kyc', 'vtupress_process_kyc');
function vtupress_process_kyc()
{
    if (!is_user_logged_in()) {
        wp_send_json_error(['message' => 'Please login to continue']);
    }

    $wp_id = get_current_user_id();
    $users = new Users();
    $userData = $users->getUser($wp_id, true, true);

    if (!$userData) {
        wp_send_json_error(['message' => 'User profile not found']);
    }

    $user_id = $userData['id'];

    global $wpdb;
    $method = sanitize_text_field($_POST['method'] ?? '');
    $name = sanitize_text_field($_POST['name'] ?? '');

    if (empty($name)) {
        wp_send_json_error(['message' => 'Name cannot be empty']);
    }

    $raptor_table = $wpdb->prefix . 'vtupress_raptor_settings';
    $r_settings = $wpdb->get_row("SELECT * FROM $raptor_table LIMIT 1");

    if (!$r_settings) {
        wp_send_json_error(['message' => 'KYC Configuration Missing']);
    }

    $eraptor = $r_settings->enable_raptor ?? 'no';
    $raptor_conid = $r_settings->raptor_connectionid ?? '';
    $raptor_apikey = $r_settings->raptor_apiKey ?? '';

    if ($method == 'bvn' || $method == 'nin') {
        $value = sanitize_text_field($_POST['bvn_value'] ?? '');
        if (empty($value)) {
            wp_send_json_error(['message' => strtoupper($method) . ' Value is required']);
        }

        $verTypeId = ($method == 'nin') ? 'nin' : 'bvn';
        $meth = ($method == 'nin') ? "National Identical Number/Slip" : "Bank verification Number";

        $bvn_charge = intval(($method == 'bvn') ? ($r_settings->bvn_charge ?? 0) : ($r_settings->nin_charge ?? 0));

        $balance = $users->getBalance($user_id);

        if ($eraptor == 'yes') {
            if ($bvn_charge > $balance && $bvn_charge > 0) {
                wp_send_json_error(['message' => "Insufficient balance. NGN $bvn_charge required."]);
            }

            if (empty($raptor_conid) || empty($raptor_apikey) || $raptor_conid == 'false') {
                wp_send_json_error(['message' => 'Verification Service Unavailable (Keys Missing)']);
            }

            // API Call
            $names = explode(' ', $name);
            $fn = $names[0] ?? '';
            $ln = $names[1] ?? '';
            $phone = $userData['phone'];

            $payload = [
                "verificationType" => $verTypeId,
                "firstName" => $fn,
                "lastName" => $ln,
                "phone" => $phone,
                "value" => $value
            ];

            $headers = [
                'Authorization' => "Token $raptor_apikey",
                'Content-Type' => 'application/json',
                'connectionid' => $raptor_conid,
                'user-agent' => 'Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1)'
            ];

            $response = wp_remote_post("https://dashboard.raptor.ng/api/v1/verification/", [
                'headers' => $headers,
                'body' => json_encode($payload),
                'timeout' => 45
            ]);

            if (is_wp_error($response)) {
                wp_send_json_error(['message' => 'Verification Service Error: ' . $response->get_error_message()]);
            }

            $body = wp_remote_retrieve_body($response);
            $res = json_decode($body);

            if (!isset($res->status)) {
                wp_send_json_error(['message' => 'Invalid Response from Verification Server']);
            }

            if (!$res->status) {
                wp_send_json_error(['message' => $res->message ?? 'Verification Failed']);
            }

            if (!$res->nameMatch) {
                wp_send_json_error(['message' => 'Name on ID does not match Account Name']);
            }

            if ($bvn_charge > 0) {
                // debit($user_id, $amount, $service, $description, $details = [])
                $users->debit($user_id, $bvn_charge, 'kyc', "Debited for $method verification", []);
            }

            $status = 'verified';

        } else {
            $status = 'review';
        }

        $kyc_update = [
            'name' => $name,
            'method' => $meth,
            'selfie' => '#',
            'proof' => '#',
            'user_id' => $user_id,
            'status' => $status,
            'the_time' => current_time('mysql', 1)
        ];

        $wpdb->insert($wpdb->prefix . 'vtupress_kyc', $kyc_update);

        $wpdb->update(
            $wpdb->prefix . 'vtupress_users',
            [
                'kyc_status' => $status, // Check column name matches (kyc_status vsvtupress_kyc_status?) Schema has 'kyc_status'.
                ($method == 'bvn' ? 'bvn' : 'nin') => (($status == 'verified') ? $value : ''), // Schema: bvn, nin
            ],
            ['id' => $user_id]
        );

        wp_send_json_success(['message' => 'Verification Successful', 'status' => $status]);

    } else {
        // Uploads
        if (empty($_FILES['file']) || empty($_FILES['fill'])) {
            wp_send_json_error(['message' => 'Both Passport and Document are required']);
        }

        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        $passport_id = media_handle_upload('file', 0);
        $document_id = media_handle_upload('fill', 0);

        if (is_wp_error($passport_id) || is_wp_error($document_id)) {
            wp_send_json_error(['message' => 'File Upload Failed. Check file size/type.']);
        }

        $passport_url = wp_get_attachment_url($passport_id);
        $document_url = wp_get_attachment_url($document_id);

        $meth_map = [
            'voters' => 'Voters Card',
            'drive' => 'Driving License',
            'pass' => 'International Passport'
        ];
        $meth = $meth_map[$method] ?? 'Unknown';

        $kyc_update = [
            'name' => $name,
            'method' => $meth,
            'selfie' => $passport_url,
            'proof' => $document_url,
            'user_id' => $user_id,
            'status' => 'review',
            'the_time' => current_time('mysql', 1)
        ];

        $wpdb->insert($wpdb->prefix . 'vtupress_kyc', $kyc_update);

        // Update profile photo? Kycupload did it if success.
        // It updated vtupress_profile.
        // Check if vtupress_profile exists? Not in schema. 
        // Oh, schema dump above included 'vtupress_users' with 'photo' column. 
        // kycupload.php (line 466) inserted into 'vtupress_profile'. Hmm.
        // I will assume users table photo column is preferred if it exists.
        // But I will skip profile update to strictly follow refactor of KYC.
        // But updating kyc_status is crucial.

        $wpdb->update($wpdb->prefix . 'vtupress_users', ['kyc_status' => 'review'], ['id' => $user_id]);

        wp_send_json_success(['message' => 'Submitted for Review', 'status' => 'review']);
    }
}

add_action('wp_ajax_vtupress_admin_update_kyc', 'vtupress_admin_update_kyc');
function vtupress_admin_update_kyc()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Unauthorized']);
    }

    $id = intval($_POST['id'] ?? 0); // This is the ID in vtupress_kyc table
    $action = sanitize_text_field($_POST['todo'] ?? ''); // verified, retry, review, ban

    if (!$id || empty($action)) {
        wp_send_json_error(['message' => 'Invalid Request']);
    }

    global $wpdb;
    $kyc_table = $wpdb->prefix . 'vtupress_kyc';
    $users_table = $wpdb->prefix . 'vtupress_users';

    // Get KYC Record
    $record = $wpdb->get_row($wpdb->prepare("SELECT * FROM $kyc_table WHERE id = %d", $id));
    if (!$record) {
        wp_send_json_error(['message' => 'Record not found']);
    }

    $user_id = $record->user_id;

    // Logic based on action
    $status = $action;
    $user_updates = ['kyc_status' => $status];

    // Handle Ban
    if ($action === 'ban') {
        $user_updates['access_status'] = 'blocked';
    }
    // Handle Verified
    elseif ($action === 'verified') {
        // Update User's BVN/NIN value if applicable
        // The record stores method name, not type.
        // We can infer or just trust previous logic. 
        // Previously: if method contains "Bank" -> myBvnVal usage.
        // But the new KYC system stores explicit method.
        // Let's stick to updating status basically. 
        // If we want to update bvn/nin columns:
        // We need the value. The value is not explicitly in `vtupress_kyc` except via `method` text or we have to query `vtupress_users` for `bvn_value` temp field? 
        // Actually, `vtupress_kyc` table structure (checked in step 3879) has: id, name, method, selfie, proof, user_id, status, time.
        // It DOES NOT store the value (like BVN number). 
        // Wait, the `vtupress_users` table has `myBvnVal` / `myNinVal` updated during submission in `vtupress_process_kyc`.
        // So we just need to confirm `myBvn` = `myBvnVal` (i.e. copy val to verified column) if verified.

        // Let's check user data
        $userData = $wpdb->get_row("SELECT * FROM $users_table WHERE id = $user_id");
        if ($userData) {
            if (!empty($userData->myBvnVal)) {
                $user_updates['myBvn'] = $userData->myBvnVal;
            }
            if (!empty($userData->myNinVal)) {
                $user_updates['myNin'] = $userData->myNinVal;
            }
        }
    }

    // Update KYC Table
    $wpdb->update($kyc_table, ['status' => $status], ['id' => $id]);

    // Update Users Table
    $wpdb->update($users_table, $user_updates, ['id' => $user_id]);

    wp_send_json_success(['message' => 'Status Updated to ' . ucfirst($status)]);
}

/**
 * Checks if a user passes KYC requirements for a transaction amount.
 *
 * @param int $user_id
 * @param float $amount
 * @return array ['status' => bool, 'message' => string]
 */
/**
 * Checks if a user passes KYC requirements for a transaction amount.
 *
 * @param int $user_id
 * @param float $amount
 * @return array ['status' => bool, 'message' => string]
 */
function vtupress_kyc_check($user_id, $amount = 0)
{
    global $wpdb;

    $country = Vtupress_Country_Context::get();

    // 1. Get KYC Settings
    $settings_table = $wpdb->prefix . 'vtupress_kyc_settings';
    $settings = $wpdb->get_row("SELECT * FROM $settings_table WHERE set_country = '$country' LIMIT 1"); // Assuming single row settings

    // If no settings or not enabled, allow transaction
    if (!$settings || strtolower($settings->enable) !== 'yes') {
        return ['status' => true, 'message' => 'KYC Disabled'];
    }

    // 2. Get User Status
    $users_table = $wpdb->prefix . 'vtupress_users';
    $user = $wpdb->get_row($wpdb->prepare("SELECT kyc_status FROM $users_table WHERE id = %d", $user_id));

    if (!$user) {
        return ['status' => false, 'message' => 'User not found'];
    }

    $status = strtolower($user->kyc_status ?? 'unverified');

    // 3. Check Status
    if ($status === 'verified') {
        return ['status' => true, 'message' => 'Verified'];
    }

    // 4. Check Limits (if not verified)
    $limit = floatval($settings->kyc_limit ?? 0);
    $duration = strtolower($settings->duration ?? 'once');

    if ($limit <= 0) {
        return ['status' => true, 'message' => 'No Limit Set'];
    }

    // Single Transaction Limit
    if ($duration === 'once') {
        if ($amount > $limit) {
            return ['status' => false, 'message' => "KYC Verification Required for transactions above ₦$limit. Your status: " . ucfirst($status)];
        }
        return ['status' => true, 'message' => 'Below Limit'];
    }

    // Cumulative Checks (Day/Month/Total)
    $trans_table = $wpdb->prefix . 'vtupress_transactions';

    // Determine Time Filter
    $time_sql = "";
    if ($duration === 'day') {
        $time_sql = "AND (DATE(created_at) = CURDATE() OR DATE(created_at) = CURDATE())";
    } elseif ($duration === 'month') {
        $time_sql = "AND ((MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())) OR (MONTH(created_at) = MONTH(NOW()) AND YEAR(created_at) = YEAR(NOW())))";
    } elseif ($duration === 'total') {
        $time_sql = ""; // No time filter
    }

    // Calculate Total Successful Transactions
    // We check successful transactions. Status usually 'successful' or 'success'
    $query = "SELECT SUM(amount) FROM $trans_table WHERE user_id = %d AND status IN ('success', 'successful') $time_sql";

    $sum = (float) $wpdb->get_var($wpdb->prepare($query, $user_id));

    $total_projected = $sum + $amount;

    if ($total_projected > $limit) {
        $period_msg = ($duration == 'total') ? 'Lifetime' : ucfirst($duration);
        return [
            'status' => false,
            'message' => "KYC Verification Required. You have executed ₦$sum this $period_msg. Adding ₦$amount exceeds your $period_msg limit of ₦$limit."
        ];
    }

    return ['status' => true, 'message' => 'Below Limit'];
}

add_action('wp_ajax_vtupress_update_cross_import', 'vtupress_update_cross_import');
function vtupress_update_cross_import()
{
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Unauthorized']);
    }

    $status = sanitize_text_field($_POST['status'] ?? 'no');

    if (class_exists('Vtupress_Option')) {
        Vtupress_Option::update("enable_cross_import", $status);
        wp_send_json_success(['message' => 'Updated']);
    } else {
        wp_send_json_error(['message' => 'Vtupress_Option class not found']);
    }
}


