<?php
if (!defined('ABSPATH'))
    exit;

/**
 * Container for Gateway Instances (Lazy Loading)
 */
class Vtupress_Gateway_Container
{
    protected static array $instances = [];

    /**
     * Get a specific gateway instance by ID
     */
    public static function get(string $id)
    {
        global $wpdb;
        $registry = vtupress_gateway_registry();

        if (!isset($registry[$id])) {
            return null;
        }

        if (!isset(self::$instances[$id])) {
            $class = $registry[$id];
            if (class_exists($class)) {
                $sv = new $class($wpdb);
            } else {
                return null;
            }
            $country = Vtupress_Country_Manager::validate($sv->current_country); //makes sure country exists in all country and its enabled (once system is configured)

            if (is_wp_error($country)) {
                // You can log it easily: error_log($country->get_error_message());
                return null;
            }

            if(!empty($sv->custom_id)){
                if(Vtupress_Option::get("vtupress_custom_{$sv->custom_id}", "false") != "yes"){
                    return null;
                }
            }

            if (
                !in_array('all', $sv->countries) &&
                !in_array($country, $sv->countries)
            ) {
                return null;
            }

            

            self::$instances[$id] = $sv;
        }

        return self::$instances[$id] ?? null;
    }

    /**
     * Get all registered gateway instances
     */
    public static function all(): array
    {
        $gateways = [];
        foreach (vtupress_gateway_registry() as $id => $class) {
            $instance = self::get($id);
            if (!$instance) {
                continue;
            }
            if ($instance) {
                $gateways[$id] = $instance;
            }
        }
        return $gateways;
    }

    /**
     * Iterate through all gateways and let them try to handle the webhook
     */
    public static function handle_webhooks()
    {
        $gateways = self::all();
        // error_log("Handling webhooks. Found " . count($gateways) . " gateways.");
        foreach ($gateways as $id => $gateway) {
            // error_log("Checking gateway: $id");
            if ($gateway->handle_webhook()) {
                // error_log("Gateway $id handled the webhook.");
                exit; // Webhook handled
            }
        }

        // If we reach here, no gateway handled it
        // error_log("Webhook not handled by any gateway.");
        http_response_code(400);
        die("Webhook not handled");
    }
    /**
     * Render Account Tabs (Unified Logic for Admin & Frontend)
     */
    public static function render_account_tabs(int $user_id, string $context = 'frontend')
    {
        //Logic: $user_id here is the WordPress User ID. We need to convert it to the VTUPress User ID to fetch accounts.
        if (!class_exists('Users')) {
            return;
        }
        $users = new Users();
        $vp_user = $users->getUser($user_id, false, true);
        
        if(empty($vp_user)){
             // If VP user doesn't exist, we can't show accounts.
             return;
        }
        $vp_user_id = $vp_user->id; // This is the VP ID

        global $wpdb;

        $manual = self::get('manual');
        if (!$manual)
            return;

        $manual_settings = $manual->get_settings();
        // 1. Get Virtual/Banking Status
        $banking_enabled = Vtupress_Option::get('banking', 'yes') === 'yes';

        // 2. Build Banks List (Virtual) & Onsite Gateways
        $gateways = self::all();
        $banks_list = [];
        $onsite_gateways = [];
        $currency_symbol = Vtupress_Country_Manager::currency(Vtupress_Country_Context::get());

        foreach ($gateways as $gateway_slug => $gateway_obj) {
            $gateway_slug = strtolower($gateway_slug);

            // Check globally disabled status via option (custom_id)

            // A. Check for Onsite Payment (For Form)
            if ($gateway_obj->onsite_payment) {
                if ($gateway_slug !== 'manual') {
                    $gw_settings = $gateway_obj->get_settings();
                    $gw_status = $gw_settings['status'] ?? 'inactive';
                    $gw_show = $gw_settings['show_on_payout'] ?? 'no';

                    if ($gw_status === 'active' && $gw_show === 'yes') {
                        $onsite_gateways[$gateway_slug] = $gateway_obj;
                    }
                }
            }

            // B. Check for Virtual Accounts (Only if Banking is Enabled AND Virtual Payment Enabled in Manual Settings)
            $virtual_gw_enabled = ($manual_settings['virtual_payment'] ?? 'no') === 'yes';

            if ($banking_enabled && $virtual_gw_enabled) {
                $settings = $gateway_obj->get_settings();

                // Check Gateway Status
                if (($settings['status'] ?? 'inactive') !== 'active') {
                    continue;
                }

                if (empty($settings['banks'] ?? [])) {
                    continue;
                }

                $gateway_accounts = array_map('trim', explode(',', $settings['banks'] ?? []));

                foreach ($gateway_accounts as $bank_entry) {
                    $bank_entry = str_replace(' ', '', $bank_entry);
                    if (!preg_match('/^\w+:\w+$/', $bank_entry)) {
                        continue;
                    }
                    [$bank_id, $bank_name] = explode(':', $bank_entry, 2);

                    $charge = $settings["charge"] ?? '0';
                    $formatted_charge = (preg_match("/%/", $charge)) ? $charge : $currency_symbol . $charge;

                    $banks_list[] = [
                        'gateway' => $gateway_slug,
                        'bank_id' => $bank_id,
                        'bank_name' => ucfirst($bank_name),
                        'charge' => $formatted_charge,
                        'tab_id' => $gateway_slug . '-' . strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $bank_name)),
                    ];
                }
            }
        }

        // 3. Get Existing User Accounts
        $accounts_table = $wpdb->prefix . 'vtupress_userAccounts';
        $country = Vtupress_Country_Context::get();
        // Uses VP ID for querying
        $my_accounts = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $accounts_table WHERE userId=%d AND set_country=%s",
                $vp_user_id,
                $country
            ),
            ARRAY_A
        );

        // Prepare Manual Gateway Data if enabled
        $manual_gw_enabled = ($manual_settings['manual_payment'] ?? 'no') === 'yes';
        $onsite_gw_enabled = ($manual_settings['onsite_payment'] ?? 'no') === 'yes';
        $manual_tab_id = 'manual_gateway_tab';

        // Check if ANYTHING is enabled. If not, return early.
        if (empty($banks_list) && !$manual_gw_enabled && (!$onsite_gw_enabled || empty($onsite_gateways))) {
            return;
        }

        ?>
        <div class="mt-4">
            <!-- TABS -->
            <ul class="nav nav-tabs" role="tablist">
                <?php $first = true;
                foreach ($banks_list as $bank): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $first ? 'active' : ''; ?>" data-bs-toggle="tab"
                            href="#<?php echo esc_attr($bank['tab_id']); ?>" role="tab">
                            <?php echo esc_html($bank['bank_name']); ?>
                        </a>
                    </li>
                    <?php $first = false;
                endforeach; ?>

                <?php if ($manual_gw_enabled): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $first ? 'active' : ''; ?>" data-bs-toggle="tab"
                            href="#<?php echo $manual_tab_id; ?>" role="tab">
                            Manual Transfer
                        </a>
                    </li>
                    <?php $first = false; endif; ?>

                <?php if ($onsite_gw_enabled && !empty($onsite_gateways)): ?>
                    <li class="nav-item">
                        <a class="nav-link <?php echo $first ? 'active' : ''; ?>" data-bs-toggle="tab" href="#onsite_payment_tab"
                            role="tab">
                            Fund Wallet
                        </a>
                    </li>
                <?php endif; ?>
            </ul>

            <!-- TAB CONTENT -->
            <div class="tab-content mt-3">
                <?php $first = true;
                foreach ($banks_list as $bank): ?>
                    <?php
                    // Match account
                    $accountName = '';
                    $accountNumber = '';
                    $isActive = false;

                    foreach ($my_accounts as $acct) {
                        if (
                            strtolower($acct['bankName']) === strtolower($bank['bank_name']) &&
                            strtolower($acct['gateway']) === $bank['gateway']
                        ) {
                            $isActive = true;
                            $accountName = $acct['accountName'];
                            $accountNumber = $acct['accountNumber'];
                            break;
                        }
                    }
                    ?>

                    <div class="tab-pane fade <?php echo $first ? 'show active' : ''; ?>"
                        id="<?php echo esc_attr($bank['tab_id']); ?>" role="tabpanel">

                        <div class="card card-dark bg-secondary-gradient">
                            <div class="card-body skew-shadow">
                                <?php
                                $img_src = get_images(strtolower($bank['bank_name']) . '.png', 'gateway_bank');
                                ?>
                                <img src="<?php echo esc_url($img_src); ?>" height="60" width="60" alt="Bank Logo"
                                    class="rounded-circle bg-white p-1">

                                <h4 class="mt-3">
                                    <?php if ($isActive): ?>
                                        Account Number: <?php echo esc_html($accountNumber); ?>
                                    <?php else: ?>
                                        <button class="btn btn-primary btn-sm generate_account"
                                            data-gateway="<?php echo esc_attr($bank['gateway']); ?>"
                                            data-bank-code="<?php echo esc_attr($bank['bank_id']); ?>"
                                            data-bank-name="<?php echo esc_attr($bank['bank_name']); ?>"
                                            data-user-id="<?php echo esc_attr($user_id); ?>">
                                            <i class="fa fa-history me-1"></i> Generate <?php echo esc_html($bank['bank_name']); ?>
                                            Account
                                        </button>
                                    <?php endif; ?>
                                </h4>

                                <?php if ($isActive): ?>
                                    <p><strong>Account Name:</strong>
                                        <?php echo esc_html(get_bloginfo('name')); ?> - <?php echo esc_html($accountName); ?>
                                    </p>
                                <?php endif; ?>

                                <p><strong>Bank:</strong> <?php echo esc_html($bank['bank_name']); ?></p>
                                <p><strong>Gateway:</strong> <?php echo esc_html(ucfirst($bank['gateway'])); ?></p>

                                <div class="col-4 pl-0 text-right">
                                    <h3 class="fw-bold mb-1"><?php echo $bank['charge']; ?></h3>
                                    <div class="text-small text-uppercase fw-bold op-8">Charge</div>
                                </div>
                            </div>
                        </div>

                    </div>
                    <?php $first = false;
                endforeach; ?>

                <?php if ($manual_gw_enabled): ?>
                    <div class="tab-pane fade <?php echo $first ? 'show active' : ''; ?>" id="<?php echo $manual_tab_id; ?>"
                        role="tabpanel">
                        <div class="card card-dark bg-secondary-gradient">
                            <div class="card-body skew-shadow">
                                <img src="<?php echo esc_url($manual->image); ?>" height="60" width="60" alt="Manual"
                                    class="rounded-circle bg-white p-1">
                                <h4 class="mt-3">Manual Bank Transfer</h4>
                                <div class="manual-info mt-3 text-white">
                                    <p><strong>Bank Name:</strong> <?php echo esc_html($manual_settings['bank_name'] ?? ''); ?></p>
                                    <p><strong>Account Name:</strong>
                                        <?php echo esc_html($manual_settings['bank_account'] ?? ''); ?>
                                    </p>
                                    <p><strong>Account Number:</strong>
                                        <?php echo esc_html($manual_settings['account_number'] ?? ''); ?>
                                    </p>
                                </div>
                                <?php if (!empty($manual_settings['information'])): ?>
                                    <div class="alert alert-info mt-2 text-dark" style="color: black !important;">
                                        <?php echo nl2br(esc_html($manual_settings['information'])); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php $first = false; endif; ?>


                <?php if ($onsite_gw_enabled && !empty($onsite_gateways)): ?>
                    <div class="tab-pane fade <?php echo $first ? 'show active' : ''; ?>" id="onsite_payment_tab" role="tabpanel">
                        <div class="card">
                            <div class="card-body">
                                <h4>Fund Wallet</h4>
                                <form method="post" class="cardPayment" action="/wp-content/plugins/vtupress/pay.php">
                                    <div class="form-group">
                                        <label class="form-label">Amount</label>
                                        <input type="number" name="amount" class="form-control mb-2 user_amount" required>
                                    </div>
                                    <div class="form-group">
                                        <label class="form-label">Choose Gateway</label>
                                        <select class="paymentchoice form-control mb-2" name="paymentchoice" required>
                                            <option value="" selected>Select Gateway</option>
                                            <?php foreach ($onsite_gateways as $slug => $gw):
                                                $gw_settings = $gw->get_settings();
                                                $charge = $gw_settings['charge'] ?? '0';
                                                ?>
                                                <option value="<?php echo esc_attr($slug); ?>"
                                                    data-charge="<?php echo esc_attr($charge); ?>">
                                                    <?php echo esc_html($gw->name); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="input-group mt-2">
                                            <span class="input-group-text dcharge text-primary fw-bold">Charge:
                                                <span class="methodinai">0</span>
                                            </span>
                                        </div>
                                    </div>

                                    <!-- Hidden Inputs needed for Legacy compatibility -->
                                    <input type="hidden" name="vpemail"
                                        value="<?php echo esc_attr(get_userdata($user_id)->user_email); ?>"
                                        class="user_email form-control">
                                    <input type="hidden" name="tcode" value="wallet">
                                    <input type="hidden" name="userid" value="<?php echo esc_attr($user_id); ?>">
                                    <input type="hidden" id="id" name="id" value="<?php echo uniqid("VTU-", false); ?>">
                                    <input type="hidden" name="charge" class="charge_back_value" value="0">
                                    <!-- End Hidden -->

                                    <button type="button"
                                        class="btn btn-primary mb-2 form-control p-3 text-white uppercase submit-onsite-payment">
                                        Fund Wallet
                                    </button>
                                    <input type="submit" class="d-none submit-form">
                                </form>
                            </div>
                        </div>

                        <!-- Output Gateway Specific JS -->
                        <?php
                        foreach ($onsite_gateways as $slug => $gw) {
                            if (method_exists($gw, 'get_payment_js')) {
                                echo $gw->get_payment_js();
                            }
                        }
                        ?>

                        <!-- JS Logic for Payment Choice & Submission -->
                        <script>
                            jQuery(document).ready(function ($) {
                                $(".paymentchoice").on("change", function () {
                                    var dchoice = $(this).val();
                                    var selectedOption = $(this).find('option:selected');
                                    var charge = selectedOption.data('charge');

                                    if (!charge) {
                                        $(".charge_back_value").val(0);
                                        $(".dcharge .methodinai").text(0);
                                        return;
                                    }

                                    // Check if percentage
                                    if (String(charge).indexOf('%') !== -1) {
                                        var percentage = parseFloat(charge.replace('%', ''));
                                        $(".charge_back_value").val(percentage); // Legacy usually expects the value here
                                        $(".dcharge .methodinai").text(charge);
                                    } else {
                                        var fixed = parseFloat(charge);
                                        $(".charge_back_value").val(fixed);
                                        $(".dcharge .methodinai").text("<?php echo esc_js($currency_symbol); ?>" + fixed);
                                    }
                                });

                                $('.submit-onsite-payment').on('click', function (event) {
                                    var the_choice = $(".paymentchoice").val();
                                    if (!the_choice) {
                                        alert("Please select a gateway");
                                        return;
                                    }

                                    var amount = $(".user_amount").val();
                                    var email = $(".user_email").val();
                                    var charge_val = $(".charge_back_value").val();

                                    if (!amount || parseFloat(amount) <= 0) {
                                        alert('Please enter a valid amount');
                                        return;
                                    }

                                    // Check for Inline JS Handler
                                    var fnName = 'vtupress_pay_' + the_choice;
                                    if (typeof window[fnName] === 'function') {
                                        window[fnName](amount, email, charge_val);
                                        return;
                                    }

                                    // Legacy Fallback
                                    $(".submit-form").click();
                                });
                            });
                        </script>
                    </div>
                <?php endif; ?>

            </div>
        </div>
        <?php
    }
}

abstract class Vtupress_Gateway
{

    protected wpdb $db;
    public string $id = '';
    public string $name = '';

    public string $current_country = '';
    public array $countries = [];
    public bool $status = false;
    public string $description = '';
    public string $image = '';
    public bool $virtual_account = false;
    public bool $onsite_payment = false;
    public bool $require_nin = true;
    public bool $require_bvn = true;
    public bool $disable_global = false;

    public $custom_id = '';

    public function get_banks(): array
    {
        return [];
    }

    public function resolve_account(string $bank_code, string $account_number): array
    {
        return ['status' => 'failed', 'message' => 'Not implemented'];
    }

    public function transfer_funds(float $amount, string $bank_code, string $account_number, string $account_name, string $ref): array
    {
        return ['status' => 'failed', 'message' => 'Not implemented'];
    }


    public function __construct(wpdb $db)
    {
        $this->db = $db;
        $this->current_country = Vtupress_Country_Context::get();
    }

    public function calculate_charge(float $amount): float
    {
        $settings = $this->get_settings();
        $charge = $settings['charge'] ?? '0';
        $funding_threshold = isset($settings['funding_threshold']) ? (float) $settings['funding_threshold'] : 0;
        $funding_threshold_fee = $settings['funding_threshold_fee'] ?? '0';

        // Check if amount meets threshold
        if ($funding_threshold > 0 && $amount >= $funding_threshold) {
            $fee_setting = $funding_threshold_fee;
        } else {
            $fee_setting = $charge;
        }

        // Calculate Fee
        if (strpos($fee_setting, '%') !== false) {
            $percentage = (float) str_replace('%', '', $fee_setting);
            return ($amount * $percentage) / 100;
        } else {
            return (float) $fee_setting;
        }
    }

    public function process_funding_reward(int $user_id, float $amount, string $reference)
    {
        global $wpdb;

        $users = new User();
        // 1. Get User Data
        $user_data = get_userdata($user_id);
        if (!$user_data)
            return;

        $user_row = $this->db->get_row($this->db->prepare("SELECT * FROM {$this->db->prefix}vtupress_users WHERE userId = %d", $user_id));
        if (!$user_row)
            return;

        $referrer_id = $user_row->referrer_id;
        $users = new Users();
        $vp_ref_user_id = $users->getUser($referrer_id,false,true)->id ?? 0;


        // Skip if no valid referrer (assuming '1' is admin/default)
        if (empty($referrer_id) || $referrer_id == '1' || $vp_ref_user_id == 0)
            return;

        // 2. Get User's Level/Package
        $plan = $user_row->plan ?? 'customer';
        // Get Level Configuration
        $level_table = $this->db->prefix . "vtupress_levels";
        $level_config = $this->db->get_row($this->db->prepare("SELECT * FROM $level_table WHERE name = %s LIMIT 1", $plan));

        if (!$level_config)
            return;

        // 3. Check Fund Minimum
        $fund_min = (float) ($level_config->referer_fund_minimum ?? 0);
        if ($amount < $fund_min)
            return;

        // 4. Check Fund Count
        $fund_count_limit = (int) ($level_config->referer_fund_count ?? 0);
        if ($fund_count_limit > 0) {
            $webhook_table = $this->db->prefix . 'vtupress_wallet_webhook';
            // Count SUCCESSFUL webhooks for this user.
            // NOTE: This includes the current one if it was already logged!
            // If the limit is 1, and we have 1 (current), we should pay.
            // If we have 2, we should NOT pay.
            $previous_funds = $this->db->get_var($this->db->prepare(
                "SELECT COUNT(id) FROM $webhook_table WHERE user_id = %d AND status = 'success'",
                $user_row->id
            ));

            if ($previous_funds > $fund_count_limit)
                return;
        }

        // 5. Credit Direct Referrer
        $direct_bonus_config = $level_config->referer_fund_bonus ?? '0';
        $direct_bonus = 0;
        if (strpos($direct_bonus_config, '%') !== false) {
            $pct = (float) str_replace('%', '', $direct_bonus_config);
            $direct_bonus = ($amount * $pct) / 100;
        } else {
            $direct_bonus = (float) $direct_bonus_config;
        }

        if ($direct_bonus > 0) {
            $referrer_user = get_userdata($referrer_id);
            // Ensure referrer exists
            if ($referrer_user) {
                //check for the user_id
                $users->credit(
                    (int) $vp_ref_user_id,
                    $direct_bonus,
                    'bonus',
                    "Referral Bonus for Funding by " . $user_data->user_login,
                    ['child_id' => $user_row->id, 'amount' => $amount]
                );
            }
        }

        // 6. Generational Rewards
        $gen_table = $this->db->prefix . 'vtupress_levelGeneration';
        $current_upline = $vp_ref_user_id;
        $max_gens = 10;

        // Start from Gen 2 (Upline of Direct Referrer)
        $upline_row = $this->db->get_row($this->db->prepare("SELECT referrer_id FROM {$this->db->prefix}vtupress_users WHERE id = %d", $current_upline));
        $current_upline = $upline_row ? $upline_row->referrer_id : null;

        for ($i = 2; $i <= $max_gens; $i++) {
            if (empty($current_upline) || $current_upline == '1')
                break;

            $gen_reward_config = $this->db->get_var($this->db->prepare(
                "SELECT fund_commission FROM $gen_table WHERE package_name = %s AND generation = %s",
                $plan,
                $i
            ));

            if ($gen_reward_config && $gen_reward_config != '0') {
                $gen_bonus = 0;
                if (strpos($gen_reward_config, '%') !== false) {
                    $pct = (float) str_replace('%', '', $gen_reward_config);
                    $gen_bonus = ($amount * $pct) / 100;
                } else {
                    $gen_bonus = (float) $gen_reward_config;
                }

                if ($gen_bonus > 0) {
                    $users = new Users();
                    $users->credit(
                        (int) $current_upline,
                        $gen_bonus,
                        'bonus',
                        "Gen $i Referral Bonus for Funding by " . $user_data->user_login,
                        ['child_id' => $user_row->id, 'amount' => $amount, 'generation' => $i]
                    );
                }
            }

            // Move to next upline
            $upline_row = $this->db->get_row($this->db->prepare("SELECT referrer_id FROM {$this->db->prefix}vtupress_users WHERE id = %d", $current_upline));
            $current_upline = $upline_row ? $upline_row->referrer_id : null;
        }

    }

    private function default_settings(array $fields): array
    {
        $defaults = [];

        foreach ($fields as $key => $field) {
            if (
                isset($field['render']['value']) &&
                is_array($field['render']['value'])
            ) {
                $values = $field['render']['value'] ?? [];

                $firstKey = array_key_first($values);

                if ($firstKey !== null) {
                    $defaults[$key] = is_string($firstKey) ? $firstKey : $values[$firstKey];
                } else {
                    $defaults[$key] = '';
                }

            } else {
                $defaults[$key] = $field['render']['value'] ?? '';
            }
        }

        return $defaults;
    }
    public function maybe_save_settings(): void
    {
        if (
            empty($_POST['vtupress_gateway_save']) ||
            empty($_POST['_wpnonce']) ||
            !wp_verify_nonce($_POST['_wpnonce'], 'vtupress_save_gateway_' . $this->id . "_{$this->current_country}")
        ) {
            return;
        }

        if (!current_user_can('manage_options')) {
            return;
        }


        $settings = [];
        $save_fields = $this->default_settings(array_merge($this->global_fields(), $this->form_fields()));
        foreach ($save_fields as $key => $val) {

            $settings[$key] = sanitize_text_field($_POST[$key] ?? '');
        }

        Vtupress_option::update($this->get_option_key(), $settings);

        add_action('vtupress_notices', function () {
            echo '<div class="alert alert-success"><p>Gateway settings saved.</p></div>';
        });
    }

    private function global_fields()
    {
        if ($this->disable_global) {
            return [];
        }

        $defaults = [
            "status" => [
                "label" => "Status",
                "description" => "",
                "render" => [
                    "value" => ["active", "inactive"]
                ]
            ],
            "account_on_registration" => [
                "label" => "Create account on users registration",
                "description" => "Create a virtual account upon user registration? Not recommended!!!",
                "show" => $this->virtual_account && (!$this->require_bvn && !$this->require_nin),
                "render" => [
                    "value" => ["yes", "no"]
                ]
            ],
            "show_on_payout" => [
                "label" => "Show on payout",
                "description" => "Show this option on payout modal",
                "show" => $this->onsite_payment,
                "render" => [
                    "value" => ["yes", "no"]
                ]
            ],
            "requires_bvn" => [
                "label" => "Require Bvn",
                "description" => "Users must provide bvn before generating account",
                "show" => $this->virtual_account && $this->require_bvn,
                "render" => [
                    "value" => ["no", "yes"]
                ]
            ],
            "requires_nin" => [
                "label" => "Require Nin",
                "description" => "Users must provide bvn before generating account",
                "show" => $this->virtual_account && $this->require_nin,
                "render" => [
                    "value" => ['no', "yes"]
                ]
            ],
            "use_admin_bio" => [
                "label" => "Use admin bio",
                "description" => "User admin biodata instead of users data on account creation",
                "show" => $this->virtual_account && ($this->require_bvn || $this->require_nin),
                "render" => [
                    "value" => ["no", "yes"]
                ]
            ],
            "minimum_funding" => [
                "label" => "Minimum funding",
                "description" => "Minimum funding for onsite payment",
                "render" => [
                    "type" => "number",
                    "value" => "100"
                ]
            ],
            "charge" => [
                "label" => "Funding charge",
                "description" => "e.g 200 for ₦200 charge or 2%",
                "render" => [
                    "value" => ""
                ]
            ],
            "funding_threshold" => [
                "label" => "Funding threshold",
                "description" => "Set an amount where you can charge differently when a user fund up to this threshold",
                "render" => [
                    "type" => "number",
                    "value" => ""
                ]
            ],
            "funding_threshold_fee" => [
                "label" => "Funding threshold fee",
                "description" => "A special charge when a user funds up to the funding threshold e.g 200 for ₦200 charge or 2%",
                "render" => [
                    "value" => ""
                ]
            ],

        ];

        return $defaults;

    }
    protected function formData(array $form = []): void
    {

        $defaults = $this->global_fields();
        $form = $this->form_fields();

        $form = array_merge($defaults, $form);
        $this->render($form);
    }
    private function render(array $form): void
    {
        $settings = $this->get_settings();
        ?>
        <div class="container-fluid container-md">
            <div class="row bg-info bordered-bottom">
                <div class="col-12 d-flex justify-content-center align-items-center py-2">
                    <span class="heading">
                        <h3 class="text-white "><i class="mdi mdi-bank me-2"></i><?php echo ucfirst($this->name); ?> Gateway
                            Settings
                        </h3>
                    </span>
                </div>
            </div>
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-4 d-flex justify-content-center">
                                    <img src="<?php echo esc_url($this->image); ?>" alt="<?php echo $this->name; ?>"
                                        class="img-fluid">
                                </div>
                                <div class="col-md-8 d-flex flex-column justify-content-center">
                                    <h4 class="card-title"><?php echo ucfirst($this->name); ?> Gateway Information</h4>
                                    <p class="card-text"><?php echo $this->description; ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <form method="post" class="row rounded bg-white p-2 bordered py-md-3 px-md-3 toSave ">
                <!-- <form method="post"> -->
                <div class="col-12 my-3">
                    <?php do_action('vtupress_notices'); ?>
                </div>

                <?php wp_nonce_field('vtupress_save_gateway_' . $this->id . "_{$this->current_country}"); ?>
                <input type="hidden" name="vtupress_gateway_save" value="1">
                <?php

                foreach ($form as $key => $value) {

                    $show = $value["show"] ?? true;
                    if (!$show) {
                        continue;
                    }
                    $column = esc_html($value["column"] ?? "4");
                    $class = esc_html($value["class"] ?? "");
                    $inner_class = esc_html($value["render"]["class"] ?? "");

                    ?>
                    <div class="col-6 col-md-<?php echo $column . " " . $class; ?> my-3">
                        <label
                            for="<?php echo esc_html($value['label'] ?? ''); ?>"><?php echo esc_html($value['label'] ?? ''); ?>:</label>
                        <?php
                        //is_bool($value["render"]["value"])
                        if (is_array($value["render"]["value"])) {
                            ?>
                            <select name="<?php echo $key; ?>" class="form-control <?php echo $inner_class; ?>">
                                <?php foreach ($value["render"]["value"] as $value2) {
                                    $f_key = strtolower($value2);
                                    $f_value = ucfirst($value2);
                                    ?>
                                    <option value="<?php echo esc_attr($f_key); ?>" <?php selected($settings[$key] ?? '', $f_key); ?>>
                                        <?php echo esc_html($f_value); ?>
                                    </option>

                                <?php } ?>
                            </select>

                            <?php
                        } else {
                            $type = esc_html($value["render"]["type"] ?? 'text');
                            $field_value = esc_html($value["render"]["value"] ?? '');

                            if ($type === 'textarea') {
                                ?>
                                <textarea class="form-control  <?php echo $inner_class; ?>" id="<?php echo $key; ?>"
                                    name="<?php echo $key; ?>" rows="5"><?php echo $settings[$key] ?? $field_value; ?></textarea>
                                <?php
                            } else {
                                ?>
                                <input class="form-control  <?php echo $inner_class; ?>" id="<?php echo $key; ?>"
                                    type="<?php echo $type; ?>" name="<?php echo $key; ?>"
                                    value="<?php echo $settings[$key] ?? $field_value; ?>">
                                <?php
                            }
                        }
                        ?>
                        <small><?php echo esc_html($value["description"] ?? ''); ?></small>
                    </div>
                    <?php
                }
                ?>
                <?php submit_button('Save Gateway Settings'); ?>
                <!-- </form> -->
            </form>
        </div>
        <?php
    }
    public function get_settings(): array
    {
        $global_defaults = $this->default_settings($this->global_fields());

        return wp_parse_args(
            Vtupress_Option::get($this->get_option_key(), []),
            array_merge($global_defaults, $this->default_settings($this->form_fields()))
        );
    }
    protected function get_option_key(): string
    {
        return 'vtupress_gateway_' . $this->id . "_{$this->current_country}";
    }


    abstract public function process_payment(float $amount, array $details): array;
    abstract public function form_fields(): array;
    abstract public function render_admin_form(): void;

    public function get_payment_js(): string
    {
        return '';
    }

    /**
     * Create Virtual Account for User
     */
    public function create_virtual_account(int $user_id): array
    {
        return ['status' => 'failed', 'message' => 'Not implemented for this gateway'];
    }

    /**
     * Handle Webhook - Override this in child classes
     * @return bool true if handled, false otherwise
     */
    public function handle_webhook(): bool
    {
        return false;
    }

    protected function log_webhook(int $user_id, string $amount, string $reference, string $status, string $response)
    {
        //get users country
        $users = new Users();
        $user = $users->getUser($user_id,true);


        $table_name = $this->db->prefix . 'vtupress_wallet_webhook';
        $this->db->insert($table_name, [
            'user_id' => $user["id"],
            'gateway' => $this->id,
            'amount' => $amount,
            'referrence' => $reference,
            'status' => $status,
            'response' => $response,
            'the_time' => current_time('mysql'),
            'set_country' => $user['set_country']
        ]);
    }

    protected function get_all_headers_helper(): array
    {
        if (function_exists('getallheaders')) {
            $headers = getallheaders();
        } else {
            $headers = [];
            foreach ($_SERVER as $name => $value) {
                if (substr($name, 0, 5) == 'HTTP_') {
                    $headers[str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))))] = $value;
                }
            }
        }
        // Normalize keys to lowercase for easier lookup
        return array_change_key_case($headers, CASE_LOWER);
    }

    /**
     * Save Generated Account to Database
     */
    protected function save_account(int $user_id, string $bank_name, string $account_number, string $account_name, string $ref = '', string $gateway = '')
    {
        //Logic: $user_id here is the WordPress User ID. We need to convert it to the VTUPress User ID for storage.
        if (!class_exists('Users')) {
            return false;
        }
        $users = new Users();
        // Get valid VP User ID from WP User ID
        $vp_user = $users->getUser($user_id, false, true);
        if(empty($vp_user)){
            return false;
        }
        $vp_user_id = $vp_user->id; // This is the VP ID

        $table = $this->db->prefix . 'vtupress_userAccounts';
        $country = $this->current_country;

        //Check if account exists using VP ID
        $exists = $this->db->get_row($this->db->prepare("SELECT id,set_country FROM $table WHERE userId = %d AND accountNumber = %s AND gateway = %s", $vp_user_id, $account_number, $gateway));

        if ($exists) {
            return true;
        }

        return $this->db->insert($table, [
            'userId' => $vp_user_id, // Storing VP ID
            'bankName' => $bank_name,
            'accountName' => $account_name,
            'accountNumber' => $account_number,
            'accountRef' => $ref,
            'gateway' => $gateway ?: $this->id,
            'set_country' => $exists['set_country'] ?? $country
        ]);
    }

    /**
     * Generate Random Email for APIs that require unique emails
     */
    protected function generate_random_email(): string
    {
        $syllables = ['ab', 'ac', 'ad', 'ba', 'be', 'ca', 'ce', 'da', 'de', 'do', 'fa', 'fi', 'ga', 'ge', 'go', 'ha', 'he', 'hi', 'ja', 'jo', 'ka', 'ko', 'la', 'le', 'li', 'lo', 'ma', 'me', 'mi', 'mo', 'na', 'ne', 'ni', 'no', 'pa', 'pe', 'pi', 'po', 'ra', 're', 'ri', 'ro', 'sa', 'se', 'si', 'so', 'ta', 'te', 'ti', 'to', 'va', 've', 'vi', 'vo', 'wa', 'we', 'wi', 'wo', 'ya', 'yo', 'za', 'ze', 'zi', 'zo'];
        $syllable = $syllables[rand(0, count($syllables) - 1)] . $syllables[rand(0, count($syllables) - 1)] . $syllables[rand(0, count($syllables) - 1)];
        return $syllable . date('mis') . '@gmail.com';
    }

    protected function get_user_details(int $user_id)
    {
        $wp_user = get_userdata($user_id);

        // Try getting internal user using WP ID
        $users_class = new Users();
        $internal_user = $users_class->getUser($user_id, false, true);


        if (!$wp_user || !$internal_user)
            return false;

        $bvn = $internal_user->bvn ?? '';
        $nin = $internal_user->nin ?? '';

        // Check for Manual Gateway Fallback
        $manual = Vtupress_Gateway_Container::get('manual');
        if ($manual) {
            $manual_settings = $manual->get_settings();
            if (($manual_settings['use_admin_bio'] ?? 'no') === 'yes') {
                if (empty($bvn)) {
                    $bvn = $manual_settings['bvn'] ?? '';
                }
                if (empty($nin)) {
                    $nin = $manual_settings['nin'] ?? '';
                }
            }
        }

        return [
            'username' => $wp_user->user_login,
            'email' => $wp_user->user_email,
            'first_name' => $internal_user->first_name ?? '',
            'last_name' => $internal_user->last_name ?? '',
            'phone' => $internal_user->phone ?? '',
            'bvn' => $bvn,
            'nin' => $nin
        ];
    }
}

class Manual_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'manual';
        $this->name = 'General';
        $this->description = 'General gateway control and admin setup';
        $this->status = true;
        $this->countries = ['all'];
        $this->disable_global = true;
        $this->image = VTUPRESS_URL . "/images/bank/manual.jpg";


        parent::__construct($db);
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "manual_payment" => [
                "label" => "Enable manual payment",
                "description" => "",
                "render" => [
                    "value" => ["no", "yes"]
                ]
            ],
            "onsite_payment" => [
                "label" => "Enable onsite payment",
                "description" => "",
                "render" => [
                    "value" => ["no", "yes"]
                ]
            ],
            "virtual_payment" => [
                "label" => "Enable virtual account payment",
                "description" => "",
                "render" => [
                    "value" => ["no", "yes"]
                ]
            ],
            "use_admin_bio" => [
                "label" => "Use Admin Bio (Fallback)",
                "description" => "Use the BVN/NIN provided below if the user does not have one.",
                "render" => [
                    "value" => ["no", "yes"]
                ]
            ],
            "information" => [
                "label" => "Information",
                "description" => "Instruction to display after account details",
                "column" => 12,
                "render" => [
                    "type" => "textarea",
                    "value" => ""
                ]
            ],
            "first_name" => [
                "label" => "First name",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "last_name" => [
                "label" => "Last name",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "email" => [
                "label" => "Email",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "phone" => [
                "label" => "Phone",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "bvn" => [
                "label" => "Bvn",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "nin" => [
                "label" => "Nin",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "bank_name" => [
                "label" => "Bank name",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "bank_account" => [
                "label" => "Bank account",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "account_number" => [
                "label" => "Account number",
                "description" => "",
                "render" => [
                    "value" => ""
                ]
            ],
            "address" => [
                "label" => "Address",
                "description" => "",
                "column" => 12,
                "render" => [
                    "value" => ""
                ]
            ],
        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
}

class Ncwallet_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'ncwallet';
        $this->name = 'Ncwallet';
        $this->description = 'A gateway that generates Providus bank account to users';
        $this->status = true;
        $this->countries = ['nigeria'];
        $this->virtual_account = true;
        $this->custom_id = 'ncwallet';
        $this->image = VTUPRESS_URL . "/images/bank/ncwallet.png";


        parent::__construct($db);
    }

    public function handle_webhook(): bool
    {
        $input = file_get_contents("php://input");
        $event = json_decode(str_replace(" ", "", $input));
        $array = json_decode(str_replace(" ", "", $input), true);

        // Check if event type is set, legacy check
        if (!isset($event->event_type) || !isset($event->settled_amount)) {
            // Maybe not an ncwallet request or invalid format
            return false;
        }

        if (strtoupper($event->event_type) != "COLLECTION") {
            // Not a successful transaction type we care about
            return true; // We handled it (it matches our criteria) but we ignore it
        }

        $headers = $this->get_all_headers_helper();
        $signature = $headers["api-key"] ?? $headers["api_key"] ?? null;

        if (!$signature) {
            return false;
        }

        $settings = $this->get_settings();
        $apikey = $settings['public_key'] ?? ''; // ncwallet_apikey usually public_key based on logic
        if (empty($apikey)) {
            // Legacy fallback
            $apikey = get_option('vtupress_ncwallet_apikey', '');
        }

        $hashed = hash_hmac('sha512', $input, $apikey);
        if ($hashed != $signature) {
            // Mismatch
            return false;
        }

        $email = $array["email"] ?? '';
        $amount = $array["amount"] ?? 0;
        $session_id = $array["session_id"] ?? '';

        if (empty($email) || empty($session_id)) {
            return true;
        }

        // duplicate check
        $existing = $this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_wallet_webhook WHERE referrence = %s AND gateway = %s", $session_id, $this->id));
        if ($existing) {
            http_response_code(200);
            return true;
        }

        // Find user by email (Legacy behavior) - But technically we should use account number if available?
        // Legacy used get_user_by("email", $array["email"])
        // Let's stick to email if that's what they send.

        $user = get_user_by("email", $email);

        if (!$user) {
            $this->log_webhook(0, (string) $amount, $session_id, 'failed', 'User not found by email: ' . $email);
            return true;
        }

        $wp_user_id = $user->ID;
        $users = new Users();
        $user_id = $users->getUser($wp_user_id,false,true)->id ?? 0;
        
        
        $charge = $this->calculate_charge((float) $amount);
        $amount_to_credit = (float) $amount - $charge;

        $credited = $users->credit($user_id, $amount_to_credit, 'deposit', "Wallet Funding via Ncwallet", ['reference' => $session_id, 'gateway' => 'Ncwallet']);

        if ($credited) {
            $this->log_webhook($user_id, (string) $amount, $session_id, 'success', json_encode($array));
            $this->process_funding_reward($wp_user_id, (float) $amount, $session_id);
            http_response_code(200);
        } else {
            $this->log_webhook($user_id, (string) $amount, $session_id, 'failed', 'Credit failed');
            http_response_code(500);
        }

        return true;
    }

    public function create_virtual_account(int $user_id): array
    {
        $settings = $this->get_settings();
        $user = $this->get_user_details($user_id);

        if (!$user)
            return ['status' => 'failed', 'message' => 'User not found'];

        $bvn = $user['bvn'];
        $nin = $user['nin'];
        $admin_bvn = ''; // Would need to fetch from option 'ncwallet_admin_bvn' if implemented, but here we stick to dynamic settings.
        // Or if the user meant dynamic population of what was there. 
        // Logic from old file: checks admin_bvn optional. 
        // For now, enforcing user BVN/NIN as per class properties require_bvn/nin.

        $validation_number = (!empty($bvn) && strlen($bvn) > 9) ? $bvn : ((!empty($nin) && strlen($nin) > 9) ? $nin : '');

        if (empty($validation_number)) {
            return ['status' => 'failed', 'message' => 'BVN or NIN required'];
        }

        $token = $settings['public_key'] ?? ''; // ncwallet_apikey
        $pin = $settings['transaction_pin'] ?? ''; // ncwallet_pin

        $payload = [
            "account_name" => $user['first_name'] . " " . $user['last_name'],
            "bank_code" => "nomba", // Hardcoded in legacy? Or use dynamic bank? Legacy used "nomba".
            "account_type" => "static",
            "email" => $user['email'],
            "validation_type" => "BVN",
            "validation_number" => $validation_number,
            "phone_number" => $user['phone']
        ];


        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://ncwallet.africa/api/v1/bank/create",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_SLASHES),
            CURLOPT_HTTPHEADER => [
                "Authorization: $token",
                "trnx_pin: $pin",
                "Accept: application/json",
                "Content-Type: application/json"
            ],
        ]);

        $res = curl_exec($curl);
        $response = json_decode($res, true);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err)
            return ['status' => 'failed', 'message' => $err];

        if (($response["status"] ?? '') !== "success") {
            return ['status' => 'failed', 'message' => $response['message'] ?? $res];
        }

        if (isset($response["data"]["account_number"])) {
            $data = $response["data"];
            $this->save_account(
                $user_id,
                'Providus', // Ncwallet usually returns Providus
                $data["account_number"],
                $data["account_name"],
                $data['account_reference'] ?? '',
                $this->id
            );

            // Backward compatibility removed as per request to avoid vp_updateuser

            return ['status' => 'success', 'message' => 'Account created successfully'];
        }

        return ['status' => 'failed', 'message' => 'No account returned'];
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "public_key" => [
                "label" => "Public key",
                "description" => "",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "transaction_pin" => [
                "label" => "Transaction pin",
                "description" => "",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "banks" => [
                "label" => "Banks",
                "description" => "Banks associated to the gateway id:name separated by comma for multiple banks",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => "2:Wema,4:sterling"
                ]
            ],
        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
}

class Demo_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'demo';
        $this->name = 'Demo';
        $this->description = 'A gateway that generates a random account to your users. Please only use this on test mode';
        $this->status = true;
        $this->countries = ['all'];
        $this->virtual_account = true;
        $this->custom_id = '';
        $this->image = VTUPRESS_URL . "/images/bank/others.jpeg";


        parent::__construct($db);
    }

    public function handle_webhook(): bool
    {
        $input = file_get_contents("php://input");
        // error_log("Demo Gateway Input: " . $input);
        
        $event = json_decode(str_replace(" ", "", $input));
        $array = json_decode(str_replace(" ", "", $input), true);

        // Check if event type is set, legacy check
        if (!isset($event->event_type)) {
            // error_log("Demo Gateway: event_type not set.");
            // Maybe not an ncwallet request or invalid format
            return false;
        }

        if (strtoupper($event->event_type) != "DEMO_COLLECTION") {
            // Not a successful transaction type we care about
            return true; // We handled it (it matches our criteria) but we ignore it
        }

        $headers = $this->get_all_headers_helper();
        $signature = $headers["secret-key"] ?? $headers["secret_key"] ?? null;

        if (!$signature) {
            return true;
        }

        $settings = $this->get_settings();
        $apikey = $settings['secret_key'] ?? '';
        if(empty($apikey)){
            return true;
        }

        if($apikey != $signature){
            return true;
        }

        $email = $array["email"] ?? '';
        $amount = $array["amount"] ?? 0;
        $session_id = $array["session_id"] ?? '';

        if (empty($email) || empty($session_id)) {
            return true;
        }

        // duplicate check
        $existing = $this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_wallet_webhook WHERE referrence = %s AND gateway = %s", $session_id, $this->id));
        if ($existing) {
            echo "Session Id $session_id already exists";
            http_response_code(200);
            return true;
        }

        // Find user by email (Legacy behavior) - But technically we should use account number if available?
        // Legacy used get_user_by("email", $array["email"])
        // Let's stick to email if that's what they send.

        $user = get_user_by("email", $email);

        if (!$user) {
            $this->log_webhook(0, (string) $amount, $session_id, 'failed', 'User not found by email: ' . $email);
            echo "User with email $email not found";
            http_response_code(500);
            return true;
        }

        $wp_user_id = $user->ID;
        $users = new Users();
        $user_id = $users->getUser($wp_user_id,false,true)->id ?? 0;
        
        $charge = $this->calculate_charge((float) $amount);
        $amount_to_credit = (float) $amount - $charge;
        
        $credited = $users->credit($user_id, $amount_to_credit, 'deposit', "Wallet Funding via Demo", ['reference' => $session_id, 'gateway' => 'Demo']);

        if ($credited) {
            $this->log_webhook($user_id, (string) $amount, $session_id, 'success', json_encode($array));
            $this->process_funding_reward($wp_user_id, (float) $amount, $session_id);
            echo "VP Core $user_id credited";
            http_response_code(200);
        } else {
            $this->log_webhook($user_id, (string) $amount, $session_id, 'failed', 'Credit failed ');
            echo "VP Core $user_id not credited";
            http_response_code(500);
        }

        return true;
    }

    public function create_virtual_account(int $user_id): array
    {
        $settings = $this->get_settings();
        $user = $this->get_user_details($user_id);

        if (!$user)
            return ['status' => 'failed', 'message' => 'User not found'];

        $bvn = $user['bvn'];
        $nin = $user['nin'];
        $admin_bvn = ''; // Would need to fetch from option 'ncwallet_admin_bvn' if implemented, but here we stick to dynamic settings.
        // Or if the user meant dynamic population of what was there. 
        // Logic from old file: checks admin_bvn optional. 
        // For now, enforcing user BVN/NIN as per class properties require_bvn/nin.

        $validation_number ='1234'.rand(1111,9999);


        $token = $settings['secret_key'] ?? '';

        $payload = [
            "account_name" => $user['first_name'] . " " . $user['last_name'],
            "bank_code" => "site", // Hardcoded in legacy? Or use dynamic bank? Legacy used "nomba".
            "account_type" => "static",
            "email" => $user['email'],
            "validation_type" => "BVN",
            "validation_number" => $validation_number,
            "phone_number" => $user['phone']
        ];


        $response = [
            'data' => [
                'account_number' => rand(0000000000,9999999999),
                'account_name' => $user['first_name'] . " " . $user['last_name'],
                'account_reference' => uniqid(),
            ]
        ];

        if (isset($response["data"]["account_number"])) {
            $data = $response["data"];
            $this->save_account(
                $user_id,
                'Demo', // Ncwallet usually returns Providus
                $data["account_number"],
                $data["account_name"],
                $data['account_reference'] ?? '',
                $this->id
            );

            // Backward compatibility removed as per request to avoid vp_updateuser

            return ['status' => 'success', 'message' => 'Account created successfully'];

        }

        return ['status' => 'failed', 'message' => 'No account returned'];
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "secret_key" => [
                "label" => "Secret key",
                "description" => "enter any random character to be used for you test api funding",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "banks" => [
                "label" => "Banks",
                "description" => "",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => "00:demo"
                ]
            ],
        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
}

class Monnify_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'monnify';
        $this->name = 'Monnify';
        $this->description = 'A powerful gateway';
        $this->status = true;
        $this->countries = ['nigeria'];
        $this->virtual_account = true;
        $this->onsite_payment = true;
        $this->require_nin = true;
        $this->require_bvn = true;
        $this->image = VTUPRESS_URL . "/images/bank/monnify.jpg";

        parent::__construct($db);

        add_action('wp_ajax_vtupress_verify_monnify', [$this, 'verify_payment_ajax']);
        add_action('wp_ajax_nopriv_vtupress_verify_monnify', [$this, 'verify_payment_ajax']);
    }

    public function get_payment_js(): string
    {
        $user_id = get_current_user_id();
        $user_data = get_userdata($user_id);
        $user_email = $user_data->user_email;
        $user_name = $user_data->display_name;
        $settings = $this->get_settings();
        $api_key = $settings['api_key'] ?? '';
        $contract_code = $settings['contract_code'] ?? '';
        $is_test = stripos($api_key, 'mk_prod') === false; // Simple heuristic

        $country_data = Vtupress_Country_Manager::all();
        $currency = $country_data[$this->current_country]['currency'] ?? 'NGN';
        $min_funding = $settings['minimum_funding'] ?? '0';

        ob_start();
        ?>
        <script src="https://sdk.monnify.com/plugin/monnify.js"></script>
        <script>
            jQuery(document).ready(function ($) {
                window.vtupress_pay_monnify = function (amount, email, charge_val) {
                    var min_funding = parseFloat(<?php echo $min_funding; ?>);
                    if (amount < min_funding) {
                        Swal.fire('Error', 'Minimum funding amount is ' + min_funding, 'error');
                        return;
                    }
                    var total_amount = parseFloat(amount) + parseFloat(charge_val);

                    MonnifySDK.initialize({
                        amount: total_amount,
                        currency: "<?php echo esc_js($currency); ?>",
                        reference: 'vtupress_' + Math.floor((Math.random() * 1000000000) + 1),
                        customerName: "<?php echo esc_js($user_name); ?>",
                        customerEmail: email,
                        apiKey: "<?php echo esc_js($api_key); ?>",
                        contractCode: "<?php echo esc_js($contract_code); ?>",
                        paymentDescription: "Wallet Funding",
                        isTestMode: <?php echo $is_test ? 'true' : 'false'; ?>,
                        onComplete: function (response) {
                            // Verify Transaction
                            Swal.fire({
                                title: 'Verifying Payment...',
                                text: 'Please wait while we confirm your transaction.',
                                icon: 'info',
                                allowOutsideClick: false,
                                showConfirmButton: false,
                                didOpen: () => {
                                    Swal.showLoading();
                                }
                            });

                            $.ajax({
                                url: '<?php echo admin_url('admin-ajax.php'); ?>',
                                type: 'POST',
                                data: {
                                    action: 'vtupress_verify_monnify',
                                    reference: response.paymentReference, // Monnify returns transactionReference or paymentReference
                                    amount: amount,
                                    charge: charge_val
                                },
                                success: function (res) {
                                    if (res.success) {
                                        Swal.fire('Success!', res.data.message || 'Payment successful', 'success').then(() => {
                                            location.reload();
                                        });
                                    } else {
                                        Swal.fire('Failed!', res.data.message || 'Verification failed', 'error');
                                    }
                                },
                                error: function () {
                                    Swal.fire('Error!', 'Validation request failed', 'error');
                                }
                            });
                        },
                        onClose: function (data) {
                            // alert('Payment window closed');
                        }
                    });
                };
            });
        </script>
        <?php
        return ob_get_clean();
    }

    public function verify_payment_ajax()
    {
        if (!is_user_logged_in()) {
            wp_send_json_error(['message' => 'Not logged in']);
        }

        $reference = sanitize_text_field($_POST['reference'] ?? '');
        if (empty($reference)) {
            wp_send_json_error(['message' => 'No reference provided']);
        }

        $settings = $this->get_settings();
        $api_key = $settings['api_key'] ?? '';
        $secret_key = $settings['secret_key'] ?? '';
        $baseurl = stripos($api_key, "prod") === false ? "https://sandbox.monnify.com" : "https://api.monnify.com";

        // 1. Authenticate to get Access Token
        $auth = base64_encode("$api_key:$secret_key");
        $token_response = wp_remote_post($baseurl . '/api/v1/auth/login/', [
            'headers' => [
                'Authorization' => "Basic $auth",
                'Content-Type' => 'application/json'
            ]
        ]);

        if (is_wp_error($token_response)) {
            wp_send_json_error(['message' => 'Auth Error: ' . $token_response->get_error_message()]);
        }

        $token_body = json_decode(wp_remote_retrieve_body($token_response), true);
        if (!isset($token_body['responseBody']['accessToken'])) {
            wp_send_json_error(['message' => 'Auth failed']);
        }
        $access_token = $token_body['responseBody']['accessToken'];

        // 2. Verify Transaction
        // Monnify API: /api/v2/transactions/{transactionReference} or query by payment ref
        // Let's assume we got paymentReference as reference
        // Note: Monnify JS onComplete returns object with `paymentReference` and `transactionReference`.
        // We generally use transactionReference for verification if possible, but paymentRef works too via query API.

        $verify_response = wp_remote_get("$baseurl/api/v2/merchant/transactions/query?paymentReference=" . urlencode($reference), [
            'headers' => [
                'Authorization' => "Bearer $access_token",
                'Content-Type' => 'application/json'
            ]
        ]);

        if (is_wp_error($verify_response)) {
            wp_send_json_error(['message' => 'Verification API Error']);
        }

        $body = wp_remote_retrieve_body($verify_response);
        $data = json_decode($body, true);

        if (!$data || !($data['requestSuccessful'] ?? false)) {
            wp_send_json_error(['message' => 'Transaction lookup failed: ' . ($data['responseMessage'] ?? 'Unknown')]);
        }

        $tx_data = $data['responseBody'];
        if ($tx_data['paymentStatus'] !== 'PAID') {
            wp_send_json_error(['message' => 'Payment Status: ' . $tx_data['paymentStatus']]);
        }

        $user_id = get_vp_current_user_id();
        $wp_user_id = get_current_user_id();
        $amount_paid = $tx_data['amountPaid']; // Monnify is NGN, no kobo division usually for amountPaid? 
        // Wait, Monnify response is usually actual amount (e.g. 500.00).

        // Prevent Double Credit
        global $wpdb;
        $table_name = $wpdb->prefix . 'vtupress_wallet_webhook';
        $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table_name WHERE referrence = %s AND gateway = %s", $reference, $this->id));

        if ($exists) {
            wp_send_json_success(['message' => 'Payment already verified']);
        }

        $users = new Users();

        // Calculate Charge
        $charge = $this->calculate_charge((float) $amount_paid);
        $credit_amount = (float) $amount_paid - $charge;

        $credited = $users->credit($user_id, $credit_amount, 'deposit', "Wallet Funding via Monnify", ['reference' => $reference, 'gateway' => 'Monnify', 'full_amount' => $amount_paid, 'charge' => $charge]);

        if ($credited) {
            $this->log_webhook($user_id, (string) $amount_paid, $reference, 'success', json_encode($data));

            // Process Referral Reward
            $this->process_funding_reward($wp_user_id, (float) $amount_paid, $reference);

            wp_send_json_success(['message' => 'Wallet successfully funded!']);
        } else {
            $this->log_webhook($user_id, (string) $amount_paid, $reference, 'failed', 'Credit failed');
            wp_send_json_error(['message' => 'Failed to credit wallet']);
        }
    }

    public function handle_webhook(): bool
    {
        $input = file_get_contents("php://input");
        $event = json_decode($input);

        if (!isset($event->eventType) || $event->eventType !== 'SUCCESSFUL_TRANSACTION') {
            // Not a successful transaction or not Monnify format
            if (isset($event->eventType))
                return true; // Handled but ignored
            return false;
        }

        $headers = $this->get_all_headers_helper();
        $signature = $headers['monnify-signature'] ?? null;

        if (!$signature) {
            return false;
        }

        $settings = $this->get_settings();
        $secret_key = $settings['secret_key'] ?? '';

        if (empty($secret_key)) {
            $secret_key = get_option('vtupress_monnifysecretkey', '');
        }

        $computedHash = computeSHA512TransactionHash($input, $secret_key); // Using global helper function from legacy index.php if available, OR we implement logic here.
        // Wait, computeSHA512TransactionHash is in index.php, might not be available here unless functions.php is loaded or we copy usage.
        // It's just hash_hmac('sha512', $stringifiedData, $clientSecret);
        // Let's use hash_hmac directly to be safe and self-contained.

        $localComputedHash = hash_hmac('sha512', $input, $secret_key);

        if ($signature !== $localComputedHash) {
            return false;
        }

        $eventData = $event->eventData;
        $amount = $eventData->amountPaid;
        $transactionReference = $eventData->transactionReference;
        $paymentReference = $eventData->paymentReference;
        $email = $eventData->customer->email;
        $productReference = $eventData->product->reference ?? '';

        // Duplicate Check
        if ($this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_transactions WHERE reference = %s", $transactionReference))) {
            http_response_code(200);
            return true;
        }

        // Check webhook table
        $existing = $this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_wallet_webhook WHERE referrence = %s AND gateway = %s", $transactionReference, $this->id));
        if ($existing) {
            http_response_code(200);
            return true;
        }

        $user_id = 0;

        // Try getting user by Account Reference (Product Reference)
        if (!empty($productReference)) {
            $user_account = $this->db->get_row($this->db->prepare(
                "SELECT userId FROM {$this->db->prefix}vtupress_userAccounts WHERE accountRef = %s",
                $productReference
            ));
            if ($user_account) {
                $user_id = intval($user_account->userId);
            }
        }

        // Fallback to Email
        if ($user_id === 0 && !empty($email)) {
            $user = get_user_by('email', $email);
            if ($user) {
                $user_id = $user->ID;
            }
        }

        if ($user_id === 0) {
            $this->log_webhook(0, (string) $amount, $transactionReference, 'failed', 'User not found. Ref: ' . $productReference . ', Email: ' . $email);
            return true;
        }

        $users = new Users();
        $credited = $users->credit($user_id, (float) $amount, 'deposit', "Wallet Funding via Monnify", ['reference' => $transactionReference, 'gateway' => 'Monnify']);

        if ($credited) {
            $this->log_webhook($user_id, (string) $amount, $transactionReference, 'success', json_encode($event));
            http_response_code(200);
        } else {
            $this->log_webhook($user_id, (string) $amount, $transactionReference, 'failed', 'Credit failed');
            http_response_code(500);
        }

        return true;
    }

    public function create_virtual_account(int $user_id): array
    {
        $settings = $this->get_settings();
        $user = $this->get_user_details($user_id);

        if (!$user)
            return ['status' => 'failed', 'message' => 'User not found'];

        $api_key = $settings['api_key'] ?? '';
        $secret_key = $settings['secret_key'] ?? '';
        $contract_code = $settings['contract_code'] ?? '';

        $baseurl = stripos($api_key, "prod") === false ? "https://sandbox.monnify.com" : "https://api.monnify.com";

        // 1. Auth
        $auth = base64_encode("$api_key:$secret_key");
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $baseurl . '/api/v1/auth/login/',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_HTTPHEADER => [
                "Content-Type: application/json",
                "Authorization: Basic $auth"
            ],
        ));

        $res = curl_exec($curl);
        $jsons = json_decode($res);
        curl_close($curl);

        if (!isset($jsons->responseBody->accessToken)) {
            return ['status' => 'failed', 'message' => 'Auth Failed: ' . ($jsons->responseMessage ?? 'Unknown error')];
        }

        $accessToken = $jsons->responseBody->accessToken;

        // 2. Create Account
        // Parse preferred banks
        $banks_str = $settings['banks'] ?? '035:Wema,232:Sterling,50515:Moniepoint';
        $preferred_banks = [];
        $banks_arr = explode(',', $banks_str);

        foreach ($banks_arr as $pair) {
            $parts = explode(':', $pair);
            if (!empty($parts[0])) {
                $preferred_banks[] = trim($parts[0]);
            }
        }
        if (empty($preferred_banks))
            $preferred_banks = ["035", "232", "50515"];

        // Basic KYC Check
        $bvn = $user['bvn'];
        $nin = $user['nin'];

        $data = [
            "accountReference" => uniqid(),
            "accountName" => $user['username'],
            "currencyCode" => "NGN",
            "contractCode" => $contract_code,
            "customerEmail" => $user['email'],
            "customerName" => $user['first_name'] . " " . $user['last_name'],
            "getAllAvailableBanks" => false,
            "preferredBanks" => $preferred_banks
        ];

        if (!empty($bvn) && strlen($bvn) > 10)
            $data["bvn"] = $bvn;
        if (!empty($nin) && strlen($nin) > 10)
            $data["nin"] = $nin;

        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $baseurl . '/api/v2/bank-transfer/reserved-accounts',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => array(
                "Authorization: Bearer $accessToken",
                'Content-Type: application/json'
            ),
        ));

        $respon = curl_exec($curl);
        curl_close($curl);
        $response = json_decode($respon, true);

        if (!isset($response["responseBody"])) {
            return ['status' => 'failed', 'message' => $response['responseMessage'] ?? 'Unknown Error'];
        }

        $accounts = $response["responseBody"]["accounts"];
        $ref = $response["responseBody"]["accountReference"];
        $saved = false;

        foreach ($accounts as $acc) {
            $this->save_account(
                $user_id,
                $acc['bankName'],
                $acc['accountNumber'],
                $acc['accountName'],
                $ref,
                $this->id
            );
            $saved = true;

            $this->save_account(
                $user_id,
                $acc['bankName'],
                $acc['accountNumber'],
                $acc['accountName'],
                $ref,
                $this->id
            );
            $saved = true;

            // Backward Compatibility removed
        }

        if ($saved) {
            return ['status' => 'success', 'message' => 'Monnify accounts generated'];
        }

        return ['status' => 'failed', 'message' => 'No accounts returned'];
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "api_key" => [
                "label" => "Api key",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "secret_key" => [
                "label" => "Secret key",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "contract_code" => [
                "label" => "Contract code",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "banks" => [
                "label" => "Banks",
                "description" => "Banks associated to the gateway id:name separated by comma for multiple banks",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
}

class Paystack_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'paystack';
        $this->name = 'Paystack';
        $this->description = 'A powerful gateway';
        $this->status = true;
        $this->countries = ['nigeria'];
        $this->onsite_payment = true;
        $this->image = VTUPRESS_URL . "/images/bank/paystack.jpg";



        parent::__construct($db);

        add_action('wp_ajax_vtupress_verify_paystack', [$this, 'verify_payment_ajax']);
        add_action('wp_ajax_nopriv_vtupress_verify_paystack', [$this, 'verify_payment_ajax']);
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "public_key" => [
                "label" => "Public key",
                "description" => "",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "secret_key" => [
                "label" => "Secret key",
                "description" => "",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "banks" => [
                "label" => "Banks",
                "description" => "Banks associated to the gateway id:name separated by comma for multiple banks",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],

        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
    public function get_payment_js(): string
    {
        $user_id = get_current_user_id();
        $user_data = get_userdata($user_id);
        $user_email = $user_data->user_email;
        $settings = $this->get_settings();
        $pk = $settings['public_key'] ?? '';

        $country_data = Vtupress_Country_Manager::all();
        $currency = $country_data[$this->current_country]['currency'] ?? 'NGN';
        $min_funding = $settings['minimum_funding'] ?? '0';

        ob_start();
        ?>
        <script src="https://js.paystack.co/v1/inline.js"></script>
        <script>
            jQuery(document).ready(function ($) {
                // Register this gateway's handler
                window.vtupress_pay_paystack = function (amount, email, charge_val) {
                    var min_funding = parseFloat(<?php echo $min_funding; ?>);
                    if (amount < min_funding) {
                        Swal.fire('Error', 'Minimum funding amount is ' + min_funding, 'error');
                        return;
                    }
                    var total_amount = parseFloat(amount) + parseFloat(charge_val);

                    var handler = PaystackPop.setup({
                        key: '<?php echo esc_js($pk); ?>',
                        email: email,
                        amount: total_amount * 100, // Amount in kobo
                        currency: "<?php echo esc_js($currency); ?>",
                        ref: 'vtupress_' + Math.floor((Math.random() * 1000000000) + 1),
                        callback: function (response) {
                            // Verify Transaction
                            Swal.fire({
                                title: 'Verifying Payment...',
                                text: 'Please wait while we confirm your transaction.',
                                icon: 'info',
                                allowOutsideClick: false,
                                showConfirmButton: false,
                                didOpen: () => {
                                    Swal.showLoading();
                                }
                            });

                            $.ajax({
                                url: '<?php echo admin_url('admin-ajax.php'); ?>',
                                type: 'POST',
                                data: {
                                    action: 'vtupress_verify_paystack',
                                    reference: response.reference,
                                    amount: amount, // Pass original amount for logging/crediting logic if needed
                                    charge: charge_val,
                                    // nonce: '...' // TODO: Add nonce
                                },
                                success: function (res) {
                                    if (res.success) {
                                        Swal.fire('Success!', res.data.message || 'Payment successful', 'success').then(() => {
                                            location.reload();
                                        });
                                    } else {
                                        Swal.fire('Failed!', res.data.message || 'Verification failed', 'error');
                                    }
                                },
                                error: function () {
                                    Swal.fire('Error!', 'Validation request failed', 'error');
                                }
                            });
                        },
                        onClose: function () {
                            // alert('Window closed.');
                        }
                    });
                    handler.openIframe();
                };
            });
        </script>
        <?php
        return ob_get_clean();
    }

    public function verify_payment_ajax()
    {
        // Simple security check (should use nonce in production)
        if (!is_user_logged_in()) {
            wp_send_json_error(['message' => 'Not logged in']);
        }

        $reference = sanitize_text_field($_POST['reference'] ?? '');
        if (empty($reference)) {
            wp_send_json_error(['message' => 'No reference provided']);
        }

        $settings = $this->get_settings();
        $sk = $settings['secret_key'] ?? '';

        if (empty($sk)) {
            // Fallback to legacy option if needed
            $sk = get_option('vtupress_paystack_secret_key', '');
        }

        $response = wp_remote_get("https://api.paystack.co/transaction/verify/" . $reference, [
            'headers' => [
                'Authorization' => "Bearer " . $sk,
                'Content-Type' => 'application/json'
            ]
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Verification error: ' . $response->get_error_message()]);
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !($data['status'] ?? false)) {
            wp_send_json_error(['message' => 'Verification failed from Paystack: ' . ($data['message'] ?? 'Unknown error')]);
        }

        if ($data['data']['status'] !== 'success') {
            wp_send_json_error(['message' => 'Transaction was not successful: ' . $data['data']['status']]);
        }

        // Credit User Logic (Simplified)
        // In a real scenario, we should verify amount matches expected, etc.
        // And record transaction ID to prevent double crediting.

        $user_id = get_vp_current_user_id();
        $wp_user_id = get_current_user_id();
        $amount_paid = $data['data']['amount'] / 100; // Kobo to NGN

        // Log webhook/transaction (Prevent Double Credit)
        // Check if reference exists
        global $wpdb;
        $table_name = $wpdb->prefix . 'vtupress_wallet_webhook';
        $exists = $wpdb->get_var($wpdb->prepare("SELECT id FROM $table_name WHERE referrence = %s AND gateway = %s", $reference, $this->id));

        if ($exists) {
            wp_send_json_success(['message' => 'Payment already verified']);
        }

        // Credit User
        // Credit User
        $users = new Users();

        // Calculate Charge
        $charge = $this->calculate_charge((float) $amount_paid);
        $credit_amount = (float) $amount_paid - $charge;

        $credited = $users->credit($user_id, $credit_amount, 'deposit', "Wallet Funding via Paystack", ['reference' => $reference, 'gateway' => 'Paystack', 'full_amount' => $amount_paid, 'charge' => $charge]);

        if ($credited) {
            // Log success
            $this->log_webhook($user_id, (string) $amount_paid, $reference, 'success', json_encode($data));

            // Process Referral Reward
            $this->process_funding_reward($wp_user_id, (float) $amount_paid, $reference);

            wp_send_json_success(['message' => 'Wallet successfully funded!']);
        } else {
            $this->log_webhook($user_id, (string) $amount_paid, $reference, 'failed', 'Credit failed');
            wp_send_json_error(['message' => 'Failed to credit wallet']);
        }
    }

    public function get_banks(): array
    {
        $settings = $this->get_settings();
        $sk = $settings['secret_key'] ?? '';

        $cache_key = 'vtupress_paystack_banks';
        $banks = get_transient($cache_key);

        if ($banks === false) {
            $response = wp_remote_get("https://api.paystack.co/bank", [
                'headers' => [
                    'Authorization' => "Bearer " . $sk,
                    'Content-Type' => 'application/json'
                ]
            ]);

            if (is_wp_error($response)) {
                return [];
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if (!($data['status'] ?? false)) {
                return [];
            }

            $banks = [];
            foreach ($data['data'] as $bank) {
                $banks[$bank['code']] = $bank['name'];
            }

            set_transient($cache_key, $banks, WEEK_IN_SECONDS);
        }

        return $banks;
    }

    public function resolve_account(string $bank_code, string $account_number): array
    {
        $settings = $this->get_settings();
        $sk = $settings['secret_key'] ?? '';

        $response = wp_remote_get("https://api.paystack.co/bank/resolve?account_number=$account_number&bank_code=$bank_code", [
            'headers' => [
                'Authorization' => "Bearer " . $sk,
                'Content-Type' => 'application/json'
            ]
        ]);

        if (is_wp_error($response)) {
            return ['status' => 'failed', 'message' => $response->get_error_message()];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (($data['status'] ?? false) && ($data['data']['account_name'] ?? false)) {
            return [
                'status' => 'success',
                'account_name' => $data['data']['account_name'],
                'account_number' => $data['data']['account_number'],
                'bank_id' => $data['data']['bank_id'] ?? ''
            ];
        }

        return ['status' => 'failed', 'message' => $data['message'] ?? 'Resolution failed'];
    }

    public function transfer_funds(float $amount, string $bank_code, string $account_number, string $account_name, string $ref): array
    {
        $settings = $this->get_settings();
        $sk = $settings['secret_key'] ?? '';

        // 1. Create Recipient
        // Paystack needs "type", "name", "account_number", "bank_code", "currency"
        $recipient_payload = [
            'type' => 'nuban',
            'name' => $account_name,
            'account_number' => $account_number,
            'bank_code' => $bank_code,
            'currency' => 'NGN'
        ];

        $response = wp_remote_post("https://api.paystack.co/transferrecipient", [
            'headers' => [
                'Authorization' => "Bearer " . $sk,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($recipient_payload)
        ]);

        if (is_wp_error($response)) {
            return ['status' => 'failed', 'message' => $response->get_error_message()];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!($data['status'] ?? false)) {
            return ['status' => 'failed', 'message' => $data['message'] ?? 'Recipient creation failed'];
        }

        $recipient_code = $data['data']['recipient_code'];

        // 2. Initiate Transfer
        $transfer_payload = [
            'source' => 'balance',
            'amount' => $amount * 100, // kobo
            'recipient' => $recipient_code,
            'reason' => 'Transfer via VtuPress',
            'reference' => $ref
        ];

        $response = wp_remote_post("https://api.paystack.co/transfer", [
            'headers' => [
                'Authorization' => "Bearer " . $sk,
                'Content-Type' => 'application/json'
            ],
            'body' => json_encode($transfer_payload)
        ]);

        if (is_wp_error($response)) {
            return ['status' => 'failed', 'message' => $response->get_error_message()];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!($data['status'] ?? false)) {
            return ['status' => 'failed', 'raw' => $body, 'message' => $data['message'] ?? 'Transfer failed'];
        }

        // Transfer queued/successful
        return [
            'status' => 'success',
            'message' => $data['message'],
            'raw' => $body,
            'reference' => $data['data']['reference'] ?? $ref
        ];
    }
}

class Paymentpoint_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'paymentpoint';
        $this->name = 'Paymentpoint';
        $this->description = 'A powerful gateway that generates palmpay account number';
        $this->status = true;
        $this->custom_id = 'paymentpoint';
        $this->countries = ['nigeria'];
        $this->virtual_account = true;
        $this->image = VTUPRESS_URL . "/images/bank/paymentpoint.webp";


        parent::__construct($db);
    }

    public function handle_webhook(): bool
    {
        $headers = $this->get_all_headers_helper();
        if (!isset($headers['paymentpoint-signature'])) {
            // Also check HTTP_PAYMENTPOINT_SIGNATURE just in case server env var behavior
            if (isset($_SERVER['HTTP_PAYMENTPOINT_SIGNATURE'])) {
                $headers['paymentpoint-signature'] = $_SERVER['HTTP_PAYMENTPOINT_SIGNATURE'];
            } else {
                return false;
            }
        }

        $input = file_get_contents("php://input");
        $settings = $this->get_settings();
        $secret_key = $settings['secret_key'] ?? '';

        if (empty($secret_key)) {
            $secret_key = get_option('vtupress_paymentpoint_secretkey', '');
        }

        $calculatedSignature = hash_hmac('sha256', $input, $secret_key);

        if (!hash_equals($calculatedSignature, $headers['paymentpoint-signature'])) {
            return false;
        }

        $body = json_decode($input, true);
        if (!$body)
            return false;

        $transactionId = $body['transaction_id'] ?? null;
        $amount = $body['amount_paid'] ?? null;
        $status = $body['transaction_status'] ?? null;
        $recipient_number = $body['receiver']["account_number"] ?? null;

        if (!$transactionId || !$amount || $status !== 'success') {
            return true;
        }

        // Check for duplicates
        if ($this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_transactions WHERE reference = %s", $transactionId))) {
            http_response_code(200);
            return true;
        }


        // Find user by account number
        $user_account = $this->db->get_row($this->db->prepare(
            "SELECT id,userId FROM {$this->db->prefix}vtupress_userAccounts WHERE accountNumber = %s",
            $recipient_number
        ));

        if (!$user_account) {
            $this->log_webhook(0, (string) $amount, $transactionId, 'failed', 'User not found for account: ' . $recipient_number);
            return true;
        }

        $user_id = intval($user_account->id);
        $wp_user_id = intval($user_account->userId);

        $users = new Users();
        // Check legacy uniqueness or just rely on Users::credit logging? Users::credit logs transaction with unique ref but we want to ensure we don't double credit unique gateway ref.
        // Users::credit generates its own ref. We should probably query transaction table by details->reference or description.
        // But for now, let's rely on wallet webhook table for dedup check which I should implement properly.

        // Actually, let's check webhook table for dedup
        $existing = $this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_wallet_webhook WHERE referrence = %s AND gateway = %s", $transactionId, $this->id));
        if ($existing) {
            http_response_code(200);
            return true;
        }

        $charge = $this->calculate_charge((float) $amount);
        $amount_to_credit = (float) $amount - $charge;

        $credited = $users->credit($user_id, $amount_to_credit, 'deposit', "Wallet Funding via Paymentpoint", ['reference' => $transactionId, 'gateway' => 'Paymentpoint']);

        if ($credited) {
            $this->log_webhook($user_id, (string) $amount, $transactionId, 'success', json_encode($body));
            $this->process_funding_reward($wp_user_id , (float) $amount, $transactionId);
            http_response_code(200);
        } else {
            $this->log_webhook($user_id, (string) $amount, $transactionId, 'failed', 'Credit failed');
            http_response_code(500);
        }

        return true;
    }

    public function create_virtual_account(int $user_id): array
    {
        $settings = $this->get_settings();
        $user = $this->get_user_details($user_id);

        if (!$user)
            return ['status' => 'failed', 'message' => 'User not found'];

        $secret_key = $settings['secret_key'] ?? '';
        $api_key = $settings['api_key'] ?? '';
        $business_id = $settings['business_id'] ?? '';

        $bank_codes = [];
        $banks_str = $settings['banks'] ?? '20946:Palmpay';
        $banks_arr = explode(',', $banks_str);

        foreach ($banks_arr as $pair) {
            $parts = explode(':', $pair);
            if (!empty($parts[0])) {
                $bank_codes[] = trim($parts[0]);
            }
        }

        if (empty($bank_codes)) {
            $bank_codes[] = "20946"; // Default fallback
        }

        $payload = [
            "name" => $user['first_name'] . " " . $user['last_name'],
            "email" => $user['email'],
            "bankCode" => $bank_codes,
            "businessId" => $business_id,
            "phoneNumber" => $user['phone']
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.paymentpoint.co/api/v1/createVirtualAccount",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_SLASHES),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $secret_key",
                "api-key: $api_key",
                "Content-Type: application/json"
            ],
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err) {
            return ['status' => 'failed', 'message' => $err];
        }

        $data = json_decode($response, true);

        if (($data['status'] ?? '') !== 'success') {
            return ['status' => 'failed', 'message' => $data['message'] ?? 'Unknown error from Paymentpoint'];
        }

        if (isset($data['bankAccounts']) && is_array($data['bankAccounts'])) {
            foreach ($data['bankAccounts'] as $acc) {
                $this->save_account(
                    $user_id,
                    $acc['bankName'],
                    $acc['accountNumber'],
                    $acc['accountName'],
                    $acc['Reserved_Account_Id'] ?? '',
                    $this->id
                );

                // Backward compatibility removed
            }
            return ['status' => 'success', 'message' => 'Account generated successfully'];
        }

        return ['status' => 'failed', 'message' => 'No accounts returned in response'];
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "business_id" => [
                "label" => "Business Id",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "api_key" => [
                "label" => "Api key",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "secret_key" => [
                "label" => "Secret key",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "banks" => [
                "label" => "Banks",
                "description" => "Banks associated to the gateway id:name separated by comma for multiple banks",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
}

class Nomba_Gateway extends Vtupress_Gateway
{

    public function __construct(wpdb $db)
    {
        $this->id = 'nomba';
        $this->name = 'Nomba';
        $this->custom_id = 'nomba';
        $this->description = 'A powerful gateway that generates nomba account number';
        $this->status = true;
        $this->countries = ['nigeria'];
        $this->virtual_account = true;
        $this->image = VTUPRESS_URL . "/images/bank/nomba.webp";


        parent::__construct($db);
    }

    public function handle_webhook(): bool
    {
        // Nomba sends a POST request with headers
        $headers = $this->get_all_headers_helper();
        // Check for specific header not always consistent, sometimes HTTP_NOMBA_SIGNATURE
        // Legacy code checked: array_key_exists('http_nomba_signature', $_SERVER)

        $signature_header = $headers['nomba-signature'] ?? $headers['http-nomba-signature'] ?? null;

        if (!$signature_header && isset($_SERVER['HTTP_NOMBA_SIGNATURE'])) {
            $signature_header = $_SERVER['HTTP_NOMBA_SIGNATURE'];
        }

        if (!$signature_header && (!isset($_SERVER['HTTP_NOMBA_SIG_VALUE']) || !isset($_SERVER['HTTP_NOMBA_TIMESTAMP']))) {
            return false;
        }

        $input = file_get_contents("php://input");
        $payload = json_decode($input, true);
        if (!$payload)
            return false;

        $settings = $this->get_settings();
        // Legacy used 'nomba_sign' option
        $signatureKey = get_option('vtupress_nomba_sign', '');
        if (empty($signatureKey)) {
            $signatureKey = $settings['secret_key'] ?? '';
        }

        $timestamp = $_SERVER['HTTP_NOMBA_TIMESTAMP'];
        $signature = $_SERVER['HTTP_NOMBA_SIG_VALUE'];

        $hashingPayload = ($payload['event_type'] ?? '') . ':' .
            ($payload['requestId'] ?? '') . ':' .
            ($payload['data']['merchant']['userId'] ?? '') . ':' .
            ($payload['data']['merchant']['walletId'] ?? '') . ':' .
            ($payload['data']['transaction']['transactionId'] ?? '') . ':' .
            ($payload['data']['transaction']['type'] ?? '') . ':' .
            ($payload['data']['transaction']['time'] ?? '') . ':' .
            ($payload['data']['transaction']['responseCode'] ?? '');

        $message = $hashingPayload . ':' . $timestamp;

        $calculatedSignature = hash_hmac('sha256', $message, $signatureKey, true);
        $encoded_data = base64_encode($calculatedSignature);

        if (!hash_equals($encoded_data, $signature)) {
            // Invalid signature
            return false;
        }

        $webhookData = $payload['data']['transaction'];
        $transactionId = $webhookData['transactionId'] ?? null;
        $amount = $webhookData['transactionAmount'] ?? null;
        $status = $payload["event_type"] ?? null;
        $recipient_number = $webhookData["aliasAccountNumber"] ?? null;

        if (!$transactionId || !$amount || $status !== 'payment_success') {
            return true;
        }

        // Check DB for duplicate
        if ($this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_transactions WHERE reference = %s", $transactionId))) {
            http_response_code(200);
            return true;
        }

        // Find user
        $user_account = $this->db->get_row($this->db->prepare(
            "SELECT id,userId FROM {$this->db->prefix}vtupress_userAccounts WHERE accountNumber = %s",
            $recipient_number
        ));

        if (!$user_account) {
            $this->log_webhook(0, (string) $amount, $transactionId, 'failed', 'User not found: ' . $recipient_number);
            return true;
        }

        $user_id = intval($user_account->id);
        $wp_user_id = intval($user_account->userId);

        // Check webhook table
        $existing = $this->db->get_var($this->db->prepare("SELECT id FROM {$this->db->prefix}vtupress_wallet_webhook WHERE referrence = %s AND gateway = %s", $transactionId, $this->id));
        if ($existing) {
            http_response_code(200);
            return true;
        }

        $users = new Users();
        
        $charge = $this->calculate_charge((float) $amount);
        $amount_to_credit = (float) $amount - $charge;

        $credited = $users->credit($user_id, $amount_to_credit, 'deposit', "Wallet Funding via Nomba", ['reference' => $transactionId, 'gateway' => 'Nomba']);

        if ($credited) {
            $this->log_webhook($user_id, (string) $amount, $transactionId, 'success', json_encode($payload));
            $this->process_funding_reward($wp_user_id, (float) $amount, $transactionId);
            http_response_code(200);
        } else {
            $this->log_webhook($user_id, (string) $amount, $transactionId, 'failed', 'Credit failed');
            http_response_code(500);
        }

        return true;
    }

    public function create_virtual_account(int $user_id): array
    {
        $settings = $this->get_settings();
        $user = $this->get_user_details($user_id);

        if (!$user)
            return ['status' => 'failed', 'message' => 'User not found'];

        $secret_key = $settings['secret_key'] ?? ''; // nomba_secretkey
        $client_id = $settings['business_id'] ?? ''; // nomba_businessid
        $account_id = $settings['api_key'] ?? ''; // nomba_apikey

        // 1. Get Access Token
        $payload_auth = [
            "grant_type" => "client_credentials",
            "client_id" => $client_id,
            "client_secret" => $secret_key
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.nomba.com/v1/auth/token/issue",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload_auth, JSON_UNESCAPED_SLASHES),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $secret_key",
                "accountId: $account_id",
                "Content-Type: application/json"
            ],
        ]);

        $res = curl_exec($curl);
        $response = json_decode($res, true);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err)
            return ['status' => 'failed', 'message' => $err];

        if (!isset($response["data"]["access_token"])) {
            return ['status' => 'failed', 'message' => 'Auth Failed: ' . ($response['message'] ?? $res)];
        }

        $token = $response["data"]["access_token"];

        // 2. Create Account
        $payload = [
            "accountName" => $user['first_name'] . " " . $user['last_name'],
            "accountRef" => uniqid('vtu-', false),
            "currency" => "NGN",
            "expiryDate" => date("Y-m-d H:i:s", strtotime("+5 years"))
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.nomba.com/v1/accounts/virtual",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_SLASHES),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $token",
                "accountId: $account_id",
                "Content-Type: application/json"
            ],
        ]);

        $res = curl_exec($curl);
        $response = json_decode($res, true);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err)
            return ['status' => 'failed', 'message' => $err];

        if (($response["description"] ?? '') !== "success") {
            return ['status' => 'failed', 'message' => $response['message'] ?? $res];
        }

        if (isset($response["data"]["bankAccountNumber"])) {
            $account = $response["data"];

            $this->save_account(
                $user_id,
                $account["bankName"] ?? 'Nomba',
                $account["bankAccountNumber"],
                $account["bankAccountName"],
                $payload['accountRef'],
                $this->id
            );

            return ['status' => 'success', 'message' => 'Nomba account generated'];
        }

        return ['status' => 'failed', 'message' => 'No account data returned'];
    }

    public function process_payment(float $amount, array $details): array
    {
        return [
            'status' => 'success',
            'transaction_id' => uniqid('demo_'),
            'message' => 'Payment processed via Demo Gateway'
        ];
    }

    public function form_fields(): array
    {
        $fields = [
            "business_id" => [
                "label" => "Business Id",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "api_key" => [
                "label" => "Api key",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "secret_key" => [
                "label" => "Secret key",
                "description" => "",
                "column" => 4,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
            "banks" => [
                "label" => "Banks",
                "description" => "Banks associated to the gateway id:name separated by comma for multiple banks",
                "column" => 6,
                "render" => [
                    "class" => "border border-success",
                    "value" => ""
                ]
            ],
        ];

        return $fields;
    }

    public function render_admin_form(): void
    {
        $this->formData();
    }
    public function get_banks(): array
    {
        $settings = $this->get_settings();
        $client_id = $settings['business_id'] ?? '';
        $secret_key = $settings['secret_key'] ?? '';
        $account_id = $settings['api_key'] ?? ''; // nomba_apikey

        $cache_key = 'vtupress_nomba_banks';
        $banks = get_transient($cache_key);

        if ($banks === false) {
            // 1. Auth (Simplified duplication for now)
            $payload_auth = [
                "grant_type" => "client_credentials",
                "client_id" => $client_id,
                "client_secret" => $secret_key
            ];

            $curl = curl_init();
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://api.nomba.com/v1/auth/token/issue",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_CUSTOMREQUEST => "POST",
                CURLOPT_POSTFIELDS => json_encode($payload_auth),
                CURLOPT_HTTPHEADER => [
                    "Authorization: Bearer $secret_key",
                    "accountId: $account_id",
                    "Content-Type: application/json"
                ],
            ]);
            $res = curl_exec($curl);
            $response = json_decode($res, true);
            curl_close($curl);

            $token = $response["data"]["access_token"] ?? null;

            if (!$token) {
                return [];
            }

            // 2. Fetch Banks
            $curl = curl_init();
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://api.nomba.com/v1/banks",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_CUSTOMREQUEST => "GET",
                CURLOPT_HTTPHEADER => [
                    "Authorization: Bearer $token",
                    "accountId: $account_id"
                ],
            ]);
            $res = curl_exec($curl);
            $data = json_decode($res, true);
            curl_close($curl);

            if (($data['description'] ?? '') !== 'success') {
                return [];
            }

            $banks = [];
            foreach ($data['data'] as $bank) {
                // Nomba Bank obj: {name, code, ...}
                $banks[$bank['code']] = $bank['name'];
            }

            set_transient($cache_key, $banks, WEEK_IN_SECONDS);
        }

        return $banks;
    }

    public function resolve_account(string $bank_code, string $account_number): array
    {
        $settings = $this->get_settings();
        $account_id = $settings['api_key'] ?? '';
        $secret_key = $settings['secret_key'] ?? '';

        // TODO: Reuse Auth Token logic properly
        $client_id = $settings['business_id'] ?? '';

        $payload_auth = [
            "grant_type" => "client_credentials",
            "client_id" => $client_id,
            "client_secret" => $secret_key
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.nomba.com/v1/auth/token/issue",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload_auth),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $secret_key",
                "accountId: $account_id",
                "Content-Type: application/json"
            ],
        ]);
        $res = curl_exec($curl);
        $response = json_decode($res, true);
        curl_close($curl);
        $token = $response["data"]["access_token"] ?? null;

        if (!$token) {
            return ['status' => 'failed', 'message' => 'Auth Failed'];
        }

        // Verify Account Name
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.nomba.com/v1/transfers/bank/lookup",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode([
                "accountNumber" => $account_number,
                "bankCode" => $bank_code
            ]),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $token",
                "accountId: $account_id",
                "Content-Type: application/json"
            ],
        ]);
        $res = curl_exec($curl);
        $data = json_decode($res, true);
        curl_close($curl);

        if (($data['code'] ?? '') === '00' && ($data['data']['accountName'] ?? false)) {
            return [
                'status' => 'success',
                'account_name' => $data['data']['accountName'],
                'account_number' => $account_number
            ];
        }

        return ['status' => 'failed', 'message' => $data['description'] ?? 'Account lookup failed'];
    }

    public function transfer_funds(float $amount, string $bank_code, string $account_number, string $account_name, string $ref): array
    {
        $settings = $this->get_settings();
        $account_id = $settings['api_key'] ?? '';
        $secret_key = $settings['secret_key'] ?? '';
        $client_id = $settings['business_id'] ?? '';

        // Auth
        $payload_auth = [
            "grant_type" => "client_credentials",
            "client_id" => $client_id,
            "client_secret" => $secret_key
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.nomba.com/v1/auth/token/issue",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload_auth),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $secret_key",
                "accountId: $account_id",
                "Content-Type: application/json"
            ],
        ]);
        $res = curl_exec($curl);
        $response = json_decode($res, true);
        curl_close($curl);
        $token = $response["data"]["access_token"] ?? null;

        if (!$token) {
            return ['status' => 'failed', 'message' => 'Auth Failed'];
        }

        // Transfer
        $transfer_payload = [
            "amount" => $amount,
            "accountNumber" => $account_number,
            "accountName" => $account_name,
            "bankCode" => $bank_code,
            "merchantReference" => $ref,
            "narration" => "Transfer via VtuPress",
            "currency" => "NGN"
        ];

        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.nomba.com/v1/transfers/bank",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($transfer_payload),
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer $token",
                "accountId: $account_id",
                "Content-Type: application/json"
            ],
        ]);
        $res = curl_exec($curl);
        $data = json_decode($res, true);
        curl_close($curl);

        // Check success code (00)
        if (($data['code'] ?? '') === '00') {
            return [
                'status' => 'success',
                'raw' => $res,
                'message' => $data['description'] ?? 'Successful',
                'reference' => $ref
            ];
        }

        return ['status' => 'failed', 'raw' => $res, 'message' => $data['description'] ?? 'Transfer failed'];
    }
}


function vtupress_gateway_registry(): array
{
    $registry = [
        'manual' => Manual_Gateway::class,
        'demo' => Demo_Gateway::class,
        'paymentpoint' => Paymentpoint_Gateway::class,
        'nomba' => Nomba_Gateway::class,
        'ncwallet' => Ncwallet_Gateway::class,
        'monnify' => Monnify_Gateway::class,
        // 'kuda'         => Kuda_Gateway::class,
        // 'vpay'         => Vpay_Gateway::class,
        // 'palmpay'      => Palmpay_Gateway::class,
        // 'opay'         => Opay_Gateway::class,
        'paystack' => Paystack_Gateway::class,
        // 'flutterwave'  => Flutterwave_Gateway::class
    ];

    return apply_filters('vtupress_gateway_registry', $registry);
}

