<?php
if (!defined('ABSPATH'))
    exit;

abstract class Vtupress_Process_Transactions
{

    public $user_country;
    public string $id;
    public array $providers = [];
    public array $plans = [];
    public array $payload = [];
    public bool $is_voucher = false;
    abstract public function get_settings(): array;

    public function start_transaction(array $user, array $payload = [])
    {
        $country = $this->user_country = $user["set_country"];

        global $wpdb;
        $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
        $ref = $payload['ref'] ?? '';
        $user_id = $user['id'];

        if (empty($ref)) {
            return ['status' => 'error', 'message' => 'No reference key'];
        }

        // 1. Check if Reference already exists (Idempotency)
        $existing = $wpdb->get_var($wpdb->prepare("SELECT id FROM {$wpdb->prefix}vtupress_transactions WHERE reference = %s", $ref));
        if ($existing) {
            return ['status' => 'error', 'message' => 'Duplicate Transaction detected (Ref)'];
        }

        // 2. Lock User (Concurrency)
        $service_id = $this->id ?? 'unknown';
        $inserted = $wpdb->query($wpdb->prepare("INSERT IGNORE INTO $vend_lock (vend, user_id, created_at) VALUES (%s, %d, NOW())", $service_id, $user_id));

        if ($inserted !== 1) {
            // Should check if it's the SAME or DIFFERENT ref?
            // If unique is on USER_ID, any insert fails if user has active transaction.
            if (Vtupress_Option::get("vtupress_custom_security") == "yes" && Vtupress_Option::get("vtupress_security_mode") == "wild") {
                //  $wpdb->query("ROLLBACK");
                return ['status' => 'error', 'message' => 'Another transaction is currently in progress. Please wait.'];
            }
            // $wpdb->query("ROLLBACK");
            return ['status' => 'error', 'message' => 'Another transaction is currently in progress. Please wait.'];
        }


        $wpdb->query("START TRANSACTION");


        $amount = $payload['amount'] ?? 0;

        if ($amount < 0) {
            if (!empty($ref)) {
                $wpdb->query("ROLLBACK");
                $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            }
            return ['status' => 'error', 'message' => 'Negative amount not allowed'];
        }

        // KYC Check
        if (function_exists('vtupress_kyc_check')) {
            $kyc = vtupress_kyc_check($user_id, $amount);
            if (!$kyc['status']) {
                if (!empty($ref)) {
                    $wpdb->query("ROLLBACK");
                    $wpdb->delete($vend_lock, ['user_id' => $user_id]);
                }
                return ['status' => 'error', 'kyc_error' => true, 'message' => $kyc['message']];
            }

        }

        $balance = $user['balance'] ?? 0;
        if ((float) $balance < (float) $amount) {
            if (!empty($ref)) {
                $wpdb->query("ROLLBACK");
                $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            }
            return ['status' => 'error', 'message' => 'Insufficient Balance'];
        }

        return ['status' => 'success'];
    }

    public function vtupress_internal_pin_api($service_id, $plan_id, $amount, $quantity = 1, $type = '')
    {
        //service as service_id like recharge_card
        //provider as plan_id like mtn such as plan / plan_id
        //amount as value/denomination
        //recipient as quantity
        //type as service type


        if (empty($service_id) || empty($plan_id)) {
            return json_encode(['status' => 'fail', 'message' => 'Missing service or plan ID']);
        }

        global $wpdb;
        $table = $wpdb->prefix . 'vtupress_internal_pins';

        // 2. Find Unused PINs
        $wpdb->query("START TRANSACTION");

        $query = "SELECT * FROM $table WHERE service_id = %s AND plan_id = %s AND status = 'unused'";
        $args = [$service_id, $plan_id];

        if (!empty($amount)) {
            $query .= " AND value = %s";
            $args[] = $amount;
        }

        if (!empty($type)) {
            $query .= " AND type = %s";
            $args[] = $type;
        }

        $query .= " ORDER BY id ASC LIMIT %d FOR UPDATE";
        $args[] = $quantity;

        $sql = $wpdb->prepare($query, ...$args);
        // error_log($sql);
        $pins = $wpdb->get_results($sql);

        if (count($pins) < $quantity) {
            $wpdb->query("ROLLBACK");
            return json_encode(['status' => 'fail', 'message' => 'Out of stock (Requested: ' . $quantity . ', Found: ' . count($pins) . ')']);
        }

        // 3. Mark as Used
        $pin_ids = array_map(function ($p) {
            return $p->id;
        }, $pins);
        $ids_string = implode(',', $pin_ids);

        // Use direct query for update to handle IN clause properly safely (ids are int)
        $updated = $wpdb->query("UPDATE $table SET status = 'used', used_at = NOW(), used_by = " . get_vp_current_user_id() . " WHERE id IN ($ids_string)");

        if ($updated === false) {
            $wpdb->query("ROLLBACK");
            return json_encode(['status' => 'fail', 'message' => 'Database error']);
        }

        $wpdb->query("COMMIT");

        // 4. Return Success
        $pin_strings = [];
        $serial_strings = [];
        $values_strings = [];

        foreach ($pins as $p) {
            $pin_strings[] = $p->pin;
            if (!empty($p->serial))
                $serial_strings[] = $p->serial;
            if (!empty($p->value) && !in_array($p->value, $values_strings))
                $values_strings[] = $p->value;
        }

        return json_encode([
            'status' => 'success',
            'pin' => implode(',', $pin_strings),
            'serial' => implode(',', $serial_strings),
            'value' => implode(',', $values_strings)
        ]);
    }


    public function execute($settings, $user_instance, $user_data, $source, array $payload): array
    {

        $start = $this->start_transaction($user_data, $payload);

        if ($start["status"] == "error") {
            // error_log(json_encode($start));
            return $start;
        }

        $settings = $this->get_settings();

        if (!isset($payload['ref'])) {
            return [
                'status' => 'error',
                'message' => "No reference key"
            ];
        }


        // if(!$this->is_voucher):

        /* -------------------------------
         * BASIC CONFIG
         * ------------------------------- */
        $baseUrl = rtrim($settings['base_url'] ?? '', '/');
        $endpoint = ltrim($settings['end_point'] ?? '', '/');
        $url = $baseUrl . '/' . $endpoint;

        $method = strtoupper($settings['request_method'] ?? 'POST');
        $resType = $settings['response_method'] ?? 'json';


        /* -------------------------------
         * BUILD HEADERS
         * ------------------------------- */
        $headers = [];

        // Authentication
        if (!empty($settings['authentication_method'])) {
            if ($settings['authentication_method'] === 'bearer_token') {
                $headers['Authorization'] = trim($settings['authentication_key']) . ' ' . trim($settings['authentication_value']);
            }
            if ($settings['authentication_method'] === 'basic_authentication') {
                $headers['Authorization'] = 'Basic ' . base64_encode(
                    trim($settings['authentication_key']) . ':' . trim($settings['authentication_value'])
                );
            }
        }

        // Custom headers (header_1 ... header_5)
        foreach ($settings as $k => $v) {
            if (strpos($k, 'header_') === 0 && !empty($v) && str_contains($v, ':')) {
                [$hk, $hv] = array_map('trim', explode(':', $v, 2));
                $headers[$hk] = $hv;
            }
        }

        /* -------------------------------
         * BUILD PAYLOAD
         * ------------------------------- */
        $body = [];
                // Required service payload (fallback)
        foreach ($this->payload as $key) {
            if (!isset($body[$key]) && isset($payload[$key])) {
                $body[$key] = $payload[$key];
            }
        }

        // error_log("FIRST FULL BODY ".json_encode($body));

        // Map payload according to the settings keys
        // Map payload according to the settings keys
        $mappingKeys = [
            'recipient_key',
            'amount_key',
            'provider_key',
            'plan_key'
        ];

        $error = false;
        foreach ($mappingKeys as $key) {
            if (!empty($settings[$key])) {
                $payloadKey = $settings[$key];
                //remove _key
                $pk = str_replace('_key', '', $key);
                $value = strtolower($payload[$pk] ?? '') ?? 0;
                //check provider id from provider value
                if ($key == 'provider_key' && $value != 0) {
                    // Check if providers list is available or generic check
                    // Ideally we should use $this->providers or $this->country_providers if available
                    // Assuming $this->providers is available in the child class context
                    if (!empty($this->providers) && !in_array(strtolower($value), array_map('strtolower', $this->providers))) {
                        $error = "Invalid provider [$value] ";
                    } elseif (($settings["{$value}_status"] ?? 'off') != "on") {
                        //check if provider is active
                        $error = "Provider [$value] not active";
                    }

                    if (isset($settings["{$value}_providerId"])) {
                        $value = $settings["{$value}_providerId"]; // Keep providerId setting key for backward compat or rename? 
                        // User said "network_key instead of provider_key. Please fix all". 
                        // So I should probably look for {$value}_providerId if possible, or assume it's still providerId in DB?
                        // "Please fix all" implies internal Logic too. But DB column names might be hard to change here.
                        // I will assume the SETTING key might still be providerId unless I see evidence otherwise.
                        // But I will try to use providerId if it exists? 
                        // Actually, let's keep it safe: The setting is likely stored as {$value}_providerId in DB.
                        // I won't change the DB key retrieval unless requested, to avoid breaking existing settings.
                        // But I will change the Variable name.
                    }

                } elseif ($key === 'plan_key' && ($this->uses_provider ?? false)) {
                    [$nt, $st, $pn] = array_map('trim', explode('_', $value, 3));

                    $provider = strtolower($payload['provider'] ?? '');

                    if ($nt !== $provider) {
                        $error = "Provider [$nt] and plan provider [$provider] mismatch!";
                        return [
                            'status' => 'error',
                            'message' => $error,
                            // 'plan' => $this->plans
                        ];
                    }

                    $planFound = false;

                    foreach ($this->plans as $plan) {

                        // provider + service type must match
                        // Assuming plan structure uses 'network' key or 'provider'? 
                        // The plan structure comes from DB or array. It likely still has 'network' key if not migrated.
                        // But user said "fix all".
                        // Use $plan['network'] ?? $plan['provider'] ?
                        $planProvider = $plan['network'] ?? $plan['provider'] ?? '';

                        if (
                            strtolower($planProvider) !== $nt ||
                            $plan['service_type'] !== $st
                        ) {
                            continue;
                        }

                        // full ID match (preferred)
                        if ($plan['id'] === $value) {
                            $value = $plan['plan_id'];
                            $planFound = true;
                            // Validate Amount matches Plans Price
                            if (floatval($payload['amount']) != floatval($plan['price'])) {
                                $error = "Amount Mismatch: Expected {$plan['price']} but got {$payload['amount']}";
                            }
                            break;
                        }

                        // fallback: plan name match (slug-safe)
                        if (
                            !empty($plan['plan_name']) &&
                            strtolower(str_replace(' ', '_', $plan['plan_name'])) === $pn
                        ) {
                            $value = $plan['plan_id'];
                            $planFound = true;
                            // Validate Amount matches Plans Price
                            if (floatval($payload['amount']) != floatval($plan['price'])) {
                                $error = "Amount Mismatch: Expected {$plan['price']} but got {$payload['amount']}";
                            }
                            break;
                        }
                    }

                    if (!$planFound) {
                        $error = "Invalid or unavailable plan [$value]";
                        break;
                    }
                }

                if(isset($body[$pk])){
                    unset($body[$pk]);
                }

                $body[$payloadKey] = $value;


            }
        }

        if ($error) {
            global $wpdb;
            $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
            $ref = $payload['ref'] ?? '';
            $user_id = $user_data['id']; // Access ID from user_data
            if (!empty($ref)) {
                $wpdb->query("ROLLBACK");
                $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            }
            return [
                'status' => 'error',
                'message' => $error,
                // 'body' => $body
            ];
        }



        // Append custom post datas (post_data_1 ... post_data_5)
        // Custom post datas (post_data_1 ... post_data_5)
        foreach ($settings as $k => $v) {
            if (strpos($k, 'post_data_') === 0 && !empty($v) && str_contains($v, ':')) {
                [$dk, $dv] = array_map('trim', explode(':', $v, 2));
                // Override with payload if exists
                if (isset($payload[$k])) {
                    $body[$dk] = $payload[$k]; // use the payload if e.g post_data_1 exists
                } else {
                    $body[$dk] = $dv; // fallback to the :value
                }
            }
        }

        if (isset($settings['request_id_attr'])):
            $body[$settings['request_id_attr']] = $payload['ref'];
        endif;

        // return [
        //     'status' => 'error',
        //     'message' => print_r($body,true)
        // ];

        /* -------------------------------
         * REQUEST ARGS
         * ------------------------------- */

        $args = [
            'method' => $method,
            'headers' => $headers,
            'timeout' => 120,
        ];

        if ($method !== 'GET') {
            $args['body'] = ($resType === 'json') ? wp_json_encode($body) : $body;
        }else{
            $base_url = rtrim($url, '?');
            $url = add_query_arg($body,$base_url);
            // $user_id = $user_data['id']; // Access ID from user_data
            // global $wpdb;
            // $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
            // $wpdb->query("ROLLBACK");
            //     $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            // return [
            //     'status' => 'error',
            //     'message' => $url
            // ];
        }

        if ($resType === 'json') {
            $args['headers']['Content-Type'] = 'application/json';
        }

        /* -------------------------------
         * SEND REQUEST
         * ------------------------------- */
        if (!$this->is_voucher):
            $response = wp_remote_request($url, $args);
        else:
            $response = $this->vtupress_internal_pin_api($payload["action"], $payload["provider"], $payload["plan"], $payload["quantity"], $payload["type"] ?? '');
            $resType = 'json';
        endif;

        if (is_wp_error($response)) {
            global $wpdb;
            $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
            $ref = $payload['ref'] ?? '';
            $user_id = $user_data['id']; // Access ID from user_data
            if (!empty($ref)) {
                $wpdb->query("ROLLBACK");
                $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            }
            return [
                'status' => 'error',
                'message' => $response->get_error_message()
            ];
        }
        $rawBody = $this->is_voucher ? $response : wp_remote_retrieve_body($response);
        $parsed = ($resType === 'json') ? json_decode($rawBody, true) : $rawBody;

        global $wpdb;
        $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
        $user_id = $user_data['id'];

        if ($resType === 'json' && !is_array($parsed)) {
            // Should duplicate handling be here? If response is bad, we probably want to fail the transaction and release lock/rollback
            // But if the API call was made, maybe we shouldn't rollback blindly?
            // Usually if API call succeeds (http 200) but returns garbage, it's ambiguous.
            // But if we rollback, user gets money back.
            // If API performed transaction, we lose money.
            // Safe approach: Don't rollback immediately if ambiguous, but here we assume invalid JSON = fail.
            // However, the user asked for locking to prevent race condition.
            // The lock prevents double spending.
            // If we fail here, we should probably rollback?
            // Legacy vend.php didn't seem to have post-request rollback (except for checking status).


            $ref = $payload['ref'] ?? '';
            if (!empty($ref)) {
                $wpdb->query("ROLLBACK");
                $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            }

            return [
                'status' => 'error',
                'message' => 'Invalid JSON response ',
                'raw' => $rawBody
            ];
        }

        /* -------------------------------
         * SUCCESS VALIDATION
         * ------------------------------- */
        $success = false;
        $responseKeys = $settings['response_keys'] ?? '';

        if ($this->is_voucher && $parsed["status"] === "success") {
            $success = true;
        } elseif (!empty($responseKeys)) {
            $checks = array_map('trim', explode(',', $responseKeys));

            foreach ($checks as $check) {
                if (!str_contains($check, ':'))
                    continue;

                [$k, $v] = array_map('trim', explode(':', $check, 2));
                // error_log($k . " - " . $v);

                $val = $this->catch_key($parsed, $k);
                if ($val !== false && (string) $val === (string) $v) {
                    $success = true;
                    break;
                }
            }
        }

        $ref = $payload['ref'] ?? '';
        // error_log(json_encode($payload));
        // error_log("ref ".$ref);

        if (!$success) {
            // error_log("Not successful");

            // If failed, rollback? 
            // In legacy, if status != success, we might refund.
            // Here, we just return failed status.
            // The Caller (e.g. process payment loop) should handle the refund/status.
            // But we have a lock open.
            // If we return 'failed', the caller should likely rollback.
            // But we opened the lock here. We should probably manage it here or rely on the caller?
            // The legacy code in vend.php *dies* with rollback if validation fails.
            // But here we are returning array.

            // If I look at the legacy code, it rolls back if inserted !== 1.
            // But after API call, it updates balance.

            // IMPORTANT: If we started a transaction with `START TRANSACTION`, we MUST Commit or Rollback.
            // If 'success' == true, we should probably COMMIT?
            // Warning: `wp_remote_request` is slow. Holding a DB transaction open during an HTTP request is bad practice (locks DB connections).
            // BUT the user ASKED for it: "process_transactions.php ... use in the procss_transaction.php".
            // And legacy did it (sort of, legacy did insertion in `vtupress_vend_lock` and `START TRANSACTION` at the top of vend.php).
            // Actually legacy:
            // 1. START TRANSACTION
            // 2. INSERT into lock
            // ... checks ...
            // 3. update_balance (deduct)
            // 4. process_transaction -> API call

            // If we hold the transaction open during API call, it's risky.
            // But assuming low volume or necessary constraint.

            global $wpdb;
            if (!empty($ref)) {
                // error_log("there's ref");

                // We don't delete lock if it's just failed? We might want to keep the record that it was attempted?
                // But we MUST commit or rollback the DB transaction.

                // If success is false, it means the API returned "failed".
                // In this case, we should probably Rollback the deduction (if we did it? We didn't do deduction here, we just checked balance).
                // Ah, deduction happens OUTSIDE this function normally? 
                // Wait, `process_transactions.php` DOES NOT DEDUCT.

                // If deduction happens outside, `START TRANSACTION` here isolates this check.
                // But `START TRANSACTION` is connection-wide.

                // If the caller handles the deduction, they should be inside the transaction too?
                // But I initiated it here.

                // If I return here, the transaction is still open.

                // Recommendation: Commit here if we are done with "our" part (locking)?
                // No, "vend_lock" is for idempotency.
                // "START TRANSACTION" is for atomicity.

                // If I want to prevent race conditions on BALANCE, the Debit must happen inside the transaction.
                // But this class doesn't debit.

                // The user says: "start-transaction is meant to make sure the amount, plan and so on is acute ... at the backend start_transaction, it must make sure hacker did not alter the amount".

                // If I can't debit here to a Balance, then the "Insufficient Balance" check I added in `start_transaction` is a "Check" not a "Reserve".
                // Race condition:
                // A: Check Balance (100) -> OK
                // B: Check Balance (100) -> OK
                // A: Debit 100
                // B: Debit 100 -> -100 (if no check on debit)

                // The locking table `vtupress_vend_lock` (UNIQUE vend) prevents A and B from processing the *same* request ref.
                // But it doesn't prevent A and B from processing *different* requests trying to spend the same money.

                // To prevent double spending with *different* refs, we need `SELECT ... FOR UPDATE` on the user row or similar.
                // Legacy used `START TRANSACTION` and `update_balance` (which does `UPDATE ... SET balance = balance - amount WHERE ... AND balance >= amount`).

                // Since I cannot change the debit logic (it's not in this file), I can only implement the `vend_lock` (Duplicate Request Protection) and Validation.
                // The user asked for `vtupress_vend_lock` "for locking to prevent race condition".
                // This usually refers to "Double processing the SAME request".


                // So I will stick to what I have:
                // 1. Lock on Ref.
                // 2. Validate Amount/Balance.
                // 3. Close the DB transaction?

                // If I `START TRANSACTION`, I should `COMMIT` if successful locking.
                // So that other connections see the lock?
                // No, `INSERT` is visible after commit.
                // So I should `COMMIT` immediately after `INSERT` if I just want to persis the lock?
                // But if I want to roll it back if validation fails, I keep it open.

                // Proposed flow:
                // 1. START TRANSACTION
                // 2. INSERT into LOCK
                // 3. Validate Amount/Balance. 
                //    If fail: ROLLBACK (Lock removed).
                // 4. COMMIT. (Lock persisted).
                // 5. Check Plan Price.
                //    If fail: DELETE LOCK (manually)? Or just fail.
                // 6. Proceed to API call.

                // This seems safer for `process_transactions.php` which doesn't control the whole flow.

                if (isset($ref) && !empty($ref)) {
                    $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
                    $user_id = $user_data['id'];

                    // DELETE LOCK (Release Mutex)
                    $wpdb->delete($vend_lock, ['user_id' => $user_id]);
                    $wpdb->query("COMMIT");
                }

                return [
                    'status' => 'failed',
                    'message' => 'Transaction failed',
                    'raw' => $rawBody //$parsed
                ];
            }
            // error_log("no ref");


        }

        // error_log("successful");


        /* -------------------------------
         * RETRIEVE EXTRA KEYS
         * ------------------------------- */
        $captured = [];
        $retrieveKeys = $settings['retrieve_keys'] ?? '';
        if (!empty($retrieveKeys)) {
            $checks = array_map('trim', explode(',', $retrieveKeys));

            foreach ($checks as $check) {
                if (!str_contains($check, ':'))
                    continue;

                [$k, $v] = array_map('trim', explode(':', $check, 2));
                // error_log($k . " - " . $v);

                $val = $this->catch_key($parsed, $v);
                if ($val !== false) {
                    $captured[$k] = $val;
                }
            }
        }

        // If we are here, it's success.
        if (isset($ref) && !empty($ref)) {
            global $wpdb;
            $vend_lock = $wpdb->prefix . 'vtupress_vend_lock';
            $user_id = $user_data['id'] ?? 0;
            if (!empty($user_id)) {
                $wpdb->delete($vend_lock, ['user_id' => $user_id]);
            }
        }

        $wpdb->query("COMMIT");

        return [
            'status' => 'success',
            'data' => $captured,
            'raw' => $rawBody
        ];
    }



}