<?php
if (!defined('ABSPATH'))
    exit;

include_once __DIR__ . "/process_transactions.php";

interface Vtupress_History_Interface
{

    /**
     * Extra columns this service wants to show
     * key => Label
     */
    public function history_columns(): array;

    /**
     * Resolve column value from transaction row
     */
    public function resolve_history_column(
        string $column,
        array $transaction
    ): string;
}


class Vtupress_Service_Container
{
    protected static array $instances = [];

    /**
     * Get a service instance
     * @param string $id Service ID
     * @param string|false $service_type Optional type
     * @param int|null $user_id Optional WP user ID for API requests
     * @return mixed|null
     */
    public static function get(string $id, $service_type = false, ?int $user_id = null, ?string $country_override = null)
    {
        global $wpdb;

        $registry = vtupress_service_registry();

        if (!isset($registry[$id])) {
            return null;
        }

        $key = $id . ':' . ($service_type ?: 'default') . ':' . ($user_id ?? 'default') . ':' . ($country_override ?? 'default');

        if (!isset(self::$instances[$key])) {
            $class = $registry[$id];

            // Pass user ID to service so it can resolve country
            $sv = new $class($wpdb, $service_type, $user_id);

            // Load Overridden Service Types (Fix for custom types not being available)
            // We must fetch the MAIN service settings (without service_type suffix) because that's where the list is stored.
            $main_option_key = 'vtupress_service_' . $sv->id . '_' . $sv->current_country;
            $main_settings = Vtupress_Option::get($main_option_key, []);

            if (!empty($main_settings['service_types']) && is_array($main_settings['service_types'])) {
                $sv->service_types = array_map('strtolower', $main_settings['service_types']);
            }

            // Resolve country
            if ($country_override) {
                $country = Vtupress_Country_Manager::validate($country_override, false);
            } else {
                $country = Vtupress_Country_Manager::validate($sv->current_country, false);
            }

            if (is_wp_error($country)) {
                return null;
            }

            if (!empty($sv->custom_id)) {
                if (Vtupress_Option::get("vtupress_custom_{$sv->custom_id}", "false") != "yes") {
                    return null;
                }
            }

            if (!in_array('all', $sv->countries) && !in_array($country, $sv->countries)) {
                return null;
            }

            self::$instances[$key] = $sv;
        }

        return self::$instances[$key];
    }

    public static function all($boot = false, ?string $country = null): array
    {
        $services = [];

        foreach (vtupress_service_registry() as $id => $class) {
            $dservice = self::get($id, false, null, $country);
            if (!$dservice) {
                continue;
            }
            $services[$id] = $dservice;
            if ($boot) {
                $services[$id]->boot();
            }
        }

        return $services;
    }
}



abstract class Vtupress_services extends Vtupress_Process_Transactions
{
    protected wpdb $db;
    protected bool $booted = false;
    public string $id = '';
    public string $name = '';
    public bool $status = false;
    public string $description = '';
    public string $image = '';

    public string $icon = '';
    public $custom_id = '';
    public bool $disable_global = false;
    public bool $uses_api = false;
    public bool $uses_provider = false;
    public bool $uses_plans = false;
    public bool $is_voucher = false;
    public bool $uses_api_setup = true;
    public bool $allow_custom_providers = false;
    public array $default_providers = [];
    public array $providers = [];

    public array $plans = [];
    public bool $uses_bonus = false;
    public $service_type = false;

    public array $service_types = [];

    public array $level_result = [];
    public string $currency = '';
    public string $currency_name = '';
    public string $currency_symbol = '';
    public string $country_code = '';
    public array $country_array = [];

    public string $current_country = '';
    public array $countries = [];

    public string $iso = '';

    protected array $settings_cache = [];
    public $uses_ports = false;
    public $ports = [];
    public $nonce;
    public array $payload = [];
    public $default_gateway_url = '';
    public $label_provider = 'Provider';
    public $label_number = 'Number';
    public $label_service_type = 'Type';
    public $label_plan = 'Plan';
    public $has_verification = false;
    protected ?int $user_id = null;
    protected $request_server = 'https://vtupress.com/';



    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->db = $db;
        $this->user_id = $user_id;
        $this->request_server = home_url();
        $this->service_type = $service_type
            ? strtolower(preg_replace("/\s|-/", "_", $service_type))
            : false;
        $this->service_types = array_map('strtolower', $this->service_types);
        $this->current_country = Vtupress_Country_Context::get($user_id);
    }


    protected function error($error)
    {
        $this->booted = false;
        throw new RuntimeException("Service {$this->id} is not accessible : reason : $error");

    }

    public function catch_key($array, $key)
    {
        if (!is_array($array))
            return false;

        $lowerKey = strtolower($key);

        foreach ($array as $k => $v) {
            if (strtolower($k) === $lowerKey) {
                return $v;
            }
        }

        foreach ($array as $element) {
            if (is_array($element)) {
                $result = $this->catch_key($element, $key);
                if ($result !== false) {
                    return $result;
                }
            }
        }

        return false;
    }

    /**
     * Verify service input (e.g. IUC, Meter Number)
     * @param array $args
     * @return string|array|WP_Error
     */
    public function verify(array $args)
    {
        return new WP_Error('not_implemented', 'Verification method not implemented for this service.');
    }

    public static function ajax_verify()
    {
        // Security check
        // check_ajax_referer('vtupress_verify_nonce', 'security'); // Optional: Add nonce check if needed in frontend

        $service_id = sanitize_text_field($_REQUEST['service_id'] ?? '');
        $service_type = sanitize_text_field($_REQUEST['service_type'] ?? 'subscription');
        $provider = sanitize_text_field($_REQUEST['provider'] ?? ''); // Previously network or cable/bill
        $number = sanitize_text_field($_REQUEST['number'] ?? ''); // IUC or Meter

        if (empty($service_id) || empty($number)) {
            wp_send_json_error(['message' => 'Missing parameters']);
        }

        try {
            $service = Vtupress_Service_Container::get($service_id, $service_type);
            $service->boot();

            $result = $service->verify([
                'provider' => $provider,
                'number' => $number,
                'type' => $service_type
            ]);

            if (is_wp_error($result)) {
                wp_send_json_error(['message' => $result->get_error_message()]);
            } else {
                wp_send_json_success($result);
            }

        } catch (\Throwable $e) {
            wp_send_json_error(['message' => $e->getMessage()]);
        }
    }

    /**
     * Handle generic frontend AJAX requests for this service.
     * Override this method in your child class to handle custom form submissions.
     * 
     * @param array $request The $_POST data
     * @return array ['status' => 'success'|'failed', 'message' => '...', 'data' => ...]
     */
    public function process_frontend_request($request)
    {
        return ['status' => 'failed', 'message' => 'Frontend processing not implemented for this service'];
    }



    /**
     * Render Mutable Amount Frontend (Airtime, Bill)
     */
    public function render_mutable_frontend($user, $runcode = "")
    {
        $balance = $user["balance"];
        $service = $this;
        $service->boot();
        $settings = $service->get_settings();
        $enableBonus = ($settings["service_bonus_enable"] ?? "no") === "yes";
        $discountMethod = $settings["discount_method"] ?? "discount";

        // Dynamic Providers & Types
        $all_providers = $service->providers;
        $service_types = $service->service_types; // e.g., ['vtu', 'share'] or ['prepaid', 'postpaid']

        $providerArray = [];
        $active_service_types = [];
        $bonuses = [];
        $serviceBonus = [];

        // Tiered Bonus Logic
        $user_plan = strtolower($user['plan'] ?? 'customer');
        $reward_table = $this->db->prefix . 'vtupress_level_service_rewards';
        $tiered_bonus_val = $this->db->get_var($this->db->prepare(
            "SELECT bonus_value FROM $reward_table WHERE level_name = %s AND service_id = %s AND set_country = %s",
            $user_plan,
            $this->id,
            $service->current_country
        ));
        $tiered_bonus_val = $tiered_bonus_val ?: "0";

        // Build Provider/Type Map
        foreach ($all_providers as $prov) {
            $prov = strtolower($prov);
            $providerArray[$prov] = [];

            foreach ($service_types as $stype) {
                // Check if this type is globally enabled
                $s = \Vtupress_Service_Container::get($this->id, $stype);
                $s->boot();
                $sset = $s->get_settings();

                if (($sset["enable"] ?? 'no') == 'yes') {
                    $active_service_types[$stype] = $stype; // Mark type as active at least once
                }

                // Check if provider is enabled for this type
                if (($sset["{$prov}_status"] ?? "off") === "on") {
                    $providerArray[$prov][] = strtolower($stype);
                    $bonuses["{$prov}_{$stype}"] = $sset["{$prov}_bonus"] ?? 0;
                }

                // Service-level bonus for this type
                $serviceBonus[$stype] = $sset[strtolower($stype) . "_bonus"] ?? 0;
            }
        }

        $ports = $service->uses_ports ? $service->ports : false;
        $transaction_settings = $this->db->get_row("SELECT enable_beneficiary FROM {$this->db->prefix}vtupress_transaction_settings WHERE id=1", ARRAY_A);

        // UI Labels
        $label_provider = $service->label_provider;
        $label_number = $service->label_number;
        $label_type = $service->label_service_type ?? 'Type';
        $has_verification = $service->has_verification;

        ob_start();
        ?>
        <style>
            select.form-select,
            input[type='number'],
            input[type='text'] {
                height: 50px !important;
            }

            .purchase-btn {
                height: 65px !important;
            }
        </style>
        <link rel="stylesheet"
            href="<?php echo plugins_url('vtupress-templates/templates/msorg_template'); ?>/msorg_template/form.css"
            media="all">

        <div class="container-md mt-3">
            <div class="service-form p-3" style="border: 1px solid grey; border-radius: 5px;">
                <div class="mb-2">
                    <form class="for" id="cfor" method="post" <?php echo apply_filters('formaction', 'target="_self"'); ?>>

                        <!-- Provider Select -->
                        <div class="mb-2">
                            <label for="provider" class="form-label"><?php echo $label_provider; ?></label>
                            <select class="form-select form-select-sm service-provider" name="provider">
                                <option value="">- Select <?php echo $label_provider; ?> -</option>
                                <?php foreach ($all_providers as $prov):
                                    // Only show if provider has AT LEAST ONE enabled type
                                    if (empty($providerArray[strtolower($prov)]))
                                        continue;
                                    ?>
                                    <option value="<?php echo strtolower($prov); ?>"><?php echo strtoupper($prov); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <div class="invalid-feedback">Error: <span class="provider-error"></span></div>
                        </div>

                        <!-- Service Type Select -->
                        <div class="mb-2">
                            <label for="network_type" class="form-label"><?php echo $label_type; ?></label>
                            <select id="servicechoice" name="servicechoice" class="service-choice form-select form-select-sm">
                                <option value="">- Select Type -</option>
                                <!-- Populated by JS -->
                            </select>
                            <div class="invalid-feedback">Error: <span class="type-error"></span></div>
                        </div>

                        <!-- Number Input & Verification -->
                        <div class="mb-2">
                            <label for="phone" class="form-label"><?php echo $label_number; ?></label>
                            <div class="input-group">
                                <input id="phone" name="phone" list="beneficiaries" type="number"
                                    class="form-control service-number" placeholder="<?php echo $label_number; ?>">
                                <?php if ($has_verification): ?>
                                    <button class="btn btn-secondary verify-service" type="button">Verify</button>
                                <?php endif; ?>
                            </div>
                            <datalist id="beneficiaries">
                                <?php
                                $bens = !empty($user["beneficiaries"]) ? explode(",", $user["beneficiaries"]) : [];
                                if (count($bens) >= 1 && ($transaction_settings["enable_beneficiary"] ?? 'no') == "yes") {
                                    foreach ($bens as $ben) {
                                        if (!empty($ben))
                                            echo "<option value='$ben'>";
                                    }
                                }
                                ?>
                            </datalist>
                            <?php if ($has_verification): ?>
                                <small class="text-success verify-name"></small>
                            <?php endif; ?>
                            <div class="invalid-feedback">Error: <span class="number-error"></span>.</div>
                        </div>

                        <!-- Amount Input -->
                        <div class="mb-2">
                            <label for="amount" class="form-label">Amount</label>
                            <div class="input-group mb-2">
                                <span class="input-group-text">NGN.</span>
                                <input id="amt" name="amount" type="number" class="form-control service-amount"
                                    oninput="calcit();" placeholder="Amount">
                                <span class="input-group-text">.00</span>
                            </div>
                        </div>

                        <!-- Payment Summary -->
                        <div class="mb-2">
                            <label
                                class="form-label"><?php echo $discountMethod == "commission" ? "Charge Back Bonus" : "Amount To Pay"; ?></label>
                            <div class="input-group mb-2">
                                <span class="input-group-text">NGN.</span>
                                <input id="amttopay" type="number" class="form-control amttopay" max="<?php echo $balance; ?>"
                                    placeholder="Amount To Pay" readonly>
                                <span class="input-group-text">.00</span>
                                <div class="invalid-feedback">Error: <span class="amount-error"></span></div>
                            </div>
                        </div>

                        <?php if ($ports): ?>
                            <div class="mb-2">
                                <div class="form-check">
                                    <input class="form-check-input bypass" type="checkbox" id="flexCheckDefault">
                                    <label class="form-check-label" for="flexCheckDefault">Bypass Validator</label>
                                </div>
                            </div>
                        <?php endif; ?>

                        <div class="vstack gap-2">
                            <button type="button"
                                class="w-full p-2 text-xs font-bold text-white uppercase bg-indigo-600 rounded shadow purchase-btn btn btn-primary fs-5"
                                data-bs-toggle="modal" data-bs-target="#confirmModal">Purchase</button>
                        </div>
                    </form>

                    <!-- Confirmation Modal -->
                    <div class="modal fade" id="confirmModal" tabindex="-1" aria-hidden="true">
                        <div class="modal-dialog">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title">Confirm Transaction</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                </div>
                                <div class="modal-body">
                                    <?php echo $label_provider; ?> : <span class="confirm-provider"></span><br>
                                    <?php echo $label_number; ?> : <span class="confirm-number"></span> <span
                                        class="confirm-verify-name text-success"></span><br>
                                    Amount: ₦<span class="confirm-amount"></span><br>
                                    To Pay: ₦<span class="amttopay2"></span><br>
                                    <?php echo $enableBonus ? "Discount" : "Commission"; ?> : <span
                                        class="confirm-discount"></span><br>
                                    Status : <span class="confirm-status"></span>
                                    <?php if (($settings["require_pin"] ?? 'no') == "yes"): ?>
                                        <div class="input-group mt-2">
                                            <span class="input-group-text">PIN</span>
                                            <input class="form-control pin" type="number" name="pin">
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn-danger" data-bs-dismiss="modal">Cancel</button>
                                    <button type="button" class="btn-success proceed-btn">Proceed</button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <script>
                        var bonusEnabled = <?php echo $enableBonus ? 'true' : 'false'; ?>;
                        var discountMethod = "<?php echo esc_js($discountMethod); ?>";
                        var networkBonus = [];
                        var serviceBonus = [];
                        var providerArray = [];
                        var tieredBonus = "<?php echo $tiered_bonus_val; ?>";
                        var providerArray = [];
                        var tieredBonus = "<?php echo $tiered_bonus_val; ?>";
                        var hasVerification = <?php echo $has_verification ? 'true' : 'false'; ?>;
                        var serviceId = "<?php echo $this->id; ?>";

                        <?php foreach ($bonuses as $key => $value): ?>
                            networkBonus["<?php echo strtolower($key); ?>"] = "<?php echo empty($value) ? 0 : $value; ?>";
                        <?php endforeach; ?>

                        <?php foreach ($providerArray as $prov => $types): ?>
                            providerArray["<?php echo strtolower($prov); ?>"] = <?php echo json_encode($types); ?>;
                        <?php endforeach; ?>

                        <?php foreach ($serviceBonus as $stype => $val): ?>
                            serviceBonus["<?php echo strtolower($stype); ?>"] = <?php echo floatval($val); ?>;
                        <?php endforeach; ?>

                        var hasPorts = <?php echo ($ports && !empty($ports)) ? 'true' : 'false'; ?>;
                        var ports = hasPorts ? JSON.parse('<?php echo json_encode($ports); ?>') : {};

                        function calcit() {
                            var original = parseFloat(jQuery("#amt").val()) || 0;
                            var provider = jQuery(".service-provider").val();
                            var type = jQuery("#servicechoice").val();

                            var discount = networkBonus[provider + '_' + type] || 0;
                            if (bonusEnabled && serviceBonus[type] > 0) discount = serviceBonus[type];

                            var bonusAmount = (String(discount).includes('%'))
                                ? (original * parseFloat(discount.replace('%', ''))) / 100
                                : parseFloat(discount);

                            // Tiered Bonus
                            var tieredAmount = 0;
                            if (tieredBonus && tieredBonus !== "0") {
                                if (tieredBonus.includes('%')) {
                                    tieredAmount = (original * parseFloat(tieredBonus.replace('%', ''))) / 100;
                                } else {
                                    tieredAmount = parseFloat(tieredBonus);
                                }
                            }
                            bonusAmount += tieredAmount;

                            var finalAmount = (discountMethod === "discount") ? (original - bonusAmount) : bonusAmount;
                            finalAmount = Math.max(0, finalAmount);

                            jQuery(".amttopay").val(finalAmount);
                            jQuery(".amttopay2").text(finalAmount);
                            jQuery(".confirm-discount").text(bonusAmount.toFixed(2));
                        }

                        // --- Event Listeners ---

                        jQuery(".service-provider").on("change", function () {
                            var provider = jQuery(this).val();
                            var choiceSelect = jQuery("#servicechoice");
                            choiceSelect.empty().append('<option value="">- Select Type -</option>');

                            if (provider && providerArray[provider]) {
                                providerArray[provider].forEach(function (t) {
                                    choiceSelect.append('<option value="' + t + '">' + t.replace(/_/g, " ").toUpperCase() + '</option>');
                                });
                            }
                            calcit();
                        });

                        jQuery("#servicechoice, #amt").on("change keyup", calcit);

                        // Verification Logic
                        if (hasVerification) {
                            jQuery(".verify-service").on("click", function () {
                                var btn = jQuery(this);
                                var num = jQuery(".service-number").val();
                                var prov = jQuery(".service-provider").val();
                                var type = jQuery("#servicechoice").val();

                                if (!prov || !num) { alert("Select Provider and Enter Number"); return; }

                                btn.text("Verifying...");
                                jQuery.post(vtupressInitPluginData.ajax_url, {
                                    action: 'vtupress_verify_service',
                                    service_id: serviceId,
                                    service_type: type, // Optional, depending on service
                                    provider: prov,
                                    number: num
                                }, function (resp) {
                                    btn.text("Verify");
                                    if (resp.success) {
                                        var name = resp.data.name || resp.data; // Adjust based on response structure
                                        jQuery(".verify-name").text(name);
                                        jQuery(".confirm-verify-name").text(name);
                                    } else {
                                        jQuery(".verify-name").text(resp.data.message || "Verification Failed").removeClass("text-success").addClass("text-danger");
                                    }
                                });
                            });
                        }

                        jQuery(".purchase-btn").on("click", function () {
                            var provider = jQuery(".service-provider").val();
                            var number = jQuery(".service-number").val();
                            var amount = jQuery("#amt").val();
                            var type = jQuery("#servicechoice").val();

                            jQuery(".confirm-provider").text(provider.toUpperCase());
                            jQuery(".confirm-number").text(number);
                            jQuery(".confirm-amount").text(amount);

                            // Simple Validation
                            var valid = true;
                            if (!provider || !number || !amount || parseFloat(amount) <= 0) valid = false;

                            if (valid) {
                                jQuery(".confirm-status").text("Ready").removeClass("text-danger").addClass("text-success");
                                jQuery(".proceed-btn").show();
                            } else {
                                jQuery(".confirm-status").text("Invalid Details").removeClass("text-success").addClass("text-danger");
                                jQuery(".proceed-btn").hide();
                            }
                        });

                        jQuery(".proceed-btn").click(function () {
                            var obj = {
                                action: serviceId,
                                type: jQuery("#servicechoice").val(),
                                provider: jQuery(".service-provider").val(),
                                recipient: jQuery(".service-number").val(),
                                amount: jQuery("#amt").val(),
                                ref: "<?php echo !empty($runcode) ? $runcode : time(); ?>"
                            };
                            <?php if (($settings["require_pin"] ?? 'no') === "yes"): ?>
                                obj.pin = jQuery(".pin").val();
                            <?php endif; ?>

                            jQuery('.btn-close').trigger('click');
                            doAjax(vtupressobj.apiUrl + obj['action'], obj, "post", "text", function (res) {
                                if (res === "100") {
                                    showToast(json.message ? json.message : (json.data.message ? json.data.message : "Successful"), "green", 3000, true);
                                    setTimeout(function () {
                                        location.reload();
                                    }, 1500);
                                } else {
                                    try {
                                        let json = JSON.parse(res);
                                        if (json.status === 'success' || json.success === true) {
                                            showToast(json.message ? json.message : (json.data.message ? json.data.message : "Successful"), "green", 3000, true);
                                            setTimeout(function () {
                                                location.reload();
                                            }, 1500);
                                        } else {
                                            showToast(json.message ? json.message : (json.data.message ? json.data.message : res), "red", 5000, true);
                                        }
                                    } catch (e) {
                                        showToast(res, "red", 5000, true);
                                    }
                                }
                            }, dprocessCallBack, derrorCallBack, true);
                        });
                    </script>
                </div>
            </div>
        </div>
        <?php
        echo ob_get_clean();
    }


    /**
     * Render Pin/Voucher Frontend (Recharge Card, Data Card, Exam Pin)
     */
    public function render_pin_frontend($user, $runcode = "")
    {
        $balance = $user["balance"];
        $service = $this;
        $service->boot();
        $settings = $service->get_settings();
        $enableBonus = ($settings["service_bonus_enable"] ?? "no") === "yes";
        $discountMethod = $settings["discount_method"] ?? "discount";

        $all_providers = $service->providers;
        $service_types = $service->service_types;
        $plans = $service->plans;

        $providerArray = [];
        $active_service_types = [];
        $bonuses = [];

        // Tiered Bonus Logic
        $user_plan = strtolower($user['plan'] ?? 'customer');
        $reward_table = $this->db->prefix . 'vtupress_level_service_rewards';
        $tiered_bonus_val = $this->db->get_var($this->db->prepare(
            "SELECT bonus_value FROM $reward_table WHERE level_name = %s AND service_id = %s AND set_country = %s",
            $user_plan,
            $this->id,
            $service->current_country
        ));
        $tiered_bonus_val = $tiered_bonus_val ?: "0";

        foreach ($all_providers as $prov) {
            $prov = strtolower($prov);
            $providerArray[$prov] = [];

            foreach ($service_types as $stype) {
                // Check if this type is globally enabled
                $s = \Vtupress_Service_Container::get($this->id, $stype);
                $s->boot();
                $sset = $s->get_settings();

                if (($sset["enable"] ?? 'no') == 'yes') {
                    $active_service_types[$stype] = $stype;
                }

                if (($sset["{$prov}_status"] ?? "off") === "on") {
                    $providerArray[$prov][] = strtolower($stype);
                    $bonuses["{$prov}_{$stype}"] = $sset["{$prov}_bonus"] ?? 0;
                }
            }
        }

        $label_provider = $service->label_provider;
        $label_number = "Quantity"; // Hardcoded for Pin services
        $label_type = $service->label_service_type ?? 'Type';
        $label_plan = $service->label_plan ?? 'Plan';
        $has_verification = false; // No verification for quantity
        $has_types = !empty($active_service_types);

        // Fetch Dynamic PIN Values
        // Fetch Dynamic PIN Values with Price (FIFO)
        global $wpdb;
        $pin_table = $wpdb->prefix . 'vtupress_internal_pins';

        // Fetch ALL unused pins ordered by ID ASC (FIFO)
        // We will filter in PHP to only pick the FIRST one for each Plan+Value combination.
        $pin_rows = $wpdb->get_results("SELECT plan_id, value, price FROM $pin_table WHERE service_id = '{$this->id}' AND status = 'unused' ORDER BY id ASC", ARRAY_A);

        $dynamic_pin_values = [];
        $seen_combinations = [];

        if (!empty($pin_rows)) {
            foreach ($pin_rows as $pr) {
                $p_prov = strtolower($pr['plan_id']);
                $p_val = $pr['value'];
                $combo_key = $p_prov . '_' . $p_val;

                // If we have already seen this Plan+Value, skip it (because FIFO means we only sell the first one)
                if (isset($seen_combinations[$combo_key])) {
                    continue;
                }

                $seen_combinations[$combo_key] = true;

                if (!isset($dynamic_pin_values[$p_prov]))
                    $dynamic_pin_values[$p_prov] = [];

                // Store object {val: '...', price: ...}
                $dynamic_pin_values[$p_prov][] = [
                    'val' => $p_val,
                    'price' => floatval($pr['price'])
                ];
            }
        }

        ob_start();
        ?>
        <style>
            select.form-select,
            input[type='number'],
            input[type='text'] {
                height: 50px !important;
            }

            .purchase-btn {
                height: 65px !important;
            }
        </style>
        <link rel="stylesheet"
            href="<?php echo plugins_url('vtupress-templates/templates/msorg_template'); ?>/msorg_template/form.css"
            media="all">

        <div class="container-md mt-3">
            <div class="service-form p-3" style="border: 1px solid grey; border-radius: 5px;">
                <div class="mb-2">
                    <form class="for" id="cfor" method="post" <?php echo apply_filters('formaction', 'target="_self"'); ?>>

                        <div class="mb-2">
                            <label class="form-label"><?php echo $label_provider; ?></label>
                            <select class="form-select form-select-sm service-provider" name="network">
                                <option value="">- Select <?php echo $label_provider; ?> -</option>
                                <?php foreach ($all_providers as $prov):
                                    if (empty($providerArray[strtolower($prov)]))
                                        continue;
                                    ?>
                                    <option value="<?php echo strtolower($prov); ?>"><?php echo strtoupper($prov); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <div class="invalid-feedback">Error: <span class="provider-error"></span></div>
                        </div>



                        <?php if ($has_types): ?>
                            <div class="mb-2 type-div">
                                <label class="form-label"><?php echo $label_type; ?></label>
                                <select id="servicechoice" name="servicechoice" class="service-choice form-select form-select-sm">
                                    <option value="">- Select Type -</option>
                                </select>
                            </div>
                        <?php endif; ?>

                        <div class="mb-2 plan-div" style="display:none;">
                            <label class="form-label"><?php echo $label_plan; ?></label>
                            <select class="form-select form-select-sm service-plan" name="cplan">
                                <option value="" price="0">---Select---</option>
                            </select>
                            <div class="invalid-feedback">Error: <span class="plan-error"></span></div>
                        </div>

                        <label class="form-label">
                            <?php
                            if ($this->id == 'data_card')
                                echo 'Data Plan';
                            elseif ($this->id == 'education')
                                echo 'Exam Type';
                            else
                                echo 'Value (Denomination)';
                            ?>
                        </label>
                        <select id="service-value" name="value" class="form-select service-value">
                            <option value="">- Select Value -</option>
                            <?php
                            // Options populated via JS
                            ?>
                        </select>
                        <label class="small text-muted mb-2">Select the value of the card/pin.</label>
                </div>

                <div class="mb-2">
                    <label class="form-label"><?php echo $label_number; ?></label>
                    <div class="input-group">
                        <input id="phone" name="phone" type="number" min="1" value="1" class="form-control service-number"
                            placeholder="Quantity">
                    </div>
                    <div class="invalid-feedback">Error: <span class="number-error"></span>.</div>
                </div>

                <div class="mb-2">
                    <label class="form-label">Amount</label>
                    <div class="input-group mb-2">
                        <span class="input-group-text">NGN.</span>
                        <input id="amt" name="amount" type="number" class="form-control service-amount" readonly
                            placeholder="Amount">
                        <span class="input-group-text">.00</span>
                    </div>
                </div>

                <div class="mb-2">
                    <label
                        class="form-label"><?php echo $discountMethod == "commission" ? "Charge Back Bonus" : "Amount To Pay"; ?></label>
                    <div class="input-group mb-2">
                        <span class="input-group-text">NGN.</span>
                        <input id="amttopay" type="number" class="form-control amttopay" max="<?php echo $balance; ?>"
                            placeholder="Amount To Pay" readonly>
                        <span class="input-group-text">.00</span>
                    </div>
                </div>

                <div class="vstack gap-2">
                    <button type="button"
                        class="w-full p-2 text-xs font-bold text-white uppercase bg-indigo-600 rounded shadow purchase-btn btn btn-primary fs-5"
                        data-bs-toggle="modal" data-bs-target="#confirmModal">Purchase</button>
                </div>
                </form>

                <!-- Confirmation Modal -->
                <div class="modal fade" id="confirmModal" tabindex="-1" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">Confirm Transaction</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                            </div>
                            <div class="modal-body">
                                <?php echo $label_provider; ?> : <span class="confirm-provider"></span><br>
                                <?php if ($this->id != 'recharge_card' && $this->id != 'data_card' && $this->id != 'education'): ?>
                                    Plan: <span class="confirm-plan"></span><br>
                                <?php endif; ?>
                                Value : <span class="confirm-value"></span><br>
                                Quantity : <span class="confirm-number"></span><br>
                                Amount: ₦<span class="confirm-amount"></span><br>
                                To Pay: ₦<span class="amttopay2"></span><br>
                                <?php echo $enableBonus ? "Discount" : "Commission"; ?> : <span
                                    class="confirm-discount"></span><br>
                                Status : <span class="confirm-status"></span>
                                <?php if (($settings["require_pin"] ?? 'no') == "yes"): ?>
                                    <div class="input-group mt-2">
                                        <span class="input-group-text">PIN</span>
                                        <input class="form-control pin" type="number" name="pin">
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn-danger" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn-success proceed-btn">Proceed</button>
                            </div>
                        </div>
                    </div>
                </div>

                <script>
                    var bonusEnabled = <?php echo $enableBonus ? 'true' : 'false'; ?>; var discountMethod = "<?php echo esc_js($discountMethod); ?>";
                    var networkBonus = [];
                    var serviceBonus = [];
                    var providerArray = [];
                    var tieredBonus = "<?php echo $tiered_bonus_val; ?>";
                    var plans = <?php echo json_encode($plans); ?>;
                    var planMap = {};
                    var serviceId = "<?php echo $this->id; ?>";
                    var dynamicPinValues = <?php echo json_encode($dynamic_pin_values); ?>;

                    <?php foreach ($bonuses as $key => $value): ?>
                        networkBonus["<?php echo strtolower($key); ?>"] = "<?php echo empty($value) ? 0 : $value; ?>";
                    <?php endforeach; ?>

                    <?php foreach ($providerArray as $prov => $types): ?>
                        providerArray["<?php echo strtolower($prov); ?>"] = <?php echo json_encode($types); ?>;
                    <?php endforeach; ?>

                    // Build Plan Map
                    plans.forEach(function (p) {
                        var net = (p.network || '').toLowerCase();
                        var type = (p.service_type || 'subscription').toLowerCase();
                        var key = net + "_" + type;
                        if (!planMap[key]) planMap[key] = [];
                        planMap[key].push(p);
                    });


                    function calcit() {
                        var qty = parseFloat(jQuery(".service-number").val()) || 1;
                        var selectedOption = jQuery(".service-plan option:selected");
                        var price = parseFloat(selectedOption.attr("price")) || 0;
                        var valStr = jQuery("#service-value").val();

                        // Use Value for Price if Manual Service (Recharge Card)
                        // Use Value for Price if Manual Service (Recharge Card) AND price attribute is missing or 0? 
                        // Actually, for Recharge Card, we might still want to use the stored price if available?
                        // User said: "value/denomination can hold e.g 200 in which the price... is populated... but for data... no price"
                        // So for Recharge Card, Value can be Price.

                        var selectedValueOption = jQuery("#service-value option:selected");
                        var valuePrice = parseFloat(selectedValueOption.attr("price")) || 0;

                        if (valuePrice > 0) {
                            price = valuePrice;
                        } else if (serviceId === 'recharge_card' && valStr) {
                            price = parseFloat(valStr) || 0;
                        }
                        // Data Card Logic? 
                        // If we don't know price, we can't calculate. 
                        // Assuming for now simple demo values or specific logic handled by plan mapping if available.
                        // But sticking to simple Value = Price doesn't work for Data.

                        var original = price * qty;
                        jQuery("#amt").val(original);

                        var provider = jQuery(".service-provider").val();
                        var type = jQuery("#servicechoice").val();

                        var discount = networkBonus[provider + '_' + type] || 0;
                        // Check for direct provider bonus if type missing
                        if (!discount && provider) discount = networkBonus[provider] || 0; // Fallback

                        var bonusAmount = 0;

                        if (String(discount).includes('%')) {
                            bonusAmount = (original * parseFloat(discount.replace('%', ''))) / 100;
                        } else {
                            bonusAmount = parseFloat(discount) * qty;
                        }

                        // Tiered Bonus
                        var tieredAmount = 0;
                        if (tieredBonus && tieredBonus !== "0") {
                            if (tieredBonus.includes('%')) {
                                tieredAmount = (original * parseFloat(tieredBonus.replace('%', ''))) / 100;
                            } else {
                                tieredAmount = parseFloat(tieredBonus) * qty;
                            }
                        }
                        bonusAmount += tieredAmount;

                        var finalAmount = (discountMethod === "discount") ? (original - bonusAmount) : original;
                        finalAmount = Math.max(0, finalAmount);

                        jQuery(".amttopay").val(finalAmount);
                        jQuery(".amttopay2").text(finalAmount);
                        jQuery(".confirm-discount").text(bonusAmount.toFixed(2));
                        jQuery(".confirm-value").text(valStr);
                    }

                    // Recalculate on Value Change
                    jQuery(document).on("change keyup", "#service-value", function () {
                        calcit();
                    });

                    // --- Event Listeners ---

                    jQuery(".service-provider").on("change", function () {
                        var provider = jQuery(this).val();
                        var choiceSelect = jQuery("#servicechoice");
                        var typeDiv = jQuery(".type-div");

                        choiceSelect.empty().append('<option value="">- Select Type -</option>');
                        jQuery(".service-plan").empty().append('<option value="" price="0">---Select---</option>');
                        jQuery(".plan-div").hide();

                        if (provider && providerArray[provider]) {
                            var types = providerArray[provider];
                            if (types.length > 0) {
                                types.forEach(function (t) {
                                    choiceSelect.append('<option value="' + t + '">' + t.replace(/_/g, " ").toUpperCase() + '</option>');
                                });
                                if (types.length === 1 && types[0] === 'subscription') {
                                    // Auto select if only subscription
                                    // choiceSelect.val('subscription').trigger('change');
                                    // Or hide types if implicit? keeping it visible for now.
                                }
                                typeDiv.show();
                            } else {
                                typeDiv.hide();
                            }
                        }

                        // Update Values
                        var valSelect = jQuery("#service-value");
                        valSelect.empty().append('<option value="">- Select Value -</option>');
                        if (provider && dynamicPinValues && dynamicPinValues[provider]) {
                            dynamicPinValues[provider].forEach(function (item) {
                                // check if item is object or string (legacy support if mixed?)
                                var val = item.val || item;
                                var price = item.price || 0;
                                valSelect.append('<option value="' + val + '" price="' + price + '">' + val + ' (₦' + price + ')</option>');
                            });
                        }

                        calcit();
                    });

                    jQuery("#servicechoice").on("change", function () {
                        var type = jQuery(this).val();
                        var provider = jQuery(".service-provider").val();
                        var planSelect = jQuery(".service-plan");

                        planSelect.empty().append('<option value="" price="0">---Select---</option>');

                        var key = provider + "_" + type; // type is already lowercase from value

                        // Fallback for 'subscription' vs ''
                        if (!type && providerArray[provider] && providerArray[provider].includes('subscription')) {
                            // key = provider + "_subscription"; 
                        }

                        if (planMap[key]) {
                            planMap[key].forEach(function (p) {
                                planSelect.append('<option value="' + p.plan_id + '" price="' + p.price + '">' + p.name + ' (₦' + p.price + ')</option>');
                            });
                            jQuery(".plan-div").show();
                        } else {
                            jQuery(".plan-div").hide();
                        }
                        calcit();
                    });

                    jQuery(".service-plan").on("change", calcit);
                    jQuery(".service-number").on("change keyup", calcit); // Quantity change

                    jQuery(".purchase-btn").on("click", function () {
                        var provider = jQuery(".service-provider").val();
                        var qty = jQuery(".service-number").val();
                        var amount = jQuery("#amt").val();
                        var planName = jQuery(".service-plan option:selected").text();

                        jQuery(".confirm-provider").text(provider.toUpperCase());
                        jQuery(".confirm-number").text(qty);
                        jQuery(".confirm-amount").text(amount);
                        jQuery(".confirm-plan").text(planName);

                        // Simple Validation
                        var valid = true;
                        if (!provider || !qty || !amount || parseFloat(amount) <= 0) valid = false;

                        if (valid) {
                            jQuery(".confirm-status").text("Ready").removeClass("text-danger").addClass("text-success");
                            jQuery(".proceed-btn").show();
                        } else {
                            jQuery(".confirm-status").text("Invalid Details").removeClass("text-success").addClass("text-danger");
                            jQuery(".proceed-btn").hide();
                        }
                    });


                    jQuery(".proceed-btn").click(function () {
                        var obj = {
                            action: "<?php echo $this->id; ?>", // Use service ID
                            type: jQuery("#servicechoice").val(),
                            provider: jQuery(".service-provider").val(),
                            // recipient: jQuery(".service-number").val(), // Passing Qty as recipient/phone for now (or handle in backend??)
                            // Actually backend expects 'recipient' to be the target. For cards, target IS the user usually, or email? 
                            // But legacy might use this field for quantity? 
                            // The user said "quantity and the likes are almost same". 
                            // I will pass quantity as specific param, and maybe recipient as current user email?
                            quantity: jQuery(".service-number").val(),
                            plan: jQuery(".service-value").val(),
                            value: jQuery("#amt").val(),
                            ref: "<?php echo !empty($runcode) ? $runcode : time(); ?>"
                        };
                        <?php if (($settings["require_pin"] ?? 'no') === "yes"): ?>
                            obj.pin = jQuery(".pin").val();
                        <?php endif; ?>

                        jQuery('.btn-close').trigger('click');
                        doAjax(vtupressobj.apiUrl + obj['action'], obj, "post", "text", function (res) {
                            if (res === "100") {
                                showToast("Successful", "green", 3000, true);
                                setTimeout(function () {
                                    location.reload();
                                }, 1500);
                            } else {
                                try {
                                    let json = JSON.parse(res);
                                    if (json.status === 'success' || json.success === true) {
                                        showToast(json.message || "Successful", "green", 3000, true);
                                        setTimeout(function () {
                                            location.reload();
                                        }, 1500);
                                    } else {
                                        showToast(json.message ? json.message : (json.data.message ? json.data.message : res), "red", 5000, true);
                                    }
                                } catch (e) {
                                    showToast(res, "red", 5000, true);
                                }
                            }
                        }, dprocessCallBack, derrorCallBack, true);
                    });
                </script>
            </div>
        </div>
        </div>
        <?php
        echo ob_get_clean();
    }


    /**
     * Render Fixed Amount Frontend (Data, Cable)
     */
    public function render_fixed_frontend($user, $runcode = "")
    {
        $balance = $user["balance"];
        $service = $this;
        $service->boot();
        $settings = $service->get_settings();
        $enableBonus = ($settings["service_bonus_enable"] ?? "no") === "yes";
        $discountMethod = $settings["discount_method"] ?? "discount";

        $all_providers = $service->providers;
        $service_types = $service->service_types;
        $plans = $service->plans;

        $providerArray = [];
        $active_service_types = [];
        $bonuses = [];

        // Tiered Bonus Logic
        $user_plan = strtolower($user['plan'] ?? 'customer');
        $reward_table = $this->db->prefix . 'vtupress_level_service_rewards';
        $tiered_bonus_val = $this->db->get_var($this->db->prepare(
            "SELECT bonus_value FROM $reward_table WHERE level_name = %s AND service_id = %s AND set_country = %s",
            $user_plan,
            $this->id,
            $service->current_country
        ));
        $tiered_bonus_val = $tiered_bonus_val ?: "0";

        foreach ($all_providers as $prov) {
            $prov = strtolower($prov);
            $providerArray[$prov] = [];

            foreach ($service_types as $stype) {
                $stype = strtolower($stype);
                // Check if this type is globally enabled (Data uses this, Cable might not)
                $s = \Vtupress_Service_Container::get($this->id, $stype);
                $s->boot();
                $sset = $s->get_settings();

                if (($sset["enable"] ?? 'no') == 'yes') {
                    $active_service_types[$stype] = $stype;
                }

                if (($sset["{$prov}_status"] ?? "off") === "on") {
                    $providerArray[$prov][] = $stype;
                    $bonuses["{$prov}_{$stype}"] = $sset["{$prov}_bonus"] ?? 0;
                }
            }
        }

        // Map Plans
        $planMap = [];
        foreach ($plans as $plan) {
            if (empty($plan['network']) || empty($plan['price']))
                continue; // Basic validation
            $net = strtolower($plan['network']);
            $type = strtolower($plan['service_type'] ?? 'subscription');
            $key = $net . "_" . $type;

            $planMap[$key][] = [
                'id' => $plan['id'],
                'name' => $plan['name'],
                'price' => $plan['price'],
                'type' => $type
            ];
        }

        $ports = $service->uses_ports ? $service->ports : false;
        $transaction_settings = $this->db->get_row("SELECT enable_beneficiary FROM {$this->db->prefix}vtupress_transaction_settings WHERE id=1", ARRAY_A);

        $label_provider = $service->label_provider;
        $label_number = $service->label_number;
        $label_type = $service->label_service_type ?? 'Type';
        $label_plan = $service->label_plan ?? 'Plan';
        $has_verification = $service->has_verification;
        $has_types = !empty($active_service_types);

        ob_start();
        ?>
        <style>
            select.form-select,
            input[type='number'],
            input[type='text'] {
                height: 50px !important;
            }

            .purchase-btn {
                height: 65px !important;
            }
        </style>
        <link rel="stylesheet"
            href="<?php echo plugins_url('vtupress-templates/templates/msorg_template'); ?>/msorg_template/form.css"
            media="all">

        <div class="container-md mt-3">
            <div class="service-form p-3" style="border: 1px solid grey; border-radius: 5px;">
                <div class="mb-2">
                    <form class="for" id="cfor" method="post" <?php echo apply_filters('formaction', 'target="_self"'); ?>>

                        <div class="mb-2">
                            <label class="form-label"><?php echo $label_provider; ?></label>
                            <select class="form-select form-select-sm service-provider" name="network">
                                <option value="">- Select <?php echo $label_provider; ?> -</option>
                                <?php foreach ($all_providers as $prov):
                                    if (empty($providerArray[strtolower($prov)]))
                                        continue;
                                    ?>
                                    <option value="<?php echo strtolower($prov); ?>"><?php echo strtoupper($prov); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <div class="invalid-feedback">Error: <span class="provider-error"></span></div>
                        </div>

                        <?php if ($has_types): ?>
                            <div class="mb-2 type-div" style="display:none;">
                                <label class="form-label"><?php echo $label_type; ?></label>
                                <select id="servicechoice" name="servicechoice" class="service-choice form-select form-select-sm">
                                    <option value="">- Select Type -</option>
                                </select>
                            </div>
                        <?php endif; ?>

                        <div class="mb-2 plan-div" style="display:none;">
                            <label class="form-label"><?php echo $label_plan; ?></label>
                            <select class="form-select form-select-sm service-plan" name="cplan">
                                <option value="" price="0">---Select---</option>
                            </select>
                            <div class="invalid-feedback">Error: <span class="plan-error"></span></div>
                        </div>

                        <div class="mb-2">
                            <label class="form-label"><?php echo $label_number; ?></label>
                            <div class="input-group">
                                <input id="phone" name="phone" list="beneficiaries" type="number"
                                    class="form-control service-number" placeholder="<?php echo $label_number; ?>">
                                <?php if ($has_verification): ?>
                                    <button class="btn btn-secondary verify-service" type="button">Verify</button>
                                <?php endif; ?>
                            </div>
                            <datalist id="beneficiaries">
                                <?php
                                $bens = !empty($user["beneficiaries"]) ? explode(",", $user["beneficiaries"]) : [];
                                if (count($bens) >= 1 && ($transaction_settings["enable_beneficiary"] ?? 'no') == "yes") {
                                    foreach ($bens as $ben) {
                                        if (!empty($ben))
                                            echo "<option value='$ben'>";
                                    }
                                }
                                ?>
                            </datalist>
                            <?php if ($has_verification): ?>
                                <small class="text-success verify-name"></small>
                            <?php endif; ?>
                            <div class="invalid-feedback">Error: <span class="number-error"></span>.</div>
                        </div>

                        <div class="mb-2">
                            <label class="form-label">Amount</label>
                            <div class="input-group mb-2">
                                <span class="input-group-text">NGN.</span>
                                <input id="amt" name="amount" type="number" class="form-control service-amount" readonly
                                    placeholder="Amount">
                                <span class="input-group-text">.00</span>
                            </div>
                        </div>

                        <div class="mb-2">
                            <label
                                class="form-label"><?php echo $discountMethod == "commission" ? "Charge Back Bonus" : "Amount To Pay"; ?></label>
                            <div class="input-group mb-2">
                                <span class="input-group-text">NGN.</span>
                                <input id="amttopay" type="number" class="form-control amttopay" max="<?php echo $balance; ?>"
                                    placeholder="Amount To Pay" readonly>
                                <span class="input-group-text">.00</span>
                            </div>
                        </div>

                        <div class="vstack gap-2">
                            <button type="button"
                                class="w-full p-2 text-xs font-bold text-white uppercase bg-indigo-600 rounded shadow purchase-btn btn btn-primary fs-5"
                                data-bs-toggle="modal" data-bs-target="#confirmModal">Purchase</button>
                        </div>
                    </form>

                    <!-- Confirmation Modal -->
                    <div class="modal fade" id="confirmModal" tabindex="-1" aria-hidden="true">
                        <div class="modal-dialog">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title">Confirm Transaction</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                </div>
                                <div class="modal-body">
                                    <?php echo $label_provider; ?> : <span class="confirm-provider"></span><br>
                                    Plan: <span class="confirm-plan"></span><br>
                                    <?php echo $label_number; ?> : <span class="confirm-number"></span> <span
                                        class="confirm-verify-name text-success"></span><br>
                                    Amount: ₦<span class="confirm-amount"></span><br>
                                    To Pay: ₦<span class="amttopay2"></span><br>
                                    <?php echo $enableBonus ? "Discount" : "Commission"; ?> : <span
                                        class="confirm-discount"></span><br>
                                    Status : <span class="confirm-status"></span>
                                    <?php if (($settings["require_pin"] ?? 'no') == "yes"): ?>
                                        <div class="input-group mt-2">
                                            <span class="input-group-text">PIN</span>
                                            <input class="form-control pin" type="number" name="pin">
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn-danger" data-bs-dismiss="modal">Cancel</button>
                                    <button type="button" class="btn-success proceed-btn">Proceed</button>
                                </div>
                            </div>
                        </div>
                    </div>

                    <script>
                        var bonusEnabled = <?php echo $enableBonus ? 'true' : 'false'; ?>;
                        var discountMethod = "<?php echo esc_js($discountMethod); ?>";
                        var networkBonus = [];
                        var providerArray = [];
                        var planMap = <?php echo json_encode($planMap); ?>;
                        var tieredBonus = "<?php echo $tiered_bonus_val; ?>";
                        var hasVerification = <?php echo $has_verification ? 'true' : 'false'; ?>;
                        var isData = <?php echo $this->id == 'data' ? 'true' : 'false'; ?>;
                        var serviceId = "<?php echo $this->id; ?>";

                        <?php foreach ($bonuses as $key => $value): ?>
                            networkBonus["<?php echo strtolower($key); ?>"] = "<?php echo empty($value) ? 0 : $value; ?>";
                        <?php endforeach; ?>

                        <?php foreach ($providerArray as $prov => $types): ?>
                            providerArray["<?php echo strtolower($prov); ?>"] = <?php echo json_encode($types); ?>;
                        <?php endforeach; ?>

                        function calcit() {
                            var planOpt = jQuery(".service-plan option:selected");
                            var original = parseFloat(planOpt.attr("price")) || 0;
                            var provider = jQuery(".service-provider").val();
                            var type = jQuery("#servicechoice").val();

                            // Type auto-select or fallback
                            if (!type && providerArray[provider] && providerArray[provider].length > 0) {
                                // Don't auto-force type here in calculation if we want user to select it
                                // But if we hide type div (not active), we might need it.
                                // If type div is shown, user must select.
                                // Logic: If type is empty, return 0? Or use first available?
                                // Let's keep existing safety but maybe remove it if we enforce selection
                            }

                            var discount = networkBonus[provider + '_' + type] || 0;

                            var bonusAmount = (String(discount).includes('%'))
                                ? (original * parseFloat(discount.replace('%', ''))) / 100
                                : parseFloat(discount);

                            // Tiered Bonus
                            var tieredAmount = 0;
                            if (tieredBonus && tieredBonus !== "0") {
                                if (tieredBonus.includes('%')) {
                                    tieredAmount = (original * parseFloat(tieredBonus.replace('%', ''))) / 100;
                                } else {
                                    tieredAmount = parseFloat(tieredBonus);
                                }
                            }
                            bonusAmount += tieredAmount;

                            var finalAmount = (discountMethod === "discount") ? (original - bonusAmount) : bonusAmount;
                            finalAmount = Math.max(0, finalAmount);

                            jQuery(".service-amount").val(original);
                            jQuery(".amttopay").val(finalAmount);
                            jQuery(".amttopay2").text(finalAmount);
                            jQuery(".confirm-discount").text(bonusAmount.toFixed(2));
                        }

                        jQuery(".service-provider").on("change", function () {
                            var provider = jQuery(this).val();
                            var choiceSelect = jQuery("#servicechoice");
                            var planSelect = jQuery(".service-plan");

                            choiceSelect.empty().append('<option value="">- Select Type -</option>');

                            if (provider && providerArray[provider]) {
                                providerArray[provider].forEach(function (t) {
                                    choiceSelect.append('<option value="' + t + '">' + t.replace(/_/g, " ").toUpperCase() + '</option>');
                                });
                            }
                            jQuery(".type-div").show();
                            jQuery(".plan-div").hide();
                        });

                        jQuery("#servicechoice").on("change", function () {
                            var provider = jQuery(".service-provider").val();
                            var type = jQuery(this).val();
                            var planSelect = jQuery(".service-plan");

                            planSelect.empty().append('<option value="" price="0">---Select---</option>');
                            var key = provider + (type ? "_" + type : "");

                            if (planMap[key]) {
                                planMap[key].forEach(function (p) {
                                    planSelect.append('<option value="' + p.id + '" price="' + p.price + '">' + p.name + '</option>');
                                });
                            }
                            jQuery(".plan-div").show();
                        });

                        jQuery(".service-plan").on("change", calcit);

                        // Verification Logic
                        if (hasVerification) {
                            jQuery(".verify-service").on("click", function () {
                                var btn = jQuery(this);
                                var num = jQuery(".service-number").val();
                                var prov = jQuery(".service-provider").val();

                                if (!prov || !num) { alert("Select Provider and Enter Number"); return; }

                                btn.text("Verifying...");
                                jQuery.post(vtupressInitPluginData.ajax_url, {
                                    action: 'vtupress_verify_service',
                                    service_id: serviceId,
                                    service_type: 'subscription', // Default for Cable
                                    provider: prov,
                                    number: num
                                }, function (resp) {
                                    btn.text("Verify");
                                    if (resp.success) {
                                        var name = resp.data.name || resp.data;
                                        jQuery(".verify-name").text(name);
                                        jQuery(".confirm-verify-name").text(name);
                                    } else {
                                        jQuery(".verify-name").text(resp.data.message || "Verification Failed").removeClass("text-success").addClass("text-danger");
                                    }
                                });
                            });
                        }

                        jQuery(".purchase-btn").on("click", function () {
                            var provider = jQuery(".service-provider").val();
                            var number = jQuery(".service-number").val();
                            var amount = jQuery("#amttopay").val();
                            var plan = jQuery(".service-plan option:selected").text().trim();

                            jQuery(".confirm-provider").text(provider.toUpperCase());
                            jQuery(".confirm-number").text(number);
                            jQuery(".confirm-amount").text(jQuery("#amt").val());
                            jQuery(".confirm-plan").text(plan);

                            var valid = true;
                            if (!provider || !number || !amount || parseFloat(amount) <= 0) valid = false;

                            if (valid) {
                                jQuery(".confirm-status").text("Ready").removeClass("text-danger").addClass("text-success");
                                jQuery(".proceed-btn").show();
                            } else {
                                jQuery(".confirm-status").text("Invalid Details").removeClass("text-success").addClass("text-danger");
                                jQuery(".proceed-btn").hide();
                            }
                        });

                        jQuery(".proceed-btn").click(function () {
                            var obj = {
                                action: serviceId,
                                type: jQuery("#servicechoice").val() || 'subscription',
                                provider: jQuery(".service-provider").val(),
                                recipient: jQuery(".service-number").val(),
                                amount: jQuery("#amt").val(),
                                plan: jQuery(".service-plan").val(), // Uses cplan for Cable, plan for Data?
                                // Airtime uses 'type', Data uses 'plan'. But Data might use 'cplan' in request?
                                // Data::render_frontend used `name="cplan" tab ID="cdat" and JS logic submit`.
                                // Let's check Data payload in constructor: `network`, `recipient`, `amount`, `plan`.
                                // Data uses `plan` key. Cable uses `cplan`?
                                // Let's assume standardisation. The API expects `plan` for data(?).
                                // I'll send `cplan` as `plan` if service is data.
                                // Actually, for Data, the form name was `cplan` but the JS logic might have remapped it?
                                // No, Data::render_frontend didn't have JS submit logic visible in the chunk I saw!
                                // Wait, I saw Airtime JS. Data JS was likely similar.
                                // I'll send both `plan` and `cplan` or check `$this->id`.
                                ref: "<?php echo !empty($runcode) ? $runcode : time(); ?>"
                            };
                            if (isData) obj.plan = jQuery(".service-plan").val();

                            <?php if (($settings["require_pin"] ?? 'no') === "yes"): ?>
                                obj.pin = jQuery(".pin").val();
                            <?php endif; ?>

                            jQuery('.btn-close').trigger('click');
                            doAjax(vtupressobj.apiUrl + obj['action'], obj, "post", "text", function (res) {
                                if (res === "100") {
                                    showToast("Successful", "green", 3000, true);
                                    setTimeout(function () {
                                        location.reload();
                                    }, 1500);
                                } else {
                                    try {
                                        let json = JSON.parse(res);
                                        if (json.status === 'success' || json.success === true) {
                                            showToast(json.message ? json.message : (json.data.message ? json.data.message : "Successful"), "green", 3000, true);
                                            setTimeout(function () {
                                                location.reload();
                                            }, 1500);
                                        } else {
                                            showToast(json.message ? json.message : (json.data.message ? json.data.message : res), "red", 5000, true);
                                        }
                                    } catch (e) {
                                        showToast(res, "red", 5000, true);
                                    }
                                }
                            }, dprocessCallBack, derrorCallBack, true);
                        });

                    </script>
                </div>
            </div>
        </div>
        <?php
        echo ob_get_clean();
    }

    public function boot(): void
    {

        if ($this->booted) {
            return;
        }


        $this->booted = true;
        $this->nonce = wp_create_nonce('vtupress_save_service_' . $this->id . "_{$this->current_country}");

        $this->status = Vtupress_Option::get(
            'vtupress_service_' . $this->id . "_{$this->current_country}" . '_status',
            false
        );


        $this->country_array = Vtupress_Country_Manager::all();

        $country = isset($this->country_array[$this->current_country]) ? $this->country_array[$this->current_country] : false;

        if (!$country) {
            $this->error("Country not found");
        }

        // $this->error(print_r($country,true));

        $this->iso = $country["iso"] ?? '';
        $this->currency = $country["currency"]; //NGN
        $this->currency_symbol = $country["symbol"]; //$
        $this->currency_name = $country["name"];//naira
        $this->country_code = $country["code"];
        // Provider Resolution Strategy
        $default_country_providers = isset($country[$this->id]) ? array_map('strtolower', $country[$this->id]) : [];

        // Allow developers to override default providers dynamically
        $default_country_providers = apply_filters('vtupress_service_providers', $default_country_providers, $this->id, $this->current_country);
        if ($this->allow_custom_providers) {
            // Check for saved custom providers
            $saved_custom = Vtupress_Option::get('vtupress_service_' . $this->id . "_{$this->current_country}" . '_custom_providers', '');
            if (!empty($saved_custom)) {
                // Parse comma-separated string
                $this->providers = array_map('trim', array_map('strtolower', explode(',', $saved_custom)));
            } else {
                // Fallback to class defaults or country defaults
                $this->providers = !empty($this->default_providers) ? $this->default_providers : $default_country_providers;
            }
        } else {
            // Default behavior: Use defaults if set, or country networks (providers)
            $this->providers = !empty($this->default_providers) ? $this->default_providers : $default_country_providers;
        }
        $this->ports = $this->uses_ports ? $country["ports"] ?? [] : [];

        $level_table = $this->db->prefix . 'vtupress_levels';
        $this->level_result = $this->db->get_results(
            "SELECT * FROM $level_table",
            ARRAY_A
        );

        //uses plans
        // uses plans
        if ($this->uses_plans) {

            $this->plans = []; // ensure array

            $table = $this->db->prefix . 'vtupress_servicePlans';

            $plans_result = $this->db->get_results(
                $this->db->prepare(
                    "SELECT * FROM $table WHERE service_name = %s AND set_country = %s",
                    $this->id,
                    $this->current_country
                ),
                ARRAY_A
            );

            foreach ($plans_result as $row) {

                $this->plans[] = [
                    "id" => strtolower("{$row["network"]}_{$row["service_type"]}_{$row["prefix"]}_{$row["surfix"]}_{$row["duration_figure"]}_{$row["duration_time"]}"),
                    "plan_id" => $row["plan_id"],
                    "plan_name" => $row["plan_name"],
                    "network" => strtolower($row["network"]),
                    "service_type" => strtolower($row["service_type"]),
                    "name" => !empty($row["prefix"]) && !empty($row["surfix"])
                        ? ucwords("{$row["network"]} {$row["prefix"]} {$row["surfix"]} {$row["duration_figure"]}{$row["duration_time"]} {$this->currency_symbol}{$row["plan_price"]}")
                        : $row["plan_name"],
                    "price" => (float) $row["plan_price"],
                ];
            }
        }

    }

    private function default_settings(array $fields)
    {

        $this->boot();

        $defaults = [];
        $defaults["service_types"] = $this->service_types;

        foreach ($fields as $k => $field) {

            foreach ($field as $key => $field) {
                if ($key == "opt_description") {
                    continue;
                }
                if (
                    isset($field['render']['value']) &&
                    is_array($field['render']['value'])
                ) {
                    $values = $field['render']['value'] ?? [];

                    $firstKey = array_key_first($values);

                    if ($firstKey !== null) {
                        $defaults[$key] = is_string($firstKey) ? $firstKey : $values[$firstKey];
                    } else {
                        $defaults[$key] = '';
                    }

                } else {
                    $defaults[$key] = '';
                }
            }

        }

        //other raw datas
        //prefeed with networks

        //other raw datas
        //prefeed with providers
        if ($this->uses_provider) {
            foreach ($this->providers as $prov) {
                $defaults["{$prov}_name"] = strtolower($prov);
                $defaults["{$prov}_providerId"] = "";
                $defaults["{$prov}_ussd"] = "";
                $defaults["{$prov}_listOrder"] = "";
                $defaults["{$prov}_pv"] = "";
                $defaults["{$prov}_bonus"] = "";
                $defaults["{$prov}_status"] = "off";
            }
        }


        return $defaults;
    }
    public function maybe_save_settings(): void
    {
        $this->boot();
        $service_type = $this->service_type ? "_" . $this->service_type : '';
        if (
            // empty($_POST['vtupress_service_save' . $service_type]) ||
            empty($_POST['_wpnonce']) ||
            !wp_verify_nonce($_POST['_wpnonce'], 'vtupress_save_service_' . $this->id . "_{$this->current_country}")
        ) {
            return;
        }

        if (!current_user_can('manage_options')) {
            return;
        }


        $settings = [];
        $merge = array_merge($this->global_fields(), $this->form_fields());
        $save_fields = $this->default_settings($merge);

        if (isset($_POST["set_service_types"])) {
            $settings["service_types"] = $_POST["service_types"] ?? [];
            Vtupress_option::update($this->get_option_key(), $settings);

            add_action('vtupress_notices', function () {
                echo '<div class="alert alert-success"><p>' . $this->name . ' service types saved.</p></div>';
            });
            return;
        } elseif (isset($_POST["custom_service_status"])) {
            $to = $_POST["custom_service_status"];
            Vtupress_option::update('vtupress_service_' . $this->id . "_{$this->current_country}" . '_status', $to);
            add_action('vtupress_notices', function () {
                $to = $_POST["custom_service_status"];
                echo '<div class="alert alert-success"><p>' . $this->name . ' service turned ' . $to . '.</p></div>';
            });
            return;
        }

        foreach ($save_fields as $key => $val) {
            if (!isset($_POST[$key]))
                continue;
            $settings[$key] = sanitize_text_field($_POST[$key] ?? '');
        }



        Vtupress_option::update($this->get_option_key(), $settings);

        $service_status = $_POST[$this->id . "_{$this->current_country}" . "_status"] ?? false;
        Vtupress_option::update('vtupress_service_' . $this->id . "_{$this->current_country}" . '_status', $service_status);

        if (isset($_POST['custom_providers'])) {
            Vtupress_option::update('vtupress_service_' . $this->id . "_{$this->current_country}" . '_custom_providers', sanitize_text_field($_POST['custom_providers']));
        }

        if ($this->uses_plans && !empty($_POST['plans'])) {
            $this->syncPlans($_POST['plans']);
        }


        add_action('vtupress_notices', function () {
            echo '<div class="alert alert-success"><p>Service settings saved.</p></div>';
        });
    }

    protected function syncPlans(array $rawPlans): void
    {
        if (!$this->uses_plans) {
            return;
        }

        $table = $this->db->prefix . 'vtupress_servicePlans';
        $serviceType = $this->service_type ?? '';

        /**
         * --------------------------------------
         * 1. Normalize plans (LAST plan_id wins)
         * --------------------------------------
         */
        $normalized = [];

        foreach ($rawPlans as $plan) {

            if (empty($plan['plan_id'])) {
                continue;
            }


            if (empty($plan['prefix']) || empty($plan['surfix'])) {
                [$prefix, $surfix, $planName] = array_pad(
                    array_map('trim', explode(' ', $plan['plan_name'], 3)),
                    3,
                    null
                );
                $plan['prefix'] = strtolower($prefix);
                $plan['surfix'] = strtolower($surfix);
            }

            $planId = trim($plan['plan_id']);

            $normalized[$planId] = [
                'plan_id' => $planId,
                'plan_name' => sanitize_text_field($plan['plan_name'] ?? ''),
                'plan_price' => floatval($plan['plan_price'] ?? 0),
                'listOrder' => intval($plan['listOrder'] ?? 0),
                'prefix' => sanitize_text_field($plan['prefix'] ?? ''),
                'surfix' => sanitize_text_field($plan['surfix'] ?? ''),
                'duration_figure' => intval($plan['duration_figure'] ?? 0),
                'duration_time' => sanitize_text_field($plan['duration_time'] ?? ''),
                'network' => sanitize_text_field($plan['network'] ?? ''),
            ];
        }

        if (empty($normalized)) {
            return;
        }


        $postedPlans = array_values($normalized);
        $postedPlanIds = array_keys($normalized);

        /**
         * --------------------------------------
         * 2. Fetch existing plan_ids from DB
         * --------------------------------------
         */
        $dbPlans = $this->db->get_col(
            $this->db->prepare(
                "SELECT plan_id FROM $table 
                WHERE service_name = %s AND service_type = %s AND set_country = %s",
                $this->id,
                $serviceType,
                $this->current_country
            )
        );

        /**
         * --------------------------------------
         * 3. Find removed plans and delete them
         * --------------------------------------
         */
        $toDelete = array_diff($dbPlans, $postedPlanIds);

        $this->db->query('START TRANSACTION');

        try {

            if (!empty($toDelete)) {

                $placeholders = implode(',', array_fill(0, count($toDelete), '%s'));

                $this->db->query(
                    $this->db->prepare(
                        "DELETE FROM $table 
                        WHERE service_name = %s 
                        AND service_type = %s 
                        AND set_country = '{$this->current_country}'
                        AND plan_id IN ($placeholders)",
                        array_merge([$this->id, $serviceType], $toDelete)
                    )
                );
            }

            /**
             * --------------------------------------
             * 4. UPSERT plans
             * --------------------------------------
             */

            foreach ($postedPlans as $plan) {

                $this->db->query(
                    $this->db->prepare(
                        "INSERT INTO $table
                        (
                            service_name,
                            service_type,
                            plan_id,
                            plan_name,
                            plan_price,
                            listOrder,
                            prefix,
                            surfix,
                            duration_figure,
                            duration_time,
                            network,
                            set_country
                        )
                        VALUES (%s,%s,%s,%s,%s,%d,%s,%s,%d,%s,%s,%s)
                        ON DUPLICATE KEY UPDATE
                            plan_name       = VALUES(plan_name),
                            plan_price      = VALUES(plan_price),
                            listOrder       = VALUES(listOrder),
                            prefix          = VALUES(prefix),
                            surfix          = VALUES(surfix),
                            duration_figure = VALUES(duration_figure),
                            duration_time   = VALUES(duration_time),
                            network         = VALUES(network),
                            set_country     = VALUES(set_country)",
                        $this->id,
                        $serviceType,
                        $plan['plan_id'],
                        $plan['plan_name'],
                        $plan['plan_price'],
                        $plan['listOrder'],
                        $plan['prefix'],
                        $plan['surfix'],
                        $plan['duration_figure'],
                        $plan['duration_time'],
                        $plan['network'],
                        $this->current_country
                    )
                );
            }

            $this->db->query('COMMIT');

        } catch (\Throwable $e) {

            $this->db->query('ROLLBACK');

            error_log('VTUPRESS PLAN SYNC ERROR: ' . $e->getMessage());
        }
    }

    private function global_fields()
    {
        global $wpdb;
        $settings = $this->get_settings(true, false);

        if ($this->disable_global) {
            return [];
        }

        $defaults = [];
        $plan = [];
        $plan[] = 'all';
        foreach ($this->level_result as $val) {
            $plan[] = $val["name"];
        }
        $defaults["Control"] = [
            "opt_description" => "Basic service control",
            "package_plan" => [
                "label" => "Package",
                "placeholder" => "",
                "description" => "",
                "render" => [
                    "value" => $plan
                ]
            ]
        ];

        if ($this->allow_custom_providers && !$this->service_type) {
            $defaults["Control"] = array_merge($defaults["Control"], [
                "custom_providers" => [
                    "label" => "Manage Providers (Comma Separated)",
                    "placeholder" => "dstv, gotv, startimes",
                    "description" => "Override default providers. Leave empty to use defaults.",
                    "render" => [
                        "value" => implode(', ', $this->providers)
                    ]
                ]
            ]);
        }

        if ($this->service_type):

            $defaults["Control"] = array_merge($defaults["Control"], [
                "enable" => [
                    "label" => "Enable service type",
                    "placeholder" => "",
                    "description" => "",
                    "render" => [
                        "value" => ['no', 'yes']
                    ]
                ]
            ]);

        else:
            $defaults["Control"] = array_merge(
                $defaults["Control"],
                [
                    "discount_method" => [
                        "label" => "Discount Method",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ["discount", "commission"]
                        ]
                    ]
                ],
                [
                    "require_pin" => [
                        "label" => "Require Pin",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ["yes", "no"]
                        ]
                    ]
                ],
                [
                    "service_bonus_enable" => [
                        "label" => "Enable Service Bonus",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ["yes", "no"]
                        ]
                    ]
                ]
            );

            $service_types = $settings["service_types"] ?? [];
            foreach ($service_types as $types) {
                $defaults["Control"] = array_merge($defaults["Control"], [
                    "{$types}_bonus" => [
                        "label" => "$types Bonus",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ""
                        ]
                    ]
                ]);
            }

        endif;
        if (empty($settings["service_types"]) || $this->service_type):
            if ($this->uses_api && $this->uses_api_setup):
                $defaults["Basic Api Setup"] = [
                    "opt_description" => "Fundamental api configuration",
                    "base_url" => [
                        "label" => "Base url",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "end_point" => [
                        "label" => "Endpoint",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "request_method" => [
                        "label" => "Request method",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ["post", "get"]
                        ]
                    ],
                    "response_method" => [
                        "label" => "Response method",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ["json", "plain"]
                        ]
                    ],
                    "request_id_attr" => [
                        "label" => "Request id Key",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => "request_id"
                        ]
                    ],
                    "response_id_key" => [
                        "label" => "Response id key",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => "id"
                        ]
                    ],
                    "authentication_method" => [
                        "label" => "Auth. method",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => ["bearer_token", "basic_authentication"]
                        ]
                    ],
                    "authentication_key" => [
                        "label" => "Auth. key",
                        "placeholder" => "",
                        "description" => "",
                        "render" => [
                            "value" => "Token"
                        ]
                    ],
                    "authentication_value" => [
                        "label" => "Auth. value",
                        "placeholder" => "",
                        "description" => "",
                        "class" => "bg-primary text-white",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "response_keys" => [
                        "label" => "Success response code",
                        "placeholder" => "",
                        "description" => "response keys and values contatenated by : and separated by comma e.g status:true,status:successful",
                        "render" => [
                            "value" => "status:true,status:successful"
                        ]
                    ],
                    "retrieve_keys" => [
                        "label" => "Retrieve keys",
                        "placeholder" => "",
                        "description" => "Keys you want to capture as key:value separated by comma e.g token:api_token where api_token a key from your api and token is a unique referenc you give",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "recipient_key" => [
                        "label" => "Recipient Key",
                        "placeholder" => "",
                        "description" => "this key would hold the value from payload and passed into the post body ",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "amount_key" => [
                        "label" => "Amount Key",
                        "placeholder" => "",
                        "description" => "this key would hold the value from payload and passed into the post body ",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "charge_key" => [
                        "label" => "Charge Key",
                        "placeholder" => "",
                        "description" => "this key would be used to calculate your profit",
                        "render" => [
                            "value" => ""
                        ]
                    ],

                ];
                if ($this->uses_provider) {
                    $defaults["Basic Api Setup"] = array_merge($defaults["Basic Api Setup"], [
                        "provider_key" => [
                            "label" => "Provider Key",
                            "placeholder" => "",
                            "description" => "this provider key would hold the value from payload and passed into the post body ",
                            "render" => [
                                "value" => ""
                            ]
                        ],
                    ]);
                }
                if ($this->uses_plans) {
                    $defaults["Basic Api Setup"] = array_merge($defaults["Basic Api Setup"], [
                        "plan_key" => [
                            "label" => "Plan Key",
                            "placeholder" => "",
                            "description" => "this plan key would hold the value from payload and passed into the post body ",
                            "render" => [
                                "value" => ""
                            ]
                        ],
                    ]);
                }
                $defaults["Headers"] = [
                    "opt_description" => "Headers to append to the payload",
                    "header_1" => [
                        "label" => "Header 1",
                        "placeholder" => "",
                        "description" => "header key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => "content-type:json"
                        ]
                    ],
                    "header_2" => [
                        "label" => "Header 2",
                        "placeholder" => "",
                        "description" => "header key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "header_3" => [
                        "label" => "Header 3",
                        "placeholder" => "",
                        "description" => "header key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "header_4" => [
                        "label" => "Header 4",
                        "placeholder" => "",
                        "description" => "header key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "header_5" => [
                        "label" => "Header 5",
                        "placeholder" => "",
                        "description" => "header key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ]
                ];
                $defaults["Custom Datas"] = [
                    "opt_description" => "Append datas to the payload",
                    "post_data_1" => [
                        "label" => "Post Data 1",
                        "placeholder" => "",
                        "description" => "Data key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => "content-type:json"
                        ]
                    ],
                    "post_data_2" => [
                        "label" => "Post Data 2",
                        "placeholder" => "",
                        "description" => "Data key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "post_data_3" => [
                        "label" => "Post Data 3",
                        "placeholder" => "",
                        "description" => "Data key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "post_data_4" => [
                        "label" => "Post Data 4",
                        "placeholder" => "",
                        "description" => "Data key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ],
                    "post_data_5" => [
                        "label" => "Post Data 5",
                        "placeholder" => "",
                        "description" => "Data key and value concatenated by : e.g content-type:json",
                        "render" => [
                            "value" => ""
                        ]
                    ]
                ];
            endif;
        endif;



        return $defaults;

    }
    protected function formData(array $form = []): void
    {

        $defaults = $this->global_fields();
        $form = $this->form_fields();

        $form = array_merge($defaults, $form);
        $this->render($form);
    }
    private function render(array $form): void
    {
        $settings = $this->get_settings(false);
        $status = Vtupress_Option::get(
            'vtupress_service_' . $this->id . "_{$this->current_country}" . '_status',
            false
        );

        ?>
        <form method="post" class="container-fluid container-md servicesDiv">

            <div class="config-box bg bg-white shadow px-3 py-4 m-md-2 rounded">
                <!-- Service Naming-->
                <div class="row px-1 px-md-3">
                    <div class="col d-flex justify-content-between ">
                        <h4>
                            <a href="?page=vtupanel&adminpage=gateway&service=<?php echo $this->id; ?>"
                                class="text-decoration-none text-dark">
                                <?php echo ucfirst(str_replace("_", " ", $this->name)); ?>
                                <span class="imgCover d-inline-block"><img for="all" src="<?php echo $this->image; ?>" alt=""
                                        class="" width="50" height="50"></span>
                            </a>

                            <label class="ms-4 switch">
                                <input type="checkbox" <?php echo ($status) ? "checked" : ""; ?> for="service_name"
                                    name="<?php echo $this->id . "_{$this->current_country}"; ?>_status">
                                <span class="slider"></span>
                            </label>
                        </h4>
                        <?php if ($this->service_type): ?>
                            <h4>
                                <?php echo ucwords($this->service_type); ?>
                            </h4>
                        <?php endif; ?>
                    </div>
                </div>
                <!-- HERE-->
                <?php

                $access_importer = Vtupress_option::get("vtupress_access_importer");

                if ($access_importer == "yes"):
                    $cross_import_status = Vtupress_Option::get('enable_cross_import', 'no');
                    // Fetch available providers from Ultimate Importer API
                    $api_url = 'https://vendor.vtupress.com/wp-json/ultimate-importer/v1/services';
                    $api_params = [
                        'country' => $this->current_country,
                        'service-name' => $this->id
                    ];
                    if ($cross_import_status != 'yes') {
                        $api_params["service-type"] = $this->service_type;
                    }

                    $args = [
                        'timeout' => 120
                    ];

                    $response = wp_remote_get(add_query_arg($api_params, $api_url), $args);
                    $show_import_button = false;
                    $available_providers = [];

                    if (!is_wp_error($response)) {
                        $body = wp_remote_retrieve_body($response);
                        $data = json_decode($body, true);

                        if (isset($data['success']) && $data['success'] && !empty($data['data'])) {
                            $show_import_button = true;
                            $available_providers = $data['data'];
                        }
                    }else{?>
                        <div class="row my-3 px-3">
                            <div class="col-12">
                                <div class="alert alert-info"><?= $response->get_error_message();?></div>
                            </div>
                        </div>
                    <?php }
                    if ($show_import_button): ?>
                        <div class="row my-3 px-3">
                            <div class="col-12">
                                <button type="button" class="btn btn-primary btn-sm" id="showImportModal">
                                    <i class="fa fa-download"></i> Import API Configuration
                                </button>
                            </div>
                        </div>
                    <?php endif; ?>
                    <div class="row my-2 px-3">
                        <div class="col-12 d-flex align-items-center">
                            <label class="switch me-4">
                                <input type="checkbox" id="cross_import_toggle" <?php echo ($cross_import_status == 'yes') ? 'checked' : ''; ?>>
                                <span class="slider round"></span>
                            </label>
                            <strong>Enable Cross Import</strong>
                            <small class="ms-2 text-muted">(Allow importing plans from different service types)</small>
                        </div>
                    </div>

                    <!-- <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.all.min.js"></script> -->
                    <script>
                        document.addEventListener('DOMContentLoaded', function () {
                            jQuery('#cross_import_toggle').change(function () {
                                var isChecked = jQuery(this).is(':checked');
                                var status = isChecked ? 'yes' : 'no';
                                var toggle = jQuery(this);

                                // Revert check immediately until confirmed
                                toggle.prop('checked', !isChecked);

                                Swal.fire({
                                    title: 'Are you sure?',
                                    text: isChecked ? "Do you want to ENABLE cross import for other service types?" : "Do you want to DISABLE cross import capability?",
                                    icon: 'warning',
                                    showCancelButton: true,
                                    confirmButtonColor: '#3085d6',
                                    cancelButtonColor: '#d33',
                                    confirmButtonText: 'Yes, update it!'
                                }).then((result) => {
                                    if (result.isConfirmed) {
                                        Swal.showLoading();
                                        jQuery.post(vtupressInitPluginData.ajax_url, {
                                            action: 'vtupress_update_cross_import',
                                            status: status
                                        }, function (response) {
                                            if (response.success) {
                                                toggle.prop('checked', isChecked);
                                                Swal.fire(
                                                    'Updated!',
                                                    'Cross import setting has been updated.',
                                                    'success'
                                                ).then(() => {
                                                    location.reload();
                                                })
                                            } else {
                                                Swal.fire(
                                                    'Error!',
                                                    response.data.message || 'Something went wrong',
                                                    'error'
                                                )
                                            }
                                        });
                                    }
                                });
                            });
                        });
                    </script>
                <?php endif; ?>
                <div class="row">
                    <!-- <form method="post"> -->
                    <div class="col-12 my-3">
                        <?php do_action('vtupress_notices'); ?>
                        <?php wp_nonce_field('vtupress_save_service_' . $this->id . "_{$this->current_country}"); ?>
                        <?php
                        $service_type = $this->service_type ? "_" . $this->service_type : '';
                        ?>
                        <input type="hidden" name="vtupress_service_save<?php echo $service_type; ?>" value="1">
                    </div>

                    <div class="col">
                        <div class="accordion accordion-flush" id="accordionFlushExample">
                            <div class="accordion-item">

                                <?php

                                foreach ($form as $keys => $values) {
                                    $accordion_id = preg_replace("/\s*|_|-/", "", $keys);
                                    $keys = ucwords($keys);
                                    ?>
                                    <h2 class="accordion-header">
                                        <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse"
                                            data-bs-target="#flush-collapse<?php echo $accordion_id; ?>" aria-expanded="false"
                                            aria-controls="flush-collapse<?php echo $accordion_id; ?>">
                                            <label for="" class="section-core fs-6 fw-bold"><?php echo $keys; ?>:</label>
                                        </button>
                                    </h2>
                                    <div id="flush-collapse<?php echo $accordion_id; ?>" class="accordion-collapse collapse"
                                        data-bs-parent="#accordionFlushExample">
                                        <div class="accordion-body">
                                            <div class="row">
                                                <?php


                                                foreach ($values as $key => $value) {
                                                    if ($key == "opt_description") {
                                                        ?>
                                                        <div class="col-12 p-2">
                                                            <span><?php echo $value; ?></span>
                                                        </div>

                                                        <?php
                                                        continue;
                                                    }

                                                    $show = $value["show"] ?? true;
                                                    if (!$show) {
                                                        continue;
                                                    }
                                                    $column = esc_html($value["column"] ?? "4");
                                                    $class = esc_html($value["class"] ?? "");
                                                    $inner_class = is_array($value["render"] ?? null) ? esc_html($value["render"]["class"] ?? "") : "";
                                                    ?>
                                                    <div class="col-6 col-md-<?php echo $column . " " . $class; ?> my-3">
                                                        <label
                                                            for="<?php echo esc_html($value['label'] ?? ''); ?>"><?php echo esc_html($value['label'] ?? ''); ?>:</label>
                                                        <?php
                                                        //is_bool($value["render"]["value"])
                                                        $render_config = $value["render"] ?? [];
                                                        $render_value = is_array($render_config) ? ($render_config["value"] ?? null) : null;

                                                        if (is_array($render_value)) {
                                                            ?>
                                                            <select name="<?php echo $key; ?>"
                                                                class="form-control <?php echo $inner_class; ?>">
                                                                <?php foreach ($render_value as $key2 => $value2) {
                                                                    $option_value = is_numeric($key2) ? $value2 : $key2;
                                                                    $selected = ($settings[$key] ?? '') === $option_value ? 'selected' : '';
                                                                    ?>
                                                                    <option value="<?php echo esc_attr($option_value); ?>" <?php echo $selected; ?>>
                                                                        <?php echo esc_html($value2); ?>
                                                                    </option>

                                                                <?php } ?>
                                                            </select>

                                                            <?php
                                                        } else {
                                                            $type = esc_html($value["render"]["type"] ?? 'text');
                                                            $field_value = esc_html($value["render"]["value"] ?? '');
                                                            ?>
                                                            <input class="form-control  <?php echo $inner_class; ?>"
                                                                id="<?php echo $key; ?>" type="<?php echo $type; ?>"
                                                                name="<?php echo $key; ?>"
                                                                value="<?php echo $settings[$key] ?? $field_value; ?>">
                                                            <?php
                                                        }
                                                        ?>

                                                        <small><?php echo esc_html($value["description"] ?? ''); ?></small>
                                                    </div>

                                                    <?php
                                                }

                                                ?>
                                            </div>
                                        </div>
                                    </div>
                                    <?php


                                }
                                $this->render_providers();
                                $this->render_plans();
                                $this->render_js();
                                ?>
                                <?php submit_button('Save Gateway Settings'); ?>
                                <!-- </form> -->
                            </div>
                        </div>
                    </div>

                </div>
            </div>
        </form>
        <!-- Import Provider Modal -->
        <?php if ($show_import_button): ?>
            <div class="modal fade" id="importProviderModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Select API Provider</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <input type="text" class="form-control mb-3" id="providerSearch" placeholder="🔍 Search...">
                            <div id="providerList" class="row">
                                <?php foreach ($available_providers as $name => $details): ?>
                                    <div class="col-md-6 provider-item mb-2"
                                        data-provider-name="<?php echo esc_attr(strtolower($name)); ?>">
                                        <div class="card">
                                            <div class="card-body">
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="import_provider"
                                                        value="<?php echo esc_attr($name); ?>"
                                                        data-config='<?php echo htmlspecialchars(json_encode($details), ENT_QUOTES, 'UTF-8'); ?>'
                                                        id="prov_<?php echo esc_attr(str_replace(' ', '_', $name)); ?>">
                                                    <label class="w-100"
                                                        for="prov_<?php echo esc_attr(str_replace(' ', '_', $name)); ?>">
                                                        <strong>
                                                            <?php echo esc_html($name); ?>
                                                        </strong> <span class="text-warning">
                                                            <?php echo str_repeat('⭐', intval($details['stars'] ?? 0)); ?>
                                                        </span>
                                                        <small class="d-block text-muted">
                                                            <?php echo esc_html($details['info'] ?? ''); ?>
                                                        </small>
                                                    </label>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="button" class="btn btn-primary" id="importSelectedProvider">Import</button>
                        </div>
                    </div>
                </div>
            </div>
            <script>
                jQuery(document).ready(function ($) {
                    $('#showImportModal').click(function () { jQuery('#importProviderModal').modal('show'); });
                    $('#providerSearch').on('keyup', function () {
                        var term = $(this).val().toLowerCase();
                        $('.provider-item').each(function () {
                            $(this).toggle($(this).data('provider-name').includes(term));
                        });
                    });
                    // Allow clicking on the card/row to select the radio
                    $('.provider-item').click(function (e) {
                        if (!$(e.target).is('input')) {
                            $(this).find('input[type="radio"]').prop('checked', true).trigger('change');
                        }
                    });
                    $('#importSelectedProvider').click(function () {
                        var selected = $('input[name="import_provider"]:checked');
                        var prov = selected.val();

                        if (!prov) {
                            Swal.fire('Error', 'Please select a provider to import', 'error');
                            return;
                        }

                        var config = selected.data('config');
                        // alert(config);
                        // console.log(config);

                        if (config) {
                            var count = 0;

                            // 1. Uncheck all providers first
                            $('input[for="provider"]').prop('checked', false).trigger('change');

                            // 2. Handle provider_map
                            if (config.provider_map) {
                                try {
                                    var map = (typeof config.provider_map === 'string') ? JSON.parse(config.provider_map) : config.provider_map;
                                    $.each(map, function (key, val) {
                                        // Enable status (e.g. mtn_status)
                                        var statusInput = $('input[name="' + key + '_status"]');
                                        if (statusInput.length) {
                                            statusInput.prop('checked', true).trigger('change');
                                        }
                                        // Set providerId (e.g. mtn_providerId)
                                        var idInput = $('input[name="' + key + '_providerId"]');
                                        if (idInput.length) {
                                            idInput.val(val).trigger('change');
                                        }
                                    });
                                } catch (e) { console.error("Error parsing provider_map", e); }
                            }

                            // 3. Handle extra
                            if (config.extra) {
                                try {
                                    var extra = (typeof config.extra === 'string') ? JSON.parse(config.extra) : config.extra;
                                    $.each(extra, function (exKey, exVal) {
                                        var extraInput = $('[name="' + exKey + '"]');
                                        if (extraInput.length) {
                                            extraInput.val(exVal).trigger('change');
                                            count++;
                                        }
                                    });
                                } catch (e) { console.error("Error parsing extra", e); }
                            }

                            // 4. Handle standard fields
                            $.each(config, function (key, value) {
                                if (key === 'provider_map' || key === 'extra') return; // Skip explicitly handled

                                var input = $('[name="' + key + '"]');
                                if (input.length > 0) {
                                    // Handle standard inputs and selects
                                    if (!input.is(':checkbox') && !input.is(':radio')) {
                                        input.val(value).trigger('change');
                                        count++;
                                    }
                                }
                            });

                            Swal.fire({
                                title: 'Imported!',
                                text: 'Successfully populated ' + count + ' fields from ' + prov + ' kindly save or complete setup before leaving',
                                icon: 'success',
                                timer: 2000,
                                showConfirmButton: false
                            });

                            jQuery('#importProviderModal').modal('hide');
                        } else {
                            Swal.fire('Error', 'No configuration data available for this provider', 'error');
                        }
                    });
                });
            </script>
        <?php endif; ?>
    <?php
    }

    protected function render_providers(): void
    {
        if (!$this->uses_provider || !empty($this->service_types) && !$this->service_type) {
            return;
        }
        $providers = $this->providers;
        ?>
        <h2 class="accordion-header">
            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse"
                data-bs-target="#flush-collapseProviders" aria-expanded="false" aria-controls="flush-collapseProviders">
                <label for="" class="section-core fs-6 fw-bold">Providers:</label>
            </button>
        </h2>

        <div id="flush-collapseProviders" class="accordion-collapse collapse" data-bs-parent="#accordionFlushExample">
            <div class="accordion-body">

                <div class="row">
                    <?php $id_icon_num = 0;
                    foreach ($providers as $prov): ?>
                        <?php $this->render_single_provider($prov, $id_icon_num); ?>
                        <?php $id_icon_num++; endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }

    protected function render_single_provider($prov, $id_icon_num): void
    {
        $settings = $this->get_settings();
        $name = $settings["{$prov}_name"];
        $status = $settings["{$prov}_status"];
        $ussd = $settings["{$prov}_ussd"];
        $order = $settings["{$prov}_listOrder"];
        $pv = $settings["{$prov}_pv"];
        $bonus = $settings["{$prov}_bonus"];
        $providerId = $settings["{$prov}_providerId"];
        ?>
        <div class="col col-md-3 my-" for="<?php echo $name; ?>">

            <div class="networkStatus d-flex flex-column align-items-center justify-content-center">
                <div class="d-flex align-items-center justify-content-center">
                    <span class="imgCover "><img for="<?php echo $name; ?>" src="<?php echo get_images($prov); ?>" alt=""
                            width="30" height="30"></span>
                    <div class="slider-div  d-flex align-items-center">
                        <label class="ms-4 switch">
                            <input type="checkbox" for="provider" name="<?php echo $prov . "_status"; ?>" <?php echo ($status == "on") ? "checked" : ""; ?>>
                            <span class="slider"></span>
                        </label>
                    </div>
                </div>
                <div class="text-center fw-bold mt-2 text-capitalize"><?php echo $name; ?></div>
            </div>
            <?php if (!$this->is_voucher): ?>
                <div class="input-group my-2 ">
                    <span class="input-group-text">
                        list Order
                    </span>
                    <input type="text" placeholder="listing order" class="form-control" name="<?php echo $prov; ?>_listOrder"
                        value="<?php echo $order; ?>">
                </div>
                <div class="input-group my-2">
                    <span class="input-group-text">
                        <i class="fa-solid fa-<?php echo $id_icon_num; ?>"></i>
                    </span>
                    <input type="text" placeholder="Provider id from provider" class="form-control"
                        name="<?php echo $prov; ?>_providerId" value="<?php echo $providerId; ?>">

                </div>
                <div class="input-group my-2">
                    <span class="input-group-text">
                        <i class="fa-solid fa-hashtag"></i>
                    </span>
                    <input type="text" class="form-control" placeholder="Ussd" name="<?php echo $prov; ?>_ussd"
                        value="<?php echo $ussd; ?>">
                </div>
            <?php endif; ?>
            <div class="input-group my-2">
                <span class="input-group-text">
                    Bonus
                </span>
                <input type="text" class="form-control" placeholder="Bonus" name="<?php echo $prov; ?>_bonus"
                    value="<?php echo $bonus; ?>">
                <small>e.g 2% or 2 for flat price. Dont worry, different plans will get it @ their bonus multiplier
                    price</small>
            </div>
            <div class="input-group my-2">
                <span class="input-group-text">
                    Pv
                </span>
                <input type="text" class="form-control" placeholder="Pv" name="<?php echo $prov; ?>_pv"
                    value="<?php echo $pv; ?>">
            </div>
        </div>

        <?php
    }

    protected function render_plans()
    {
        if (!$this->uses_api_setup) {
            return;
        }
        if (!$this->uses_plans || !empty($this->service_types) && !$this->service_type) {
            return;
        }
        global $wpdb;
        $and = $this->service_type ? "  AND service_type = '{$this->service_type}'  " : '';
        $totalPlans = false;
        $vtupress_servicePlan = $this->db->prefix . 'vtupress_servicePlans';
        $servicePlan_table = $this->db->get_results("SELECT * FROM $vtupress_servicePlan WHERE service_name = '{$this->id}' AND set_country = '{$this->current_country}' $and ORDER BY network ", ARRAY_A);
        if (count($servicePlan_table) >= 1) {
            $totalPlans = true;
        }
        $networks = $this->providers;
        $settings = $this->get_settings();

        ?>
        <h2 class="accordion-header">
            <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse"
                data-bs-target="#flush-collapseFive" aria-expanded="false" aria-controls="flush-collapseFive">
                <label for="" class="section-core fs-6 fw-bold">Plans & Price:</label>
            </button>
        </h2>
        <div id="flush-collapseFive" class="accordion-collapse collapse" data-bs-parent="#accordionFlushExample">
            <div class="accordion-body">

                <div class="row">
                    <div class="col-12 ">
                        <?php $plan_name = (!$totalPlans) ? "No Plans Found" : "Plans & price"; ?>
                        <?php $plan_message = (!$totalPlans) ? "Please add plans to save configuration" : ""; ?>

                        <div class="card">
                            <div class="card-head">
                                <h5 class="card-title plan-title"><?php echo $plan_name; ?></h5>
                            </div>
                            <div class="card-body">
                                <p class="plan-message"><?php echo $plan_message; ?></p>
                                <div class="networksOptDiv row my-2">
                                    <?php if ($this->uses_provider): ?>
                                        <div class="col-1 d-md-none"></div>
                                        <div class="col-2 col-md d-flex flex-column align-items-center justify-content-center">
                                            <!--<span for="all" class="btn btn-sm btn-info showNetwork">All</span>-->
                                            <span class="imgCover "><img for="all"
                                                    src="<?php echo get_images("data", "service"); ?>" alt="" class="showNetwork"
                                                    width="50" height="50"></span>
                                            <div class="text-center fw-bold mt-2 text-capitalize">All</div>
                                        </div>
                                        <?php foreach ($networks as $net):
                                            $net = strtolower($net);
                                            $name = $settings["{$net}_name"];
                                            $networkId = $settings["{$net}_providerId"];

                                            ?>
                                            <div class="col-2 col-md d-flex flex-column align-items-center justify-content-center">

                                                <span class="imgCover "><img for="<?php echo $net; ?>"
                                                        src="<?php echo get_images($net); ?>" alt="" class="showNetwork" width="50"
                                                        height="50"></span>
                                                <div class="text-center fw-bold mt-2 text-capitalize"><?php echo $name; ?></div>
                                            </div>
                                        <?php endforeach; ?>
                                        <div class="col-1 d-md-none"></div>
                                    <?php endif; ?>
                                </div>
                                <?php if ($totalPlans):
                                    if ($this->uses_provider):
                                        $col = "col-6 col-md-4";
                                    else:
                                        $col = "col-4 col-md-3";

                                    endif;
                                    $i = 1;
                                    foreach ($servicePlan_table as $thisPlan): ?>
                                        <div class="thePlan border rounded py-2 px-1 my-2 row my-3"
                                            for="<?php echo $thisPlan["network"]; ?>">
                                            <div class="closeThis-div d-flex justify-content-end">
                                                <i class="fa-regular closeThis fa-circle-xmark text-danger" for="<?php echo $i; ?>">
                                                </i>
                                            </div>
                                            <div class="<?php echo $col; ?>">
                                                <label>Order</label>
                                                <input type="text" class="form-control" name="plans[<?php echo $i; ?>][listOrder]"
                                                    value="<?php echo add_service_value("listOrder", $thisPlan); ?>">
                                            </div>
                                            <div class="<?php echo $col; ?>">
                                                <label>Plan Id</label>
                                                <input type="text" class="form-control" name="plans[<?php echo $i; ?>][plan_id]"
                                                    value="<?php echo add_service_value("plan_id", $thisPlan); ?>">
                                            </div>
                                            <?php if ($this->id == "data"): ?>
                                                <div class="<?php echo $col; ?> col-12">
                                                    <label>Data Bundle</label>
                                                    <div class="d-flex">
                                                        <input type="number" step="0.01" min="0" class="form-control"
                                                            name="plans[<?php echo $i; ?>][prefix]" placeholder="e.g 200"
                                                            value="<?php echo add_service_value("prefix", $thisPlan); ?>">
                                                        <select class="surfix" name="plans[<?php echo $i; ?>][surfix]">
                                                            <?php echo get_select_options(['mb', 'gb'], add_service_value("surfix", $thisPlan)); ?>
                                                        </select>
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                                <div class="<?php echo $col; ?>">
                                                    <label>Plan Name</label>
                                                    <input type="text" class="form-control" name="plans[<?php echo $i; ?>][plan_name]"
                                                        value="<?php echo add_service_value("plan_name", $thisPlan); ?>">
                                                </div>
                                            <?php endif; ?>

                                            <div class="<?php echo $col; ?>">
                                                <label>Plan Price</label>
                                                <input type="text" class="form-control" name="plans[<?php echo $i; ?>][plan_price]"
                                                    value="<?php echo add_service_value("plan_price", $thisPlan); ?>">
                                            </div>
                                            <div class="<?php echo $col; ?>">
                                                <label>Duration</label>
                                                <div class="d-flex">
                                                    <input type="text" class="form-control"
                                                        name="plans[<?php echo $i; ?>][duration_figure]"
                                                        value="<?php echo add_service_value("duration_figure", $thisPlan); ?>">
                                                    <select class="form-control" name="plans[<?php echo $i; ?>][duration_time]">
                                                        <?php
                                                        $alldurations = [
                                                            "days",
                                                            "weeks",
                                                            "months",
                                                            "year"
                                                        ];
                                                        echo get_select_options($alldurations, add_service_value("duration_time", $thisPlan));
                                                        ?>
                                                    </select>
                                                </div>
                                            </div>
                                            <?php if ($this->uses_provider): ?>
                                                <div class="<?php echo $col; ?>">
                                                    <label>Provider</label>
                                                    <select class="form-control theNetwork" name="plans[<?php echo $i; ?>][network]">
                                                        <?php $allNetworks = [];
                                                        foreach ($networks as $net):
                                                            $allNetworks[] = strtolower($net);
                                                        endforeach;
                                                        echo get_select_options($allNetworks, add_service_value("network", $thisPlan)); ?>
                                                    </select>
                                                </div>
                                            <?php endif; ?>


                                        </div>
                                        <?php $i++;
                                    endforeach;
                                endif;
                                ?>

                                <div class="servicePlans"></div>
                                <button type="button" class="add_plan btn btn-sm btn-info rounded" extra="addPlans">Add
                                    Plans</button>

                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div><?php
    }

    protected function render_js()
    {
        $settings = $this->get_settings();
        $networks = $this->providers;
        ?>
        <script>

            jQuery(document).on('click', '.add_plan', function () {
                var totalPlan = jQuery(".thePlan").length;
                let index = totalPlan;
                const dataPlanTemplate = `
                <?php
                if ($this->uses_provider):
                    $col = "col-6 col-md-4";
                else:
                    $col = "col-4 col-md-3";
                endif;
                ?>
                    <div class="thePlan border rounded py-2 px-1 my-2 row my-3" for="">
                        <div class="closeThis-div d-flex justify-content-end" ><i class="fa-regular closeThis fa-circle-xmark text-danger" for=""></i></div>
                        <div class="<?php echo $col; ?>">
                            <label>Order</label>
                            <input type="text" class="form-control" name="plans[${index}][listOrder]" value="">
                        </div>
                        <div class="<?php echo $col; ?>">
                            <label>Plan Id</label>
                            <input type="text" class="form-control" name="plans[${index}][plan_id]" value="">
                        </div>
                        <?php if ($this->id == "data"): ?>
                            <div class="<?php echo $col; ?> col-12">
                                <label>Data Bundle</label>
                                <div class="d-flex">
                                    <input type="number" step="0.01" min="0" class="form-control" name="plans[${index}][prefix]" value="e.g 200">
                                    <select class="surfix" name="plans[${index}][surfix]">
                                        <option value="mb">MB</option>
                                        <option value="gb">GB</option>
                                    </select>
                                </div>
                            </div>
                        <?php else: ?>
                        <div class="<?php echo $col; ?>">
                            <label>Plan Name</label>
                            <input type="text" class="form-control" name="plans[${index}][plan_name]" value="">
                        </div>
                        <?php endif; ?>

                        <div class="<?php echo $col; ?>">
                            <label>Plan Price</label>
                            <input type="text" class="form-control" name="plans[${index}][plan_price]" value="">
                        </div>
                        <div class="<?php echo $col; ?>">
                            <label>Duration</label>
                            <div class="d-flex">
                                <input type="text" class="form-control"
                                    name="plans[${index}][plan_price]"
                                    value="<?php echo add_service_value("duration_figure", ["duration_figure" => 1]); ?>">
                                <select class="form-control" name="plans[${index}][duration_time]">
                                    <?php
                                    $alldurations = [
                                        "days",
                                        "weeks",
                                        "months",
                                        "year"
                                    ];
                                    echo get_select_options($alldurations, add_service_value("duration_time", ["duration_time" => 'months']));
                                    ?>
                                </select>
                            </div>
                        </div>
                        <?php if ($this->uses_provider): ?>
                            <div class="<?php echo $col; ?>">
                                <label>Provider</label>
                                <select class="form-control" name="plans[${index}][network]">
                                <?php foreach ($networks as $net):
                                    $net = strtolower($net);
                                    ?>
                                        <option value="<?php echo $net; ?>"><?php echo ucfirst($net); ?></option>
                                <?php endforeach; ?>
                                </select>
                            </div>
                        <?php endif; ?>

                    </div>
            `;

                jQuery(".plan-title").text("Plans & price");
                jQuery(".plan-message").text("");
                jQuery(".servicePlans ").append(dataPlanTemplate);
                jQuery(".servicePlans > div.thePlan:last-of-type").attr("for", totalPlan + 1);
                jQuery(".servicePlans > div.thePlan:last-of-type .closeThis ").attr("for", totalPlan + 1);
            });

            jQuery(document).on('click', '.closeThis', function () {
                jQuery(this).parents(".thePlan").remove();
            });
            jQuery(document).on('click', '.showNetwork', function () {

                jQuery(".imgCover").removeClass("active");
                jQuery(this).parents(".imgCover").addClass("active");

                var showFor = jQuery(this).attr("for").toLowerCase();
                if (showFor == "all") {
                    jQuery(".thePlan").show();
                    return;
                }
                jQuery(".thePlan").hide();
                jQuery(".thePlan[for='" + showFor + "']").show();

            });
        </script>

        <?php
    }

    /**
     * Default columns common to ALL services
     */
    public function base_history_columns(): array
    {
        return [
            'reference' => 'Reference',
            'user_id' => 'Member ID',
            'amount' => 'Amount',
            'profit' => 'Profit',
            'status' => 'Status',
            'via' => 'Via',
            'created_at' => 'Date',
        ];
    }

    /**
     * Helper: get details safely
     */
    protected function details(array $txn): array
    {
        return is_array($txn['details'])
            ? $txn['details']
            : json_decode($txn['details'], true) ?? [];
    }


    public function get_settings($cache = true, $global = true): array
    {
        if ($this->settings_cache && $cache) {
            return $this->settings_cache;
        }

        $defaults = $this->default_settings(
            $global ? array_merge($this->global_fields(), $this->form_fields()) : $this->form_fields()
        );

        $this->settings_cache = wp_parse_args(
            Vtupress_Option::get($this->get_option_key(), []),
            $defaults
        );

        if (!$this->uses_api_setup && !empty($this->default_gateway_url)) {
            $this->settings_cache['base_url'] = $this->default_gateway_url;
            $this->settings_cache['end_point'] = '';
        }

        return $this->settings_cache;
    }

    protected function get_option_key(): string
    {
        return 'vtupress_service_' . $this->id . "_{$this->current_country}" . ($this->service_type ? "_{$this->service_type}" : '');
    }


    /**
     * ############
     * POST SERVICE
     */

    public function is_provider_enabled_for_type(string $provider, string $type): bool
    {
        if (!$this->uses_provider) {
            return true;
        }

        $settings = $this->get_settings();

        // global provider disabled
        if (($settings[strtolower($provider) . '_status'] ?? 'off') !== 'on') {
            return false;
        }

        // service-type specific enable toggle
        $typeKey = strtolower($type) . '_enable';

        if (isset($settings[$typeKey])) {
            return $settings[$typeKey] === 'yes';
        }

        return true;
    }

    /* ======================================================
     * BONUS / DISCOUNT ENGINE (SETTINGS-DRIVEN)
     * ====================================================== */

    /**
     * Whether this service supports service-type bonus
     */
    protected function uses_service_bonus(): bool
    {
        $settings = $this->get_settings();

        return !empty($settings['service_bonus_enable'])
            && $settings['service_bonus_enable'] === 'yes';
    }

    /**
     * Get service-type bonus (fallback = 0)
     */
    /**
     * Get service-type bonus (fallback = 0)
     */
    protected function get_service_bonus(string $serviceType)
    {
        $settings = $this->get_settings();

        $key = strtolower($serviceType) . '_bonus';

        return (isset($settings[$key]) && (is_numeric($settings[$key]) || is_string($settings[$key])))
            ? $settings[$key]
            : 0;
    }

    /**
     * Apply bonus using settings config
     */
    protected function apply_bonus(float $amount, $bonus): array
    {
        $settings = $this->get_settings();

        $method = $settings['discount_method'] ?? 'discount';

        // Handle Percentage (check for string '%' if passed as string)
        if (is_string($bonus) && strpos($bonus, '%') !== false) {
            $bonusVal = (float) str_replace('%', '', $bonus);
            $bonusAmount = ($amount * $bonusVal) / 100;
        } else {
            $bonusAmount = (float) $bonus;
        }


        // Commission: User pays Full Amount. Commission is handled separately (e.g. credited to wallet).
        // Discount: User pays Discounted Amount.
        $payable = ($method === 'discount')
            ? $amount - $bonusAmount
            : $amount;

        return [
            'original' => round($amount, 2),
            'bonus' => round($bonusAmount, 2),
            'payable' => max(0, round($payable, 2)),
        ];
    }

    /**
     * Network bonus
     */
    public function get_network_bonus(string $network)
    {
        if (!$this->uses_bonus) {
            return 0;
        }

        $settings = $this->get_settings();
        $key = strtolower($network) . '_bonus';

        return (isset($settings[$key]) && (is_numeric($settings[$key]) || is_string($settings[$key])))
            ? $settings[$key]
            : 0;
    }

    /**
     * Calculate final bonus (network → service override)
     */
    public function calculate_bonus(
        float $amount,
        string $network,
        string $serviceType
    ): array {

        $bonus = $this->get_network_bonus($network);

        if ($this->uses_service_bonus()) {
            $serviceBonus = $this->get_service_bonus($serviceType);
            // Override if service bonus is set (and greater than 0/not empty)
            if (!empty($serviceBonus)) {
                $bonus = $serviceBonus;
            }
        }

        $base_result = $this->apply_bonus($amount, $bonus);

        // --- TIERED LEVEL REWARDS (ADDITIVE) ---
        $tiered_bonus_amount = 0;
        if ($this->user_id) {
            $users_table = $this->db->prefix . 'vtupress_users';
            $reward_table = $this->db->prefix . 'vtupress_level_service_rewards';

            $user_plan = $this->db->get_var($this->db->prepare(
                "SELECT plan FROM $users_table WHERE id = %d",
                $this->user_id
            ));

            if ($user_plan) {
                $tiered_bonus_val = $this->db->get_var($this->db->prepare(
                    "SELECT bonus_value FROM $reward_table WHERE level_name = %s AND service_id = %s AND set_country = %s",
                    strtolower($user_plan),
                    $this->id,
                    $this->current_country
                ));

                if ($tiered_bonus_val && $tiered_bonus_val !== "0") {
                    if (strpos($tiered_bonus_val, '%') !== false) {
                        $p_val = (float) str_replace('%', '', $tiered_bonus_val);
                        $tiered_bonus_amount = ($amount * $p_val) / 100;
                    } else {
                        $tiered_bonus_amount = (float) $tiered_bonus_val;
                    }
                }
            }
        }

        $final_bonus = $base_result['bonus'] + $tiered_bonus_amount;
        $payable = $base_result['original'] - $final_bonus;

        return [
            'original' => $base_result['original'],
            'bonus' => round($final_bonus, 2),
            'payable' => max(0, round($payable, 2)),
        ];
    }


    public function calculate_pay_amount(array $param): array
    {
        $amount = (float) ($param['amount'] ?? $param['value'] ?? 0);
        $network = strtolower($param['network'] ?? $param['provider'] ?? '');
        $type = strtolower($param['type'] ?? ''); // Service Type
        $plan_id = $param['plan'] ?? '';

        // For Vouchers (Internal PINs), the 'plan_id' column in DB usually stores the Provider/Network (e.g. 'mtn').
        // If 'plan' param is empty (because uses_plans=false), fallback to network.
        if ($this->is_voucher && empty($plan_id)) {
            $plan_id = $network;
        }

        $base_amount = $amount;
        $planFound = false;

        // VOUCHER / INTERNAL PIN LOGIC
        if ($this->is_voucher) {
            global $wpdb;
            $table = $wpdb->prefix . 'vtupress_internal_pins';
            $plan_id = $param['provider'] ?? '';
            $amount = $param['plan'] ?? '';
            $quantity = $param['quantity'] ?? 1;

            // Map params to DB columns
            // service_id = $this->id
            // plan_id = $plan_id (which is usually the provider slug e.g 'mtn')
            // value = $amount (input amount/value)
            // type = $type

            // We need to match case-insensitively or loosely?
            // Usually IDs are strict. plan_id from frontend is usually slug.

            $query = "SELECT price FROM $table WHERE service_id = %s AND plan_id = %s AND value = %s AND type = %s AND status = 'unused' ORDER BY id ASC LIMIT 1";
            $price_check = $wpdb->get_var($wpdb->prepare($query, $this->id, $plan_id, $amount, $type));

            if ($price_check !== null) {
                $base_amount = (float) $price_check * $quantity;
                $planFound = true; // Treated as "Plan Found" i.e. authoritative price
            }
        } elseif ($this->uses_plans && !empty($this->plans)) {
            // Existing Plan Logic


            $planPrice = 0;
            $planName = '';

            // Normalize Input Slug for comparison
            // Expected format: network_type_size_duration (e.g. mtn_sme_1gb_1month)
            $inputSlug = strtolower(preg_replace('/[^a-zA-Z0-9]/', '', $plan_id));

            foreach ($this->plans as $plan) {
                // Filter by Network/Type if they are set in the plan?
                // Actually, let's just search all, assuming ID/Slug implies specificity.

                // A. Exact ID Match (Provider ID or System ID)
                if (
                    (string) $plan['id'] === (string) $plan_id ||
                    (string) $plan['plan_id'] === (string) $plan_id
                ) {
                    $planPrice = (float) $plan['price'];
                    $planFound = true;
                    break;
                }

                // B. Smart Slug Match
                // Construct slug from plan details: network + service_type + name (size) + duration
                // We strip non-alphanumeric chars to make it robust (e.g. "1 GB" -> "1gb")

                $p_net = strtolower($plan['network']);
                $p_type = strtolower($plan['service_type'] ?? $type);
                $p_name = strtolower($plan['plan_name'] ?? ''); // usually contains size e.g. "1GB"
                $p_dur = strtolower((string) ($plan['duration_time'] ?? '')); // e.g. "Month"
                $p_fig = (string) ($plan['duration_figure'] ?? '');      // e.g. "1"

                // Construct variations of slugs to check against?
                // User said: mtn_sme_1gb_1months
                // Let's Clean them:
                $clean_net = preg_replace('/[^a-z0-9]/', '', $p_net);
                $clean_type = preg_replace('/[^a-z0-9]/', '', $p_type);
                $clean_name = preg_replace('/[^a-z0-9]/', '', $p_name); // "1GB Monthly" -> "1gbmonthly"
                $clean_dur = preg_replace('/[^a-z0-9]/', '', $p_fig . $p_dur); // "1month"

                // Combined Slug Options
                // 1. network_type_name_duration (mtn_sme_1gb_1month)
                // 2. name_duration (1gb_1month) - if user omitted network/type in ID
                // 3. name (1gb) - for simple matches

                $slug1 = $clean_net . $clean_type . $clean_name . $clean_dur;
                $slug2 = $clean_name . $clean_dur;
                $slug3 = $clean_name;

                // Also support underscore separated input
                // But we stripped inputSlug of underscores above. 
                // So "mtn_sme_1gb_1months" -> "mtnsme1gb1months"

                if (
                    $inputSlug === $slug1 ||
                    $inputSlug === $slug2 ||
                    $inputSlug === $slug3
                ) {
                    $planPrice = (float) $plan['price'];
                    $planFound = true;
                    break;
                }
            }

            if ($planFound) {
                // Plan Found: The Plan Price is authoritative.
                $base_amount = $planPrice;
            } else {
                // If plans are enabled but no plan found:
                // If it's strictly a plan-based service (Data), we should error.
                // But for flexibility, if an Amount was explicitly provided, maybe we use it?
                // User said: "make sure services ... can be used for api purposes".
                // Usually API users strictly want the plan they asked for.
                throw new Exception("Invalid Plan Selected [$plan_id]");
            }
        } else {
            // 2. Non-Plan Logic (Airtime, etc.)
            // Validate Min/Max if settings allow?
            if ($base_amount <= 0) {
                throw new Exception("Invalid Amount");
            }
        }

        // 2. Calculate Bonus/Payable on the Truth Amount
        return $this->calculate_bonus($base_amount, $network, $type);
    }



    public function process_transaction($settings, $user_instance, $user_data, $source, array $param): array
    {
        // 0. SECURE CALCULATION & VERIFICATION
        try {
            $calculation = $this->calculate_pay_amount($param);
        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }

        $amount_to_charge = $calculation['payable'];
        $original_amount = $calculation['original']; // The "Truth" amount
        $bonus_amount = $calculation['bonus'] ?? 0;

        // Update param with the Truth Amount (Original) to ensure API receives correct value
        // even if frontend sent something else (though we verified mismatch, good to be safe)
        $param['amount'] = $original_amount;

        $payload = $param;

        // 1. Debit User
        $user_id = $user_data['id'];
        $service_name = $this->id;
        $description = "Purchase of $service_name ($original_amount)";

        if ($user_data['balance'] < $amount_to_charge) {
            wp_send_json([
                'status' => 'error',
                'message' => 'Insufficient Balance'
            ]);
        }

        // Pass payload/param as details to be logged
        $debited_txn_id = $user_instance->debit($user_id, $amount_to_charge, $service_name, $description, $param);

        if (!$debited_txn_id) {
            wp_send_json([
                'status' => 'error',
                'message' => 'Could not debit user'
            ]);
        }


        // 2. Execute Transaction
        $response = $this->execute($settings, $user_instance, $user_data, $source, $payload);

        $trans_table = $this->db->prefix . 'vtupress_transactions';

        if ($response['status'] !== 'success') {
            global $wpdb;
            $trans_settings_table = $wpdb->prefix . 'vtupress_transaction_settings';
            $trans_settings = $wpdb->get_row("SELECT auto_refund FROM $trans_settings_table WHERE id=1");

            $refund_option = $trans_settings->auto_refund ?? 'yes';

            if ($refund_option === 'yes') {
                $user_instance->credit($user_id, $amount_to_charge, 'refund', "Refund for failed $service_name transaction"); // Refund the CHARGED amount
                $response['message'] .= " (Amount Refunded)";
            } else {
                $response['message'] .= " (Amount Not Refunded)";
            }


            // Success - Update Transaction Log with Response Data (PINs, Serial, etc)
            if ($debited_txn_id && !empty($response['raw'])) {
                $responseData = json_decode($response['raw'], true);
                if (is_array($responseData)) {
                    $newDetails = array_merge($param, array_merge(['software' => 'vtupress'], $responseData));
                } else {
                    $newDetails = $param;
                }
                // $responseData = $response['raw'];
                // if (is_array($responseData)) {
                // Merge initial details with response details
                // $newDetails = $param;
                $data = ['details' => json_encode($newDetails)];
                if ($param["ref"]) {
                    $data = array_merge($data, ['reference' => $param["ref"]], ['response_message' => json_encode($responseData)], ['status' => 'failed']);
                }

                $this->db->update($trans_table, $data, ['id' => $debited_txn_id]);
                // }
            } else {
                $responseData = [];
                $responseData["system_error"] = $response["message"] ?? 'Message Not Detected';
                $data = ['details' => json_encode($param)];
                if ($param["ref"]) {
                    $data = array_merge($data, ['reference' => $param["ref"]], ['response_message' => json_encode($responseData)], ['status' => 'failed']);
                }
                $this->db->update($trans_table, $data, ['id' => $debited_txn_id]);
            }

            if (isset($response["kyc_error"])) {
                $this->db->delete($trans_table, ['id' => $debited_txn_id]);

                $response['message'] .= " And Transaction excluded against kyc check";

            }

            // wp_send_json($response);
            return $response;
        } else {

            // Success - Update Transaction Log with Response Data (PINs, Serial, etc)
            if ($debited_txn_id && !empty($response['raw'])) {
                $responseData = json_decode($response['raw'], true);
                if (is_array($responseData)) {
                    $newDetails = array_merge($param, array_merge(['software' => 'vtupress'], $responseData));
                } else {
                    $newDetails = $param;
                }

                if (!empty($response['data']) && is_array($response['data'])) {
                    $newDetails = array_merge($newDetails, $response['data']);
                }
                // if (is_array($responseData)) {
                // Merge initial details with response details
                // $newDetails = array_merge($param, $responseData);
                $data = ['details' => json_encode($newDetails)];
                $profit = 0;

                $ch_key = !empty($settings["charge_key"] ?? 'amount') ? $settings["charge_key"] ?? 'amount': 'amount';

                //capture charge
                if (is_array($responseData)) {
                    $charged = $this->catch_key($responseData, $ch_key);
                    if ($charged != false) {
                        $profit = (int) $amount_to_charge - (int) $charged;
                        $data = array_merge($data, ['profit' => $profit]);
                    }
                }

                if ($param["ref"]) {
                    $data = array_merge($data, ['reference' => $param["ref"]], ['response_message' => json_encode($responseData)]);
                }

                $this->db->update($trans_table, $data, ['id' => $debited_txn_id]);
                // }
            }
        }

        // 4. Success - Credit Commissions
        // We pass the Original Amount for commission calculation, and the bonus amount for Self-Commission
        $this->credit_commissions($user_id, $original_amount, $bonus_amount);

        return $response;
    }

    /**
     * Credit Commissions (Self & Referral)
     */
    protected function credit_commissions(int $user_id, float $transaction_amount, float $self_bonus)
    {
        global $wpdb;
        $users_table = $wpdb->prefix . 'vtupress_users';

        // 1. SELF COMMISSION (Cashback)
        // If discount method was 'commission', $self_bonus contained the cashback amount.
        if ($self_bonus > 0) {
            // "referral_rewards" is the withdrawable wallet for commissions
            // "transactions_commission" tracks total self-commission earned
            $wpdb->query("UPDATE $users_table SET
                referral_rewards = referral_rewards + $self_bonus,
                transactions_commission = transactions_commission + $self_bonus
                WHERE id = $user_id");
        }

        // 2. REFERRAL COMMISSION (Uplines)
        $levels_table = $wpdb->prefix . 'vtupress_levels';

        $current_upline_id = $wpdb->get_var($wpdb->prepare("SELECT referrer_id FROM $users_table WHERE id = %d", $user_id));

        $generation = 1;
        while ($current_upline_id > 1 && $generation <= 10) { // Limit to 10 generations for safety
            $upline_data = $wpdb->get_row($wpdb->prepare("SELECT plan, referrer_id FROM $users_table WHERE id = %d", $current_upline_id));
            if (!$upline_data)
                break;

            $plan = $upline_data->plan;
            $level_col = "level_" . $generation;

            // 1. Check for Service-Specific Override for this Gen
            $gen_table = $wpdb->prefix . "vtupress_levelGeneration";
            $service_override = $wpdb->get_var($wpdb->prepare(
                "SELECT transaction_commission FROM $gen_table WHERE package_name = %s AND generation = %s AND service_name = %s AND set_country = %s",
                $plan,
                $generation,
                $this->id,
                $this->current_country
            ));

            $plan_data = $wpdb->get_row($wpdb->prepare("SELECT $level_col, total_level FROM $levels_table WHERE name = %s", $plan));

            if ($plan_data && (int) $plan_data->total_level >= $generation) {
                // If override exists and is > 0, use it. Otherwise use plan default.
                if ($service_override !== null && (float) $service_override > 0) {
                    $percent = (float) $service_override;
                } else {
                    $percent = (float) ($plan_data->$level_col ?? 0);
                }

                if ($percent > 0) {
                    $comm = ($transaction_amount * $percent) / 100;
                    $comm = round($comm, 2);
                    if ($comm > 0) {
                        // Update upline's referral rewards and stats based on level
                        $stat_col = "other_referral_commission";
                        if ($generation == 1)
                            $stat_col = "referral_commission";
                        elseif ($generation == 2)
                            $stat_col = "indirect_referral_commission";

                        $wpdb->query($wpdb->prepare(
                            "UPDATE $users_table SET
                                referral_rewards = referral_rewards + %f,
                                $stat_col = $stat_col + %f
                                WHERE id = %d",
                            $comm,
                            $comm,
                            $current_upline_id
                        ));
                    }
                }
            }

            $current_upline_id = $upline_data->referrer_id;
            $generation++;
        }
    }
    abstract public function form_fields(): array;

    public function render_admin_form(): void
    {
        $this->maybe_save_settings();
        $this->formData();
    }
}


class Airtime extends Vtupress_services implements Vtupress_History_Interface
{

    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'airtime';
        $this->name = 'Airtime';
        $this->description = 'An airtime api';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/airtime.jpeg";
        $this->uses_api = true;
        $this->uses_provider = true;
        // Default providers (dynamic)
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$this->current_country]['airtime'] ?? [];
        $this->uses_bonus = true;
        $this->icon = 'fas fa-phone menu-icon';
        $this->service_types = array_map('strtolower', ["vtu", "awoof", "share_and_sell"]);
        $this->uses_ports = true;
        $this->payload = ["type", "provider", "recipient", "amount", "ref"];
        $this->label_provider = 'Network';
        $this->label_number = 'Phone Number';
        parent::__construct($db, $service_type, $user_id);
    }

    public function history_columns(): array
    {
        return [
            'provider' => 'Provider',
            'type' => 'Type',
        ];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);

        return match ($column) {
            'provider' => $details['provider'] ?? '-',
            'type' => strtoupper($details['type'] ?? '-'),
            default => '-',
        };
    }

    public function form_fields(): array
    {
        $fields = [];

        return $fields;
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_mutable_frontend($user, $runcode);
    }

}

class Data extends Vtupress_services implements Vtupress_History_Interface
{

    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'data';
        $this->name = 'Data';
        $this->description = 'Data service';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/data.jpg";
        $this->uses_api = true;
        $this->uses_provider = true;
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$this->current_country]['data'] ?? [];
        $this->uses_bonus = true;
        $this->uses_plans = true;
        $this->uses_ports = true;
        $this->icon = "mdi mdi-database menu-icon";
        $this->payload = ["provider", "recipient", "plan", "ref"];
        $this->service_types = array_map('strtolower', ["sme", "corporate", "gifting"]);
        $this->label_provider = 'Network';
        $this->label_number = 'Phone Number';
        parent::__construct($db, $service_type, $user_id);
    }


    public function form_fields(): array
    {
        $fields = [];

        return $fields;
    }

    public function history_columns(): array
    {
        return [
            'plan' => 'Plan',
        ];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);

        return $column === 'plan'
            ? ($details['plan'] ?? '-')
            : '-';
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_fixed_frontend($user, $runcode);
    }

}




class Cable extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'cable';
        $this->name = 'Cable';
        $this->description = 'Cable TV Subscription';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/cable.jpg"; // Placeholder
        $this->uses_api = true;
        $this->uses_provider = true; // Re-enable provider settings
        $this->allow_custom_providers = true;
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$this->current_country]['cable'] ?? [];
        $this->uses_plans = true;
        $this->uses_bonus = true;
        $this->service_types = ['subscription']; // Generic type
        $this->icon = "mdi mdi-television-classic menu-icon";
        $this->payload = ["provider", "recipient", "plan", "amount", "ref"];
        $this->label_provider = 'Provider';
        $this->label_number = 'Smartcard / IUC';
        $this->has_verification = true;
        parent::__construct($db, $service_type, $user_id);
    }

    public function form_fields(): array
    {
        return [];
    }

    public function history_columns(): array
    {
        return [
            'plan' => 'Plan',
            'provider' => 'Provider',
            'recipient' => 'Recipient'
        ];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);
        // return $column === 'plan' ? ($details['plan'] ?? '-') : '-';
        return match ($column) {
            'plan' => $details['plan'] ?? '-',
            'provider' => $details['provider'] ?? '-',
            'recipient' => $details['recipient'] ?? '-',
            default => '-',
        };
    }

    public function verify(array $args)
    {
        $iuc = $args['number'];
        $cable = $args['provider'];

        $http_args = array(
            'headers' => array(
                'cache-control' => 'no-cache',
                'Content-Type' => 'application/json'
            ),
            'timeout' => '300',
            'user-agent' => 'Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1)',
            'sslverify' => false
        );

        // Call the centralized verification API (formerly in cableget.php)
        $response = wp_remote_get("https://vtupress.com/billget.php?cableget=yes&service=$cable&meter=$iuc", $http_args);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);

        // Check for specific error responses from API or return body
        if (empty($body) || strpos($body, 'error') !== false || strpos($body, 'Error') !== false) {
            return new WP_Error('verification_failed', $body ?: 'Verification failed');
        }

        return $body;
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_fixed_frontend($user, $runcode);
    }
}


class Bill extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'bill';
        $this->name = 'Bill';
        $this->description = 'Utility Bill Payment';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/utility.jpg"; // Placeholder
        $this->uses_api = true;
        $this->uses_api = true;
        $this->uses_provider = true; // Re-enable provider settings
        $this->allow_custom_providers = true;
        // Common Discos (Dynamic)
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$this->current_country]['bill'] ?? [];
        $this->uses_bonus = true;
        $this->service_types = ['prepaid', 'postpaid'];
        $this->icon = "mdi mdi-lightbulb-on menu-icon";
        $this->payload = ["provider", "recipient", "amount", "type", "ref"];
        $this->label_provider = 'Disco';
        $this->label_number = 'Meter Number';
        $this->label_service_type = 'Meter Type';
        $this->has_verification = true;
        parent::__construct($db, $service_type, $user_id);
    }

    public function form_fields(): array
    {
        return [];
    }

    public function history_columns(): array
    {
        return [
            'type' => 'Type',
            'provider' => 'Provider',
            'recipient' => 'Recipient',
            'address' => 'Address',
            'units' => 'Units',
            'token' => 'Token'
        ];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);
        // return $column === 'type' ? strtoupper($details['type'] ?? '-') : '-';
        return match ($column) {
            'type' => strtoupper($details['type'] ?? '-'),
            'provider' => $details['provider'] ?? '-',
            'recipient' => $details['recipient'] ?? '-',
            'address' => $details['address'] ?? '-',
            'token' => $details['token'] ?? '-',
            'units' => $details['units'] ?? '-',
            default => '-',
        };
    }

    public function verify(array $args)
    {
        $meter = $args['number'];
        $bill = $args['provider'];
        // Type is not passed in the unified ajax check (it is passed but might be needed here)
        // However, the base verify method only receives provider and number essentially from the stub.
        // Wait, ajax_verify sends 'service_type' which is 'prepaid'/'postpaid' usually generally.
        // But for Bill, 'type' is distinct (prepaid/postpaid).
        // Let's modify ajax_verify to pass more args or handle it here?
        // Actually, for bills, the previous `billget.php` required `type`.
        // I need to ensure `ajax_verify` passes `service_type` (which we used for subscription vs bills)
        // But here 'type' means prepaid/postpaid.
        // Let's assume the frontend sends 'service_type' as prepaid/postpaid or we need to look at how ajax_verify is called.
        // In the frontend JS for Bill, we will pass 'service_type': jQuery(".bill-type").val().

        $type = $args['type'] ?? 'prepaid'; // Default or extracted

        $http_args = array(
            'headers' => array(
                'cache-control' => 'no-cache',
                'Content-Type' => 'application/json'
            ),
            'timeout' => '300',
            'user-agent' => 'Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1)',
            'sslverify' => false
        );

        $response = wp_remote_get("https://vtupress.com/billget.php?billget=yes&bill=$bill&meter=$meter&type=$type", $http_args);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);

        if (empty($body) || strpos($body, 'error') !== false || strpos($body, 'Error') !== false) {
            return new WP_Error('verification_failed', $body ?: 'Verification failed');
        }

        return $body;
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_mutable_frontend($user, $runcode);
    }
}

function vtupress_service_registry(): array
{
    return apply_filters('vtupress_service_registry', [
        'airtime' => Airtime::class,
        'data' => Data::class,
        'cable' => Cable::class,
        'bill' => Bill::class,
        'recharge_card' => RechargeCard::class,
        'data_card' => DataCard::class,
        'education' => ExamPin::class,
        'transfer' => Transfer::class,
        'withdrawal' => Withdrawal::class,
        'savings' => Vtupress_Savings::class,
        'ng_verification' => ng_verification::class
    ]);
}

class RechargeCard extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'recharge_card';
        $this->name = 'Recharge Card';
        $this->description = 'Print Recharge Cards';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/recharge_card.jpeg"; // Placeholder
        $this->uses_api = true;
        $this->uses_provider = true;
        // Default to internal API
        $this->default_gateway_url = home_url('/vtupress/internal-pin-api');

        $current_country = Vtupress_Country_Context::get($user_id);
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$current_country]['recharge_card'] ?? ['mtn', 'glo', 'airtel', '9mobile'];
        $this->uses_plans = false;
        $this->uses_bonus = true;
        $this->service_types = ['local'];
        $this->icon = "mdi mdi-printer menu-icon";
        // Payload for internal api or external? 
        // If internal, we need service, plan, quantity. 
        // Existing logic maps 'recipient' -> number.
        // We will repurpose fields.
        $this->payload = ["provider", "plan", "quantity", "type", "ref"];
        $this->label_provider = 'Proivder';
        $this->label_number = 'Quantity';
        $this->is_voucher = true;
        $this->uses_api_setup = false;
        parent::__construct($db, $service_type, $user_id);
    }

    public function form_fields(): array
    {
        return [];
    }

    public function history_columns(): array
    {
        return ['quantity' => 'Quantity', 'pin' => 'PIN', 'serial' => 'Serial'];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);
        if ($column == 'pin')
            return $details['pin'] ?? '-';
        if ($column == 'serial')
            return $details['serial'] ?? '-';
        if ($column == 'quantity')
            return $details['quantity'] ?? '-';
        return '-';
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_pin_frontend($user, $runcode);
    }
}

class DataCard extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'data_card';
        $this->name = 'Data Card';
        $this->description = 'Data PINs';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/data-card.png";
        $this->uses_api = true;
        $this->uses_provider = true;
        $this->default_gateway_url = home_url('/vtupress/internal-pin-api');

        $current_country = Vtupress_Country_Context::get($user_id);
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$current_country]['data_card'] ?? ['mtn', 'glo', 'airtel', '9mobile'];
        $this->uses_plans = false;
        $this->uses_bonus = true;
        $this->service_types = ['local'];
        $this->icon = "mdi mdi-wifi menu-icon";
        $this->payload = ["provider", "plan", "quantity", "type", "ref"];
        $this->label_provider = 'Network';
        $this->label_number = 'Quantity';
        $this->is_voucher = true;
        $this->uses_api_setup = false;
        parent::__construct($db, $service_type, $user_id);
    }

    public function form_fields(): array
    {
        return [];
    }

    public function history_columns(): array
    {
        return ['quantity' => 'Quantity', 'pin' => 'PIN', 'serial' => 'Serial'];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);
        if ($column == 'pin')
            return $details['pin'] ?? '-';
        if ($column == 'serial')
            return $details['serial'] ?? '-';
        if ($column == 'quantity')
            return $details['quantity'] ?? '-';
        return '-';
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_pin_frontend($user, $runcode);
    }
}

class ExamPin extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'education';
        $this->name = 'Exam Pin';
        $this->description = 'WAEC/NECO Pins';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/exam_pin.jpeg";
        $this->uses_api = true;
        $this->uses_provider = true;
        $this->default_gateway_url = home_url('/vtupress/internal-pin-api');

        $current_country = Vtupress_Country_Context::get($user_id);
        $country_data = Vtupress_Country_Manager::all();
        $this->default_providers = $country_data[$current_country]['education'] ?? ['waec', 'neco', 'nabteb'];
        $this->uses_plans = false;
        $this->uses_bonus = true;
        $this->service_types = ['local'];
        $this->icon = "mdi mdi-school menu-icon";
        $this->payload = ["provider", "plan", "quantity", "type", "ref"];
        $this->label_provider = 'Exam Body';
        $this->label_number = 'Quantity';
        $this->is_voucher = true;
        $this->uses_api_setup = false;
        parent::__construct($db, $service_type, $user_id);
    }

    public function form_fields(): array
    {
        return [];
    }

    public function history_columns(): array
    {
        return ['quantity' => 'Quantity', 'pin' => 'PIN', 'serial' => 'Serial'];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        $details = $this->details($txn);
        if ($column == 'pin')
            return $details['pin'] ?? '-';
        if ($column == 'serial')
            return $details['serial'] ?? '-';
        if ($column == 'quantity')
            return $details['quantity'] ?? '-';
        return '-';
    }

    public function render_frontend($user, $runcode = "")
    {
        $this->render_pin_frontend($user, $runcode);
    }
}

class Transfer extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'transfer';
        $this->name = 'Transfer';
        $this->description = 'Bank Transfer';
        $this->countries = ['nigeria'];
        $this->image = VTUPRESS_URL . "/images/service/transfer.webp";
        $this->uses_api = true; // Use Gateway
        $this->custom_id = 'transfer'; //Transfer feature
        $this->uses_provider = true; // Banks
        $this->default_providers = [];
        $this->uses_plans = false;
        $this->icon = "mdi mdi-bank menu-icon";
        $this->payload = ["provider", "recipient", "amount"];
        $this->label_provider = 'Bank';
        $this->label_number = 'Account Number';
        $this->has_verification = true; // Name enquiry
        $this->uses_api_setup = false;
        parent::__construct($db, $service_type, $user_id);
    }

    public function form_fields(): array
    {
        // Get available gateways
        $gateways = vtupress_gateway_registry();
        $options = []; // Array of Gateways supporting Transfer

        global $wpdb;
        foreach ($gateways as $id => $class) {
            if (class_exists($class)) {
                $instance = new $class($wpdb);
                if (method_exists($instance, 'get_banks')) {
                    $options[] = $id;
                }
            }
        }

        return [
            "Transfer Settings" => [
                "gateway" => [
                    "label" => "Processing Gateway",
                    "description" => "Select the gateway to use for transfers (and bank lists)",
                    "column" => 6,
                    "render" => [
                        "class" => "border border-success",
                        "value" => $options
                    ]
                ]
            ]
        ];
    }

    public function history_columns(): array
    {
        return ['bank' => 'Bank'];
    }
    public function resolve_history_column(string $c, array $t): string
    {
        return '-';
    }
    public function verify(array $args)
    {
        return "Account Name";
    }

    public function render_frontend($user, $runcode = "")
    {
        $settings = $this->get_settings();
        $gateway_id = $settings['gateway'] ?? 'manual';

        // Load Gateway
        $gateways = vtupress_gateway_registry();
        $banks = [];

        if (isset($gateways[$gateway_id])) {
            global $wpdb;
            $gateway_class = $gateways[$gateway_id];
            if (class_exists($gateway_class)) {
                $gateway_instance = new $gateway_class($wpdb);
                if (method_exists($gateway_instance, 'get_banks')) {
                    $banks = $gateway_instance->get_banks();
                }
            }
        }

        // Sort Banks Alphabetically
        usort($banks, function ($a, $b) {
            return strcmp($a['name'], $b['name']);
        });

        ob_start();
        ?>
        <style>
            select.form-select,
            input[type='number'],
            input[type='text'] {
                height: 50px !important;
            }

            .purchase-btn {
                height: 65px !important;
            }
        </style>
        <link rel="stylesheet"
            href="<?php echo plugins_url('vtupress-templates/templates/msorg_template'); ?>/msorg_template/form.css"
            media="all">

        <div class="container-md mt-3">
            <div class="service-form p-3" style="border: 1px solid grey; border-radius: 5px;">
                <div class="mb-2">
                    <form id="transfer-form">

                        <!-- Bank Select -->
                        <div class="mb-2">
                            <label class="form-label">Bank Name</label>
                            <select class="form-select form-select-sm bank-code" name="bank_code">
                                <option value="">- Select Bank -</option>
                                <?php foreach ($banks as $bank): ?>
                                    <option value="<?php echo esc_attr($bank['code']); ?>"><?php echo esc_html($bank['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Account Number & Verify -->
                        <div class="mb-2">
                            <label class="form-label">Account Number</label>
                            <div class="input-group">
                                <input type="number" class="form-control account-number" placeholder="Enter Account Number">
                                <button class="btn btn-outline-secondary verify-btn" type="button">Verify</button>
                            </div>
                            <div class="form-text text-success fw-bold account-name-display"></div>
                        </div>

                        <!-- Amount -->
                        <div class="mb-2">
                            <label class="form-label">Amount</label>
                            <input type="number" class="form-control amount" placeholder="Amount">
                        </div>

                        <!-- Submit -->
                        <div class="mb-2">
                            <button type="button" class="btn btn-primary w-100 purchase-btn" id="transfer-btn">Transfer</button>
                        </div>

                    </form>
                </div>
            </div>
        </div>

        <script>
            jQuery(document).ready(function ($) {
                var gateway = "<?php echo $gateway_id; ?>";
                var accountName = "";
                // Verify Action
                $('.verify-btn').click(function () {
                    var btn = $(this);
                    var bankCode = $('.bank-code').val();
                    var accountNum = $('.account-number').val();
                    var display = $('.account-name-display');

                    if (!bankCode || !accountNum) {
                        Swal.fire('Error', 'Please select bank and enter account number', 'error');
                        return;
                    }
                    btn.prop('disabled', true).text('Verifying...');
                    display.text('');

                    $.post(vtupressInitPluginData.ajax_url, {
                        action: 'vtupress_resolve_bank_account',
                        bank_code: bankCode,
                        account_number: accountNum,
                        gateway: gateway
                    }, function (res) {
                        btn.prop('disabled', false).text('Verify');
                        if (res.success) {
                            accountName = (typeof res.data === 'object' && res.data.account_name) ? res.data.account_name : res.data;
                            display.removeClass('text-danger').addClass('text-success').text(accountName);
                            Swal.fire('Verified', 'Account: ' + accountName, 'success');
                        } else {
                            accountName = "";
                            var msg = (typeof res.data === 'object') ? res.data.message : res.data;
                            display.removeClass('text-success').addClass('text-danger').text(msg || 'Verification Failed');
                            Swal.fire('Error', msg || 'Verification Failed', 'error');
                        }
                    });
                });

                // Transfer Action
                $('#transfer-btn').click(function () {
                    var amount = $('.amount').val();
                    var bankCode = $('.bank-code').val();
                    var accountNum = $('.account-number').val();
                    if (!amount || amount <= 0 || !bankCode || !accountNum) {
                        Swal.fire('Error', 'Please fill all fields correctly', 'error');
                        return;
                    }
                    if (!accountName) {
                        Swal.fire('Warning', 'Please verify the account number first', 'warning');
                        return;
                    }

                    Swal.fire({
                        title: 'Confirm Transfer?',
                        html: `Transfer <b>₦${amount}</b> to <br><b>${accountName}</b><br>(${accountNum})`,
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonText: 'Yes, Transfer',
                        cancelButtonText: 'Cancel'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            Swal.fire({
                                title: 'Processing...',
                                text: 'Please wait while we process the transfer',
                                allowOutsideClick: false,
                                didOpen: () => { Swal.showLoading(); }
                            });

                            $.post(vtupressInitPluginData.ajax_url, {
                                action: 'vtupress_transfer_funds',
                                amount: amount,
                                bank_code: bankCode,
                                account_number: accountNum,
                                account_name: accountName,
                                gateway: gateway
                            }, function (res) {
                                if (res.success) {
                                    Swal.fire('Success', res.data.message, 'success').then(() => {
                                        location.reload();
                                    });
                                } else {
                                    Swal.fire('Failed', res.data.message, 'error');
                                }
                            }).fail(function () {
                                Swal.fire('Error', 'Network Error', 'error');
                            });
                        }
                    });
                })
            });
        </script>
        <?php
    }
}

class Withdrawal extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'withdrawal';
        $this->name = 'Withdrawal';
        $this->description = 'Convert Referral Bonus to Main Wallet';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/withdrawal.webp";
        $this->uses_api = false;
        $this->uses_provider = false;
        $this->uses_plans = false;
        $this->icon = "mdi mdi-cash-multiple menu-icon";
        $this->payload = [];
        $this->label_provider = 'Method'; // Not used
        $this->label_number = 'Amount';
        parent::__construct($db, $service_type, $user_id);
    }
    public function form_fields(): array
    {
        return [];
    }
    public function history_columns(): array
    {
        return [];
    }
    public function resolve_history_column(string $c, array $t): string
    {
        return '-';
    }

    /**
     * Handle the Fund Swap (Referral -> Main Wallet)
     */
    public function process_frontend_request($request)
    {
        $amount = floatval($request['amount'] ?? $request['value'] ?? 0);
        $wp_user_id = get_current_user_id();

        $users = new Users();
        $vp_user = $users->getUser($wp_user_id, false, true);

        if (!$vp_user) {
            return ['status' => 'failed', 'message' => 'VtuPress User account not found'];
        }

        $user_id = $vp_user->id; // VtuPress Internal ID

        // 1. Transaction Lock (Concurrency Protection)
        $vend_lock_table = $this->db->prefix . 'vtupress_vend_lock';
        $inserted = $this->db->query($this->db->prepare(
            "INSERT IGNORE INTO $vend_lock_table (vend, user_id, created_at) VALUES (%s, %d, NOW())",
            'withdrawal',
            $user_id
        ));

        if (!$inserted) {
            return ['status' => 'failed', 'message' => 'Another transaction is in progress. Please wait.'];
        }

        if ($amount <= 0) {
            $this->db->delete($vend_lock_table, ['user_id' => $user_id]);
            return ['status' => 'failed', 'message' => 'Invalid Amount'];
        }

        // Get Current Rewards Balance safely
        $table_name = $this->db->prefix . 'vtupress_users';
        $user_row = $this->db->get_row($this->db->prepare("SELECT referral_rewards, balance FROM $table_name WHERE id = %d", $user_id));

        if (!$user_row) {
            $this->db->delete($vend_lock_table, ['user_id' => $user_id]);
            return ['status' => 'failed', 'message' => 'User not found'];
        }


        $current_rewards = floatval($user_row->referral_rewards);

        if ($amount > $current_rewards) {
            $this->db->delete($vend_lock_table, ['user_id' => $user_id]);
            return ['status' => 'failed', 'message' => "Insufficient Referral Balance. You have ₦$current_rewards"];
        }

        // Perform Swap (Debit Rewards, Credit Balance)
        // We do this in one query if possible, or verify update count
        $updated = $this->db->query($this->db->prepare(
            "UPDATE $table_name SET referral_rewards = referral_rewards - %f, balance = balance + %f WHERE id = %d AND referral_rewards >= %f",
            $amount,
            $amount,
            $user_id,
            $amount
        ));

        // Release Lock
        $this->db->delete($vend_lock_table, ['user_id' => $user_id]);

        if ($updated) {
            // Log Transaction
            $users = new Users();
            $users->logTransaction(
                $user_id,
                'withdrawal',
                $amount,
                $user_row->balance, // Before Balance
                $user_row->balance + $amount, // After Balance
                'success',
                ['type' => 'bonus_withdrawal', 'description' => 'Bonus to Main Wallet']
            );

            // Log Wallet
            $users->logTransaction(
                $user_id,
                'wallet',
                $amount,
                0,
                0,
                "success",
                ["type" => "credit", "description" => "Bonus Withdrawal", 'amount' => $request['amount']]
            );


            return ['status' => 'success', 'message' => 'Withdrawal Successful! Wallet Credited.'];
        } else {
            return ['status' => 'failed', 'message' => 'Transaction Failed. Please try again.'];
        }
    }

    public function render_frontend($user, $runcode = "")
    {
        $balance = $user['referral_rewards'] ?? 0;
        $service_id = $this->id;

        ?>
        <div class="row">
            <div class="col-12 mb-4 text-center">
                <div class="card bg-secondary-gradient text-white">
                    <div class="card-body">
                        <h3>Referral Bonus Balance</h3>
                        <h1>₦<?php echo number_format($balance, 2); ?></h1>
                        <small>Funds available to move to main wallet</small>
                    </div>
                </div>
            </div>

            <div class="col-12 col-md-8 mx-auto">
                <form id="withdrawal-form">
                    <div class="form-group mb-3">
                        <label class="form-label fw-bold">Amount to Withdraw</label>
                        <div class="input-group">
                            <span class="input-group-text">₦</span>
                            <input type="number" step="0.01" name="amount" id="withdraw_amount" class="form-control"
                                placeholder="0.00" required max="<?php echo esc_attr($balance); ?>">
                        </div>
                    </div>
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="checkbox" id="withdraw_all">
                        <label class="form-check-label" for="withdraw_all">
                            Withdraw Everything (₦<?php echo number_format($balance, 2); ?>)
                        </label>
                    </div>

                    <button type="submit" class="btn btn-success btn-lg w-100" id="withdraw-btn">Process Withdrawal</button>
                </form>
            </div>
        </div>

        <script>
            jQuery(document).ready(function ($) {
                var maxBalance = <?php echo floatval($balance); ?>;

                $('#withdraw_all').change(function () {
                    if ($(this).is(':checked')) {
                        $('#withdraw_amount').val(maxBalance);
                    } else {
                        $('#withdraw_amount').val('');
                    }
                });

                $('#withdrawal-form').submit(function (e) {
                    e.preventDefault();
                    var amount = $('#withdraw_amount').val();
                    var btn = $('#withdraw-btn');

                    if (amount <= 0) {
                        Swal.fire('Error', 'Invalid Amount', 'error');
                        return;
                    }
                    if (amount > maxBalance) {
                        Swal.fire('Error', 'Insufficient Balance', 'error');
                        return;
                    }

                    Swal.fire({
                        title: 'Confirm Withdrawal?',
                        text: "Move ₦" + amount + " to Main Wallet?",
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonText: 'Yes, Proceed'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            btn.prop('disabled', true).text('Processing...');

                            $.post(vtupressInitPluginData.ajax_url, {
                                action: 'vtupress_process_service',
                                service_id: '<?php echo $service_id; ?>',
                                amount: amount
                            }, function (res) {
                                btn.prop('disabled', false).text('Process Withdrawal');
                                if (res.success) {
                                    Swal.fire('Success', res.data.message, 'success').then(() => {
                                        location.reload();
                                    });
                                } else {
                                    Swal.fire('Failed', res.data.message || 'Error occurred', 'error');
                                }
                            });
                        }
                    });
                });
            });
        </script>
        <?php
    }
}


// add_filter('vtupress_service_providers', function ($providers, $service_id, $country) {
// 	if ($service_id === 'airtime' && $country === 'nigeria') {
// 		// Add a new provider or modify the list
// 		$providers[] = 'zoom-mobile';
// 	}
// 	return $providers;
// }, 10, 3);

class Vtupress_Savings extends Vtupress_services implements Vtupress_History_Interface
{
    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        $this->id = 'savings';
        $this->name = 'Savings';
        $this->description = 'Save money securely with interest';
        $this->countries = ['all'];
        $this->image = VTUPRESS_URL . "/images/service/savings.png";
        $this->uses_api = false;
        $this->uses_provider = false;
        $this->uses_plans = false;
        $this->is_voucher = false;
        $this->uses_api_setup = false;
        $this->disable_global = true;
        $this->icon = 'mdi mdi-piggy-bank';
        $this->custom_id = 'isavings';

        parent::__construct($db, $service_type, $user_id);
    }

    public function get_settings($cache = true, $global = true): array
    {
        return [];
    }

    public function form_fields(): array
    {
        return [];
    }

    public function history_columns(): array
    {
        // return ['type' => 'Type', 'amount' => 'Amount', 'balance_after' => 'Balance', 'description' => 'Description'];
        return [];
    }

    public function resolve_history_column(string $column, array $txn): string
    {
        if ($column == 'type')
            return ucfirst($txn['type'] ?? '-');
        if ($column == 'amount')
            return $txn['amount'] ?? '0.00';
        if ($column == 'balance_after')
            return $txn['balance_after'] ?? '0.00';
        if ($column == 'description')
            return $txn['description'] ?? '-';
        return '-';
    }

    public function maybe_save_settings(): void
    {
        if (isset($_POST['custom_service_status']) && check_admin_referer('vtupress_savings_admin_nonce')) {
            $option_key = 'vtupress_service_' . $this->id . '_' . $this->current_country . '_status';
            Vtupress_Option::update($option_key, sanitize_text_field($_POST['custom_service_status']));
        }

        if (isset($_POST['save_savings_product']) && check_admin_referer('vtupress_savings_admin_nonce')) {
            global $wpdb;
            $table = $wpdb->prefix . 'vtupress_savings_products';
            $product_name = sanitize_text_field($_POST['name']);

            // Check for duplicate name
            $exists = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table WHERE name = %s", $product_name));
            if ($exists > 0) {
                echo '<div class="notice notice-error"><p>Error: A savings product with the name "' . esc_html($product_name) . '" already exists. Please choose a different name.</p></div>';
                return;
            }

            $wpdb->insert($table, [
                'name' => $product_name,
                'type' => sanitize_text_field($_POST['type']),
                'amount' => floatval($_POST['amount']),
                'cycle_value' => intval($_POST['cycle_value']),
                'cycle_type' => sanitize_text_field($_POST['cycle_type']),
                'liquidation_charge' => floatval($_POST['liquidation_charge']),
                'liquidation_charge_type' => sanitize_text_field($_POST['liquidation_charge_type']),
                'withdrawal_charge' => floatval($_POST['withdrawal_charge']),
                'withdrawal_charge_type' => sanitize_text_field($_POST['withdrawal_charge_type']),
                'interest_rate' => floatval($_POST['interest_rate']),
                'status' => sanitize_text_field($_POST['status']),
            ]);
            echo '<div class="notice notice-success"><p>Savings product created successfully!</p></div>';
        }

        if (isset($_POST['delete_savings_product']) && check_admin_referer('vtupress_savings_admin_nonce')) {
            global $wpdb;
            $table = $wpdb->prefix . 'vtupress_savings_products';
            $wpdb->delete($table, ['id' => intval($_POST['product_id'])]);
            echo '<div class="notice notice-success"><p>Savings product deleted successfully!</p></div>';
        }

        if (isset($_POST['update_savings_product']) && check_admin_referer('vtupress_savings_admin_nonce')) {
            global $wpdb;
            $table = $wpdb->prefix . 'vtupress_savings_products';
            $product_id = intval($_POST['product_id']);
            $product_name = sanitize_text_field($_POST['name']);

            // Check for duplicate name (excluding current product)
            $exists = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table WHERE name = %s AND id != %d", $product_name, $product_id));
            if ($exists > 0) {
                echo '<div class="notice notice-error"><p>Error: A savings product with the name "' . esc_html($product_name) . '" already exists. Please choose a different name.</p></div>';
                return;
            }

            $wpdb->update($table, [
                'name' => $product_name,
                'type' => sanitize_text_field($_POST['type']),
                'amount' => floatval($_POST['amount']),
                'cycle_value' => intval($_POST['cycle_value']),
                'cycle_type' => sanitize_text_field($_POST['cycle_type']),
                'liquidation_charge' => floatval($_POST['liquidation_charge']),
                'liquidation_charge_type' => sanitize_text_field($_POST['liquidation_charge_type']),
                'withdrawal_charge' => floatval($_POST['withdrawal_charge']),
                'withdrawal_charge_type' => sanitize_text_field($_POST['withdrawal_charge_type']),
                'interest_rate' => floatval($_POST['interest_rate']),
                'status' => sanitize_text_field($_POST['status']),
            ], ['id' => $product_id]);
            echo '<div class="notice notice-success"><p>Savings product updated successfully!</p></div>';
        }
    }

    public function render_admin_form(): void
    {
        $this->maybe_save_settings();
        global $wpdb;

        $option_key = 'vtupress_service_' . $this->id . '_' . $this->current_country . '_status';
        $current_status = Vtupress_Option::get($option_key, 'off');

        $table = $wpdb->prefix . 'vtupress_savings_products';
        $products = $wpdb->get_results("SELECT * FROM $table ORDER BY id DESC");

        ?>
        <div class="card mb-3">
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center">
                    <h4 class="header-title mb-0">Savings Service Status</h4>
                    <form method="post" class="d-inline">
                        <?php wp_nonce_field('vtupress_savings_admin_nonce'); ?>
                        <input type="hidden" name="custom_service_status"
                            value="<?php echo $current_status == 'on' ? 'off' : 'on'; ?>">
                        <button type="submit" class="btn btn-<?php echo $current_status == 'on' ? 'danger' : 'success'; ?>">
                            <?php echo $current_status == 'on' ? 'Disable Service' : 'Enable Service'; ?>
                        </button>
                    </form>
                </div>
                <p>Current Status: <strong><?php echo strtoupper($current_status); ?></strong></p>
            </div>
        </div>

        <div class="card">
            <div class="card-body">
                <h4 class="header-title">Savings Products</h4>

                <form method="post">
                    <?php wp_nonce_field('vtupress_savings_admin_nonce'); ?>
                    <div class="row">
                        <div class="col-md-4 mb-2">
                            <label>Name</label>
                            <input type="text" name="name" class="form-control" required placeholder="Ex: Daily Thrift">
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>Type</label>
                            <select name="type" class="form-control">
                                <option value="fixed">Fixed</option>
                                <option value="flexible">Flexible</option>
                            </select>
                        </div>
                        <div class="col-md-3 mb-2">
                            <label>Amount (Min if Flexible)</label>
                            <input type="number" step="0.01" name="amount" class="form-control" required>
                        </div>
                        <div class="col-md-3 mb-2">
                            <label>Interest Rate (%)</label>
                            <input type="number" step="0.01" name="interest_rate" class="form-control" value="0">
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-2 mb-2">
                            <label>Cycle Val</label>
                            <input type="number" name="cycle_value" class="form-control" value="30">
                        </div>
                        <div class="col-md-2 mb-2">
                            <label>Cycle Unit</label>
                            <select name="cycle_type" class="form-control">
                                <option value="day">Day(s)</option>
                                <option value="month">Month(s)</option>
                                <option value="year">Year(s)</option>
                            </select>
                        </div>
                        <div class="col-md-4 mb-2">
                            <label>Liquidation Charge</label>
                            <div class="input-group">
                                <input type="number" step="0.01" name="liquidation_charge" class="form-control" value="0">
                                <select name="liquidation_charge_type" class="form-control">
                                    <option value="percent">%</option>
                                    <option value="fixed">Fixed</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-4 mb-2">
                            <label>Withdrawal Charge</label>
                            <div class="input-group">
                                <input type="number" step="0.01" name="withdrawal_charge" class="form-control" value="0">
                                <select name="withdrawal_charge_type" class="form-control">
                                    <option value="percent">%</option>
                                    <option value="fixed">Fixed</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-12 mb-2">
                            <label>Status</label>
                            <select name="status" class="form-control">
                                <option value="active">Active</option>
                                <option value="inactive">Inactive</option>
                            </select>
                        </div>
                    </div>
                    <button type="submit" name="save_savings_product" class="btn btn-primary">Create Product</button>
                </form>

                <hr>

                <h5 class="mt-4">Existing Products</h5>
                <div class="table-responsive">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Type</th>
                                <th>Amount</th>
                                <th>Cycle</th>
                                <th>Interest</th>
                                <th>Charges (L/W)</th>
                                <th>Status</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($products as $prod): ?>
                                <tr>
                                    <form method="post">
                                        <?php wp_nonce_field('vtupress_savings_admin_nonce'); ?>
                                        <input type="hidden" name="product_id" value="<?php echo $prod->id; ?>">
                                        <td><input type="text" name="name" value="<?php echo esc_attr($prod->name); ?>"
                                                class="form-control form-control-sm"></td>
                                        <td>
                                            <select name="type" class="form-control form-control-sm">
                                                <option value="fixed" <?php selected($prod->type, 'fixed'); ?>>Fixed</option>
                                                <option value="flexible" <?php selected($prod->type, 'flexible'); ?>>Flexible
                                                </option>
                                            </select>
                                        </td>
                                        <td><input type="number" step="0.01" name="amount" value="<?php echo $prod->amount; ?>"
                                                class="form-control form-control-sm"></td>
                                        <td>
                                            <div class="input-group input-group-sm">
                                                <input type="number" name="cycle_value" value="<?php echo $prod->cycle_value; ?>"
                                                    class="form-control">
                                                <select name="cycle_type" class="form-control">
                                                    <option value="day" <?php selected($prod->cycle_type, 'day'); ?>>D</option>
                                                    <option value="month" <?php selected($prod->cycle_type, 'month'); ?>>M</option>
                                                    <option value="year" <?php selected($prod->cycle_type, 'year'); ?>>Y</option>
                                                </select>
                                            </div>
                                        </td>
                                        <td><input type="number" step="0.01" name="interest_rate"
                                                value="<?php echo $prod->interest_rate; ?>" class="form-control form-control-sm">
                                        </td>
                                        <td>
                                            <small>
                                                L:
                                                <?php echo $prod->liquidation_charge . ($prod->liquidation_charge_type == 'percent' ? '%' : ''); ?><br>
                                                W:
                                                <?php echo $prod->withdrawal_charge . ($prod->withdrawal_charge_type == 'percent' ? '%' : ''); ?>
                                            </small>
                                            <input type="hidden" name="liquidation_charge"
                                                value="<?php echo $prod->liquidation_charge; ?>">
                                            <input type="hidden" name="liquidation_charge_type"
                                                value="<?php echo $prod->liquidation_charge_type; ?>">
                                            <input type="hidden" name="withdrawal_charge"
                                                value="<?php echo $prod->withdrawal_charge; ?>">
                                            <input type="hidden" name="withdrawal_charge_type"
                                                value="<?php echo $prod->withdrawal_charge_type; ?>">
                                        </td>
                                        <td>
                                            <select name="status" class="form-control form-control-sm">
                                                <option value="active" <?php selected($prod->status, 'active'); ?>>Active</option>
                                                <option value="inactive" <?php selected($prod->status, 'inactive'); ?>>Inactive
                                                </option>
                                            </select>
                                        </td>
                                        <td>
                                            <button type="submit" name="update_savings_product"
                                                class="btn btn-sm btn-success">Update</button>
                                            <button type="submit" name="delete_savings_product" class="btn btn-sm btn-danger"
                                                onclick="return confirm('Are you sure?')">Del</button>
                                        </td>
                                    </form>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php
    }

    public function process_frontend_request($request)
    {
        if (!is_user_logged_in()) {
            return ['status' => 'failed', 'message' => 'Login required'];
        }

        global $wpdb;
        $wp_user_id = get_current_user_id();

        $users = new Users();
        $userData = $users->getUser($wp_user_id, true, true);
        if (!$userData)
            return ['status' => 'failed', 'message' => 'User profile not found'];
        $user_id = $userData["id"]; // VTUPress User ID

        $sub_action = $request['sub_action'] ?? '';
        $products_table = $wpdb->prefix . 'vtupress_savings_products';
        $accounts_table = $wpdb->prefix . 'vtupress_savings_accounts';
        $trans_table = $wpdb->prefix . 'vtupress_savings_transactions';
        $vp_trans_table = $wpdb->prefix . 'vtupress_transactions';

        if ($sub_action == 'start_saving' || $sub_action == 'deposit') {
            $product_id = intval($request['product_id']);
            $account_id = intval($request['account_id'] ?? 0);
            $amount = floatval($request['amount']);

            // If depositing, we might not have product_id but we have account_id
            if ($product_id <= 0 && $account_id > 0) {
                $account = $wpdb->get_row("SELECT * FROM $accounts_table WHERE id = $account_id AND user_id = $user_id");
                if ($account) {
                    $product_id = $account->product_id;
                } else {
                    return ['status' => 'failed', 'message' => 'Invalid Saving Account'];
                }
            }

            // 1. Validate Product
            $product = $wpdb->get_row("SELECT * FROM $products_table WHERE id = $product_id");
            if (!$product || $product->status != 'active')
                return ['status' => 'failed', 'message' => 'Invalid or Inactive product (ID: ' . $product_id . ')'];

            // 2. Validate Amount
            if ($amount <= 0)
                return ['status' => 'failed', 'message' => "Amount must be greater than zero"];

            if ($product->type == 'fixed' && $amount != $product->amount)
                return ['status' => 'failed', 'message' => "Fixed Amount must be ₦{$product->amount}"];
            if ($product->type == 'flexible' && $amount < $product->amount)
                return ['status' => 'failed', 'message' => "Minimum amount is ₦{$product->amount}"];

            // 3. Check Balance
            $balance = $users->getBalance($user_id);
            if ($balance < $amount)
                return ['status' => 'failed', 'message' => 'Insufficient wallet balance'];

            // 4. Determine Account (Create if new, or use existing for deposit)
            // If start_saving, we create new
            // If deposit, we use existing

            $existing_account = $wpdb->get_row("SELECT * FROM $accounts_table WHERE user_id = $user_id AND product_id = $product_id AND status = 'active' ORDER BY id DESC LIMIT 1");

            $account_id = 0;
            $is_new = false;
            $description = "";

            if ($sub_action == 'start_saving') {
                if ($existing_account)
                    return ['status' => 'failed', 'message' => 'You already have an active savings of this type. Use Deposit to add funds.'];

                $description = "Savings Sub: {$product->name}";
                // Debit User
                $debit_txrn_id = $users->debit($user_id, $amount, 'savings', $description, ['description' => $description]);
                if (!$debit_txrn_id) {
                    return ['status' => 'failed', 'message' => 'Debit failed'];
                }

                $start_date = current_time('mysql');
                $end_date = date('Y-m-d H:i:s', strtotime("+{$product->cycle_value} {$product->cycle_type}", strtotime($start_date)));

                $wpdb->insert($accounts_table, [
                    'user_id' => $user_id,
                    'product_id' => $product_id,
                    'balance' => $amount,
                    'status' => 'active',
                    'start_date' => $start_date,
                    'end_date' => $end_date,
                    'last_saving_date' => $start_date
                ]);
                $account_id = $wpdb->insert_id;
                $is_new = true;
            } else {
                // Deposit
                if (!$existing_account)
                    return ['status' => 'failed', 'message' => 'No active savings found. Start Saving first.'];
                $account_id = $existing_account->id;

                $description = "Savings Deposit: {$product->name}";
                $debit_txrn_id = $users->debit($user_id, $amount, 'savings', $description);
                // Debit User
                if (!$debit_txrn_id) {
                    return ['status' => 'failed', 'message' => 'Debit failed'];
                }

                // Update Balance
                $wpdb->query("UPDATE $accounts_table SET balance = balance + $amount, last_saving_date = NOW() WHERE id = $account_id");
            }

            // Log Transaction (Savings History)
            $wpdb->insert($trans_table, [
                'account_id' => $account_id,
                'user_id' => $user_id,
                'type' => $is_new ? 'subscription' : 'deposit',
                'amount' => $amount,
                'balance_after' => ($is_new ? $amount : ($existing_account->balance + $amount)),
                'description' => $is_new ? "Started {$product->name}" : "Deposit to {$product->name}"
            ]);

            $wpdb->update($vp_trans_table, ['details' => json_encode(['type' => $is_new ? 'subscription' : 'deposit', 'description' => $description, 'balance_after' => ($is_new ? $amount : ($existing_account->balance + $amount))])], ['id' => $debit_txrn_id]);
            return ['status' => 'success', 'message' => 'Transaction Successful!'];
        }

        if ($sub_action == 'withdraw' || $sub_action == 'liquidate') {
            $account_id = intval($request['account_id']);
            $amount = floatval($request['amount']); // If partial withdrawal allowed? For now assuming full withdrawal/liquidation based on user request "withdrawal charge e.g 2 i.e it will remove 2 from the total wallet balance i.e if total saving + principal ... is 10 then 2 is taken"
            // Wait, usually liquidation means "Taking ALL money before maturity". Withdrawal might mean "Taking mature money" OR partial.
            // User said: "liquidation charge ... withdrawal charge e.g 2 i.e it will remove 2 from the total wallet balance"

            $account = $wpdb->get_row("SELECT * FROM $accounts_table WHERE id = $account_id AND user_id = $user_id");
            if (!$account || $account->status != 'active')
                return ['status' => 'failed', 'message' => 'Invalid Account'];

            $product = $wpdb->get_row("SELECT * FROM $products_table WHERE id = {$account->product_id}");

            $is_mature = (strtotime($account->end_date) <= current_time('timestamp'));
            $is_liquidation = ($sub_action == 'liquidate');

            if ($is_liquidation && $is_mature) {
                // If it's mature, it's a withdrawal, not liquidation
                $is_liquidation = false;
                $sub_action = 'withdraw';
            }

            if (!$is_liquidation && !$is_mature) {
                return ['status' => 'failed', 'message' => 'Savings not yet mature. Use Liquidate option to withdraw early (with penalty).'];
            }

            // Calculate Charges
            $total_saved = floatval($account->balance);
            $charge = 0;

            if ($is_liquidation) {
                // Liquidation Charge
                $charge_val = floatval($product->liquidation_charge);
                if ($product->liquidation_charge_type == 'percent') {
                    $charge = ($total_saved * $charge_val) / 100;
                } else {
                    $charge = $charge_val;
                }
            } else {
                // Withdrawal Charge
                $charge_val = floatval($product->withdrawal_charge);
                if ($product->withdrawal_charge_type == 'percent') {
                    $charge = ($total_saved * $charge_val) / 100;
                } else {
                    $charge = $charge_val;
                }
            }

            // Interest (Simple Interest calculation for now, applied at end)
            // User didn't specify interest logic explicitly but mentioned "principal in the savings wallet".
            // I added 'interest_rate' to product.
            $interest = 0;
            if ($product->interest_rate > 0) {
                // Calculate interest based on duration held? Or fixed % of total?
                // Assuming flat % for the cycle.
                $interest = ($total_saved * floatval($product->interest_rate)) / 100;

                // If liquidation, maybe forfeit interest? 
                if ($is_liquidation)
                    $interest = 0;
            }

            $final_amount = $total_saved + $interest - $charge;
            if ($final_amount < 0)
                $final_amount = 0;

            // Update Account Status
            $wpdb->update($accounts_table, ['status' => ($is_liquidation ? 'liquidated' : 'completed'), 'balance' => 0], ['id' => $account_id]);

            // Credit User
            if ($final_amount > 0) {
                // $users->credit is not standard function in generic Users class usually, it is refund or similar.
                // vtupress_users table has balance.
                // manual update query or refund method?
                // functions.php uses `refund`.
                $users->refund($user_id, $final_amount, "Savings " . ucfirst($sub_action));
            }

            // Log
            $wpdb->insert($trans_table, [
                'account_id' => $account_id,
                'user_id' => $user_id,
                'type' => $sub_action,
                'amount' => $final_amount, // Amount paid out
                'balance_after' => 0,
                'description' => ucfirst($sub_action) . " (Principal: $total_saved, Interest: $interest, Charge: $charge)"
            ]);

            return ['status' => 'success', 'message' => "Successfully " . ucfirst($sub_action) . "d ₦$final_amount"];
        }

        if ($sub_action == 'setup_auto_deduct') {
            $account_id = intval($request['account_id']);
            $enabled = sanitize_text_field($request['enabled'] ?? 'no');
            $amount = floatval($request['amount'] ?? 0);
            $frequency = sanitize_text_field($request['frequency'] ?? '');

            $account = $wpdb->get_row("SELECT * FROM $accounts_table WHERE id = $account_id AND user_id = $user_id");
            if (!$account || $account->status != 'active')
                return ['status' => 'failed', 'message' => 'Invalid Account'];

            $update_data = [
                'auto_deduct_enabled' => $enabled
            ];

            if ($enabled == 'yes') {
                if ($amount <= 0)
                    return ['status' => 'failed', 'message' => 'Amount must be greater than zero'];
                if (!in_array($frequency, ['daily', 'weekly', 'monthly', 'yearly'])) {
                    return ['status' => 'failed', 'message' => 'Invalid frequency'];
                }

                $update_data['auto_deduct_amount'] = $amount;
                $update_data['auto_deduct_frequency'] = $frequency;
                $update_data['next_auto_deduct_date'] = self::calculate_next_deduct_date($frequency);
            } else {
                $update_data['auto_deduct_amount'] = 0;
                $update_data['auto_deduct_frequency'] = null;
                $update_data['next_auto_deduct_date'] = null;
            }

            $wpdb->update($accounts_table, $update_data, ['id' => $account_id]);

            $message = $enabled == 'yes'
                ? "Automated savings configured! ₦$amount will be deducted $frequency."
                : "Automated savings disabled.";
            return ['status' => 'success', 'message' => $message];
        }

        return ['status' => 'failed', 'message' => 'Unknown action'];
    }

    /**
     * Cron job callback for processing automated savings
     */
    public static function process_automated_savings_cron()
    {
        global $wpdb;
        $accounts_table = $wpdb->prefix . 'vtupress_savings_accounts';
        $trans_table = $wpdb->prefix . 'vtupress_savings_transactions';

        // Get all accounts with auto-deduct enabled that are due
        $accounts = $wpdb->get_results(
            "SELECT a.*, p.name as product_name 
            FROM $accounts_table a
            JOIN {$wpdb->prefix}vtupress_savings_products p ON a.product_id = p.id
            WHERE a.auto_deduct_enabled = 'yes' 
            AND a.status = 'active'
            AND a.next_auto_deduct_date <= NOW()"
        );

        $users = new Users();

        foreach ($accounts as $acc) {
            $amount = floatval($acc->auto_deduct_amount);
            if ($amount <= 0)
                continue;

            // Check if user has sufficient balance
            $balance = $users->getBalance($acc->user_id);
            if ($balance < $amount) {
                // Log failed attempt
                $wpdb->insert($trans_table, [
                    'account_id' => $acc->id,
                    'user_id' => $acc->user_id,
                    'type' => 'auto_deposit_failed',
                    'amount' => 0,
                    'balance_after' => $acc->balance,
                    'description' => "Automated deposit failed: Insufficient balance (Need: ₦$amount)"
                ]);
                continue;
            }

            // Debit user
            if ($users->debit($acc->user_id, $amount, 'savings', "Auto-Deposit: {$acc->product_name}", ['description' => "Auto-Deposit: {$acc->product_name}"])) {
                // Update account balance
                $new_balance = floatval($acc->balance) + $amount;
                $wpdb->update($accounts_table, [
                    'balance' => $new_balance,
                    'last_saving_date' => current_time('mysql')
                ], ['id' => $acc->id]);

                // Log transaction
                $wpdb->insert($trans_table, [
                    'account_id' => $acc->id,
                    'user_id' => $acc->user_id,
                    'type' => 'auto_deposit',
                    'amount' => $amount,
                    'balance_after' => $new_balance,
                    'description' => "Automated deposit: {$acc->product_name}"
                ]);
            }

            // Calculate next deduct date
            $next_date = self::calculate_next_deduct_date($acc->auto_deduct_frequency);
            $wpdb->update($accounts_table, [
                'next_auto_deduct_date' => $next_date
            ], ['id' => $acc->id]);
        }
    }

    /**
     * Calculate next auto-deduct date based on frequency
     */
    private static function calculate_next_deduct_date($frequency)
    {
        $now = current_time('timestamp');
        switch ($frequency) {
            case 'daily':
                return date('Y-m-d H:i:s', strtotime('+1 day', $now));
            case 'weekly':
                return date('Y-m-d H:i:s', strtotime('+1 week', $now));
            case 'monthly':
                return date('Y-m-d H:i:s', strtotime('+1 month', $now));
            case 'yearly':
                return date('Y-m-d H:i:s', strtotime('+1 year', $now));
            default:
                return null;
        }
    }

    public function render_frontend($user, $runcode = "")
    {
        global $wpdb;
        $user_id = $user['id'];

        $products_table = $wpdb->prefix . 'vtupress_savings_products';
        $accounts_table = $wpdb->prefix . 'vtupress_savings_accounts';

        $products = $wpdb->get_results("SELECT * FROM $products_table WHERE status = 'active'");
        $accounts = $wpdb->get_results("SELECT a.*, p.name as product_name, p.cycle_value, p.cycle_type, p.type as product_type, p.interest_rate, p.liquidation_charge, p.liquidation_charge_type, p.withdrawal_charge, p.withdrawal_charge_type, p.amount as product_amount
                                        FROM $accounts_table a 
                                        JOIN $products_table p ON a.product_id = p.id 
                                        WHERE a.user_id = $user_id AND a.status = 'active' ORDER BY a.id DESC");

        // Calculate summary statistics
        $total_saved = 0;
        $total_expected = 0;
        $active_count = count($accounts);

        foreach ($accounts as $acc) {
            $total_saved += floatval($acc->balance);
            $interest = ($acc->balance * floatval($acc->interest_rate)) / 100;
            $is_mature = (strtotime($acc->end_date) <= current_time('timestamp'));
            $charge = 0;
            if (!$is_mature) {
                // Liquidation
                if ($acc->liquidation_charge_type == 'percent') {
                    $charge = ($acc->balance * floatval($acc->liquidation_charge)) / 100;
                } else {
                    $charge = floatval($acc->liquidation_charge);
                }
            } else {
                // Withdrawal
                if ($acc->withdrawal_charge_type == 'percent') {
                    $charge = ($acc->balance * floatval($acc->withdrawal_charge)) / 100;
                } else {
                    $charge = floatval($acc->withdrawal_charge);
                }
            }
            $total_expected += ($acc->balance + $interest - $charge);
        }

        ?>
        <style>
            .savings-container {
                background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            }

            .savings-summary-box {
                border-radius: 16px;
                padding: 25px;
                background: rgba(255, 255, 255, 0.98);
                backdrop-filter: blur(15px);
                -webkit-backdrop-filter: blur(15px);
                border: 1px solid rgba(255, 255, 255, 0.5);
                box-shadow: 0 8px 24px rgba(0, 0, 0, 0.12);
                transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            }

            .savings-summary-box:hover {
                transform: translateY(-5px);
                box-shadow: 0 12px 32px rgba(0, 0, 0, 0.18);
            }

            .savings-summary-box .summary-icon {
                width: 50px;
                height: 50px;
                border-radius: 12px;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                font-size: 24px;
                margin-bottom: 12px;
                color: white;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            }

            .savings-summary-box.purple .summary-icon {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }

            .savings-summary-box.pink .summary-icon {
                background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            }

            .savings-summary-box.blue .summary-icon {
                background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            }

            .savings-summary-box h3 {
                font-size: 1.6rem;
                font-weight: 700;
                margin-bottom: 5px;
                color: #2d3748;
            }

            .savings-summary-box p {
                font-size: 0.75rem;
                color: #718096;
                margin-bottom: 0;
                font-weight: 500;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }

            .savings-card {
                transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
                cursor: pointer;
                border: none !important;
                box-shadow: 0 6px 20px rgba(0, 0, 0, 0.12);
                border-radius: 16px;
                overflow: hidden;
                background: white;
            }

            .savings-card:hover {
                transform: translateY(-10px);
                box-shadow: 0 16px 32px rgba(0, 0, 0, 0.2);
            }

            .savings-gradient-1 {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            }

            .savings-gradient-2 {
                background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            }

            .savings-gradient-3 {
                background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
            }

            .savings-gradient-4 {
                background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);
            }

            .plan-badge {
                display: inline-block;
                padding: 5px 14px;
                border-radius: 20px;
                font-size: 0.7rem;
                font-weight: 600;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }

            .icon-wrapper {
                width: 60px;
                height: 60px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 28px;
                margin-bottom: 15px;
                background: rgba(255, 255, 255, 0.2);
            }
        </style>

        <script>
            function processSavings(action, id, amount = 0, accountId = 0) {
                Swal.fire({
                    title: 'Processing...',
                    text: 'Please wait',
                    icon: 'info',
                    allowOutsideClick: false,
                    showConfirmButton: false
                });

                jQuery.post(vtupressInitPluginData.ajax_url, {
                    action: 'vtupress_process_service',
                    service_id: 'savings',
                    sub_action: action,
                    product_id: id,
                    account_id: accountId,
                    amount: amount
                }, function (res) {
                    if (res.success) {
                        Swal.fire('Success', res.data.message, 'success').then(() => location.reload());
                    } else {
                        Swal.fire('Failed', res.data.message || 'Error occurred', 'error');
                    }
                });
            }

            function openStartModal(id, name, type, min_amount) {
                const isFixed = type === 'fixed';
                Swal.fire({
                    title: 'Start Saving: ' + name,
                    html: `
                        <div style="text-align: left; padding: 10px;">
                            <p><strong>Plan Type:</strong> <span class="badge bg-info">${type.toUpperCase()}</span></p>
                            <p><strong>${isFixed ? 'Fixed Amount' : 'Minimum Amount'}:</strong> ₦${Number(min_amount).toLocaleString()}</p>
                            <div class="form-group mt-3">
                                <label for="save_amount"><strong>Amount to Save:</strong></label>
                                <input type="number" id="save_amount" class="swal2-input" placeholder="Amount" value="${min_amount}" ${isFixed ? 'readonly' : ''} style="${isFixed ? 'background: #f0f0f0; cursor: not-allowed;' : ''}">
                                ${isFixed ? '<small class="text-muted">Fixed amount cannot be changed</small>' : '<small class="text-muted">You can save more than the minimum</small>'}
                            </div>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: '💰 Start Saving',
                    confirmButtonColor: '#667eea',
                    preConfirm: () => {
                        const amount = Swal.getPopup().querySelector('#save_amount').value;
                        if (!amount || amount < min_amount && type == 'flexible') {
                            Swal.showValidationMessage(`Minimum amount is ₦${Number(min_amount).toLocaleString()}`);
                        }
                        if (amount != min_amount && type == 'fixed') {
                            Swal.showValidationMessage(`Fixed amount must be ₦${Number(min_amount).toLocaleString()}`);
                        }
                        return { amount: amount }
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        processSavings('start_saving', id, result.value.amount, 0);
                    }
                });
            }

            function openDepositModal(accId, name, currentBalance) {
                Swal.fire({
                    title: 'Deposit to ' + name,
                    html: `
                        <div style="text-align: left; padding: 10px;">
                            <p><strong>Current Balance:</strong> ₦${Number(currentBalance).toLocaleString()}</p>
                            <div class="form-group mt-3">
                                <label for="deposit_amount"><strong>Amount to Deposit:</strong></label>
                                <input type="number" id="deposit_amount" class="swal2-input" placeholder="Enter amount" min="1">
                            </div>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: '💵 Deposit Now',
                    confirmButtonColor: '#43e97b',
                    preConfirm: () => {
                        const amount = Swal.getPopup().querySelector('#deposit_amount').value;
                        if (!amount || amount <= 0) {
                            Swal.showValidationMessage('Please enter a valid amount');
                        }
                        return { amount: amount }
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        processSavings('deposit', 0, result.value.amount, accId);
                    }
                });
            }

            function openManageModal(accId, name, balance, productType, endDate, productId, autoEnabled, autoAmount, autoFreq) {
                const now = new Date();
                const end = new Date(endDate);
                const isMature = now >= end;

                let buttons = '';
                if (productType === 'flexible') {
                    buttons += `<button onclick="Swal.close(); openDepositModal(${accId}, '${name}', ${balance});" class="btn btn-info m-1"><i class="mdi mdi-plus-circle"></i> Deposit</button>`;
                }

                if (isMature) {
                    buttons += `<button onclick="processSavings('withdraw', 0, 0, ${accId});" class="btn btn-success m-1"><i class="mdi mdi-cash-multiple"></i> Withdraw (Mature)</button>`;
                } else {
                    buttons += `<button onclick="confirmLiquidate(${accId});" class="btn btn-warning m-1"><i class="mdi mdi-alert"></i> Liquidate (Early)</button>`;
                }

                // Auto-save button
                const autoLabel = autoEnabled === 'yes' ? '⚡ Auto-Save (ON)' : '🔧 Setup Auto-Save';
                const autoBtnClass = autoEnabled === 'yes' ? 'btn-dark' : 'btn-secondary';
                buttons += `<button onclick="Swal.close(); setupAutoSave(${accId}, '${name}', '${autoEnabled}', ${autoAmount}, '${autoFreq}');" class="btn ${autoBtnClass} m-1">${autoLabel}</button>`;

                Swal.fire({
                    title: 'Manage: ' + name,
                    html: `
                        <div style="text-align: left; padding: 15px;">
                            <div class="mb-3">
                                <p><strong>Current Balance:</strong> <span class="badge bg-success" style="font-size: 1.1rem;">₦${Number(balance).toLocaleString()}</span></p>
                                <p><strong>Maturity Date:</strong> ${new Date(endDate).toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })}</p>
                                <p><strong>Status:</strong> ${isMature ? '<span class="badge bg-success">Mature ✓</span>' : '<span class="badge bg-info">Ongoing</span>'}</p>
                                ${autoEnabled === 'yes' ? `<p><strong>Auto-Deduct:</strong> <span class="badge bg-dark">₦${Number(autoAmount).toLocaleString()} ${autoFreq}</span></p>` : ''}
                            </div>
                            <div class="d-flex justify-content-center flex-wrap">${buttons}</div>
                        </div>
                    `,
                    showConfirmButton: false,
                    showCancelButton: true,
                    cancelButtonText: 'Close',
                    width: '550px'
                });
            }

            function setupAutoSave(accId, name, currentlyEnabled, currentAmount, currentFreq) {
                const isEnabled = currentlyEnabled === 'yes';

                Swal.fire({
                    title: '⚡ Automated Savings',
                    html: `
                        <div style="text-align: left; padding: 10px;">
                            <p><strong>Plan:</strong> ${name}</p>
                            <hr>
                            <div class="form-check form-switch mb-3">
                                <input class="form-check-input" type="checkbox" id="auto_enable" ${isEnabled ? 'checked' : ''}>
                                <label class="form-check-label" for="auto_enable">
                                    <strong>Enable Automated Savings</strong>
                                </label>
                            </div>
                            <div id="auto_settings" style="${isEnabled ? '' : 'display: none;'}">
                                <div class="mb-3">
                                    <label><strong>Amount to Deduct:</strong></label>
                                    <input type="number" id="auto_amount" class="swal2-input" placeholder="Amount" value="${currentAmount || ''}" min="1">
                                </div>
                                <div class="mb-3">
                                    <label><strong>Frequency:</strong></label>
                                    <select id="auto_frequency" class="swal2-input">
                                        <option value="daily" ${currentFreq === 'daily' ? 'selected' : ''}>Daily</option>
                                        <option value="weekly" ${currentFreq === 'weekly' ? 'selected' : ''}>Weekly</option>
                                        <option value="monthly" ${currentFreq === 'monthly' ? 'selected' : ''}>Monthly</option>
                                        <option value="yearly" ${currentFreq === 'yearly' ? 'selected' : ''}>Yearly</option>
                                    </select>
                                </div>
                                <p class="text-muted small"><i class="mdi mdi-information"></i> Funds will be automatically debited from your wallet balance.</p>
                            </div>
                        </div>
                    `,
                    showCancelButton: true,
                    confirmButtonText: '💾 Save Configuration',
                    confirmButtonColor: '#667eea',
                    didOpen: () => {
                        document.getElementById('auto_enable').addEventListener('change', function () {
                            document.getElementById('auto_settings').style.display = this.checked ? 'block' : 'none';
                        });
                    },
                    preConfirm: () => {
                        const enabled = document.getElementById('auto_enable').checked ? 'yes' : 'no';
                        const amount = document.getElementById('auto_amount').value;
                        const frequency = document.getElementById('auto_frequency').value;

                        if (enabled === 'yes' && (!amount || amount <= 0)) {
                            Swal.showValidationMessage('Please enter a valid amount');
                        }

                        return { enabled, amount, frequency }
                    }
                }).then((result) => {
                    if (result.isConfirmed) {
                        Swal.fire({
                            title: 'Processing...',
                            text: 'Please wait',
                            icon: 'info',
                            allowOutsideClick: false,
                            showConfirmButton: false
                        });

                        jQuery.post(vtupressInitPluginData.ajax_url, {
                            action: 'vtupress_process_service',
                            service_id: 'savings',
                            sub_action: 'setup_auto_deduct',
                            account_id: accId,
                            enabled: result.value.enabled,
                            amount: result.value.amount,
                            frequency: result.value.frequency
                        }, function (res) {
                            if (res.success) {
                                Swal.fire('Success', res.data.message, 'success').then(() => location.reload());
                            } else {
                                Swal.fire('Failed', res.data.message || 'Error occurred', 'error');
                            }
                        });
                    }
                });
            }

            function confirmLiquidate(accId) {
                Swal.fire({
                    title: 'Confirm Early Withdrawal',
                    html: '<p>You are withdrawing before maturity. <strong>Liquidation charges will apply!</strong></p><p>Are you sure you want to continue?</p>',
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#d33',
                    confirmButtonText: 'Yes, Liquidate',
                    cancelButtonText: 'Cancel'
                }).then((result) => {
                    if (result.isConfirmed) {
                        processSavings('liquidate', 0, 0, accId);
                    }
                });
            }
        </script>

        <!-- Savings Container with Padding -->
        <div class="savings-container"
            style="padding: 20px; background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%); min-height: 100vh;">
            <div class="container-fluid">

                <!-- Summary Cards -->
                <div class="row mb-4">
                    <div class="col-md-4 mb-3">
                        <div class="savings-summary-box purple">
                            <div class="summary-icon">
                                <i class="mdi mdi-piggy-bank"></i>
                            </div>
                            <h3>₦<?php echo number_format($total_saved, 2); ?></h3>
                            <p>Total Saved</p>
                        </div>
                    </div>
                    <div class="col-md-4 mb-3">
                        <div class="savings-summary-box pink">
                            <div class="summary-icon">
                                <i class="mdi mdi-cash-multiple"></i>
                            </div>
                            <h3>₦<?php echo number_format($total_expected, 2); ?></h3>
                            <p>Expected Cashout</p>
                        </div>
                    </div>
                    <div class="col-md-4 mb-3">
                        <div class="savings-summary-box blue">
                            <div class="summary-icon">
                                <i class="mdi mdi-chart-line"></i>
                            </div>
                            <h3><?php echo $active_count; ?></h3>
                            <p>Active Plans</p>
                        </div>
                    </div>
                </div>

                <!-- My Savings Section -->
                <div class="row">
                    <div class="col-md-12 mb-4">
                        <h4 class="mb-3"><i class="mdi mdi-wallet"></i> My Savings Portfolio</h4>
                        <?php if (empty($accounts)): ?>
                            <div class="alert alert-light border text-center py-5">
                                <i class="mdi mdi-information-outline" style="font-size: 3rem; color: #ccc;"></i>
                                <p class="mt-3 mb-0">You have no active savings. Browse plans below to start building your wealth!
                                </p>
                            </div>
                        <?php else: ?>
                            <div class="row">
                                <?php
                                $gradients = ['savings-gradient-1', 'savings-gradient-2', 'savings-gradient-3', 'savings-gradient-4'];
                                $gradient_index = 0;
                                foreach ($accounts as $acc):
                                    $is_mature = (strtotime($acc->end_date) <= current_time('timestamp'));
                                    $days_left = max(0, ceil((strtotime($acc->end_date) - current_time('timestamp')) / (60 * 60 * 24)));
                                    ?>
                                    <div class="col-md-6 col-lg-4 mb-3">
                                        <div class="card savings-card h-100">
                                            <div class="card-body">
                                                <div class="d-flex justify-content-between align-items-start mb-3">
                                                    <div>
                                                        <h5 class="card-title mb-1"><?php echo esc_html($acc->product_name); ?></h5>
                                                        <span
                                                            class="plan-badge <?php echo $acc->product_type == 'fixed' ? 'bg-danger' : 'bg-primary'; ?>">
                                                            <?php echo ucfirst($acc->product_type); ?>
                                                        </span>
                                                    </div>
                                                    <i class="mdi mdi-piggy-bank-outline" style="font-size: 2rem; color: #667eea;"></i>
                                                </div>

                                                <div class="mb-3">
                                                    <div class="d-flex justify-content-between mb-2">
                                                        <span class="text-muted">Balance:</span>
                                                        <strong
                                                            style="font-size: 1.2rem; color: #43e97b;">₦<?php echo number_format($acc->balance, 2); ?></strong>
                                                    </div>
                                                    <div class="d-flex justify-content-between mb-2">
                                                        <span class="text-muted">Interest Rate:</span>
                                                        <strong><?php echo $acc->interest_rate; ?>%</strong>
                                                    </div>
                                                    <div class="d-flex justify-content-between mb-2">
                                                        <span class="text-muted">Matures On:</span>
                                                        <span><?php echo date('M d, Y', strtotime($acc->end_date)); ?></span>
                                                    </div>
                                                    <div class="d-flex justify-content-between">
                                                        <span class="text-muted">Status:</span>
                                                        <?php if ($is_mature): ?>
                                                            <span class="badge bg-success">✓ Mature</span>
                                                        <?php else: ?>
                                                            <span class="badge bg-info"><?php echo $days_left; ?> days left</span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>

                                                <button class="btn btn-primary w-100 rounded-pill"
                                                    onclick="openManageModal(<?php echo $acc->id; ?>, '<?php echo esc_js($acc->product_name); ?>', '<?php echo $acc->balance; ?>', '<?php echo $acc->product_type; ?>', '<?php echo $acc->end_date; ?>', <?php echo $acc->product_id; ?>, '<?php echo $acc->auto_deduct_enabled ?? 'no'; ?>', <?php echo $acc->auto_deduct_amount ?? 0; ?>, '<?php echo $acc->auto_deduct_frequency ?? ''; ?>')">
                                                    <i class="mdi mdi-cog"></i> Manage Savings
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                    <?php
                                    $gradient_index = ($gradient_index + 1) % count($gradients);
                                endforeach;
                                ?>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Available Plans Section -->
                    <div class="col-md-12">
                        <h4 class="mb-3"><i class="mdi mdi-shopping"></i> Browse Savings Plans</h4>
                        <div class="row">
                            <?php
                            if (empty($products)): ?>
                                <div class="col-12">
                                    <div class="alert alert-warning text-center">
                                        <i class="mdi mdi-alert-circle-outline" style="font-size: 2rem;"></i>
                                        <p class="mt-2 mb-0">No savings plans are currently available. Please check back later!</p>
                                    </div>
                                </div>
                            <?php else:
                                $gradients = ['savings-gradient-1', 'savings-gradient-2', 'savings-gradient-3', 'savings-gradient-4'];
                                foreach ($products as $index => $prod):
                                    $gradient_class = $gradients[$index % count($gradients)];
                                    ?>
                                    <div class="col-md-6 col-lg-4 mb-3">
                                        <div class="card savings-card h-100 border-0">
                                            <div class="<?php echo $gradient_class; ?> text-white p-4">
                                                <div class="text-center mb-3">
                                                    <div class="icon-wrapper mx-auto">
                                                        <i class="mdi mdi-wallet-giftcard"></i>
                                                    </div>
                                                    <h5 class="mb-2"><?php echo esc_html($prod->name); ?></h5>
                                                    <span class="plan-badge bg-white text-dark"><?php echo ucfirst($prod->type); ?>
                                                        Plan</span>
                                                </div>
                                            </div>
                                            <div class="card-body">
                                                <ul class="list-unstyled mb-4">
                                                    <li class="mb-2">
                                                        <i class="mdi mdi-check-circle text-success me-2"></i>
                                                        <strong><?php echo $prod->type == 'fixed' ? 'Amount:' : 'Min Amount:'; ?></strong>
                                                        ₦<?php echo number_format($prod->amount, 2); ?>
                                                    </li>
                                                    <li class="mb-2">
                                                        <i class="mdi mdi-clock-outline text-info me-2"></i>
                                                        <strong>Duration:</strong>
                                                        <?php echo $prod->cycle_value . ' ' . ucfirst($prod->cycle_type) . ($prod->cycle_value > 1 ? 's' : ''); ?>
                                                    </li>
                                                    <li class="mb-2">
                                                        <i class="mdi mdi-trending-up text-primary me-2"></i>
                                                        <strong>Interest:</strong> <?php echo $prod->interest_rate; ?>%
                                                    </li>
                                                    <?php if ($prod->type == 'flexible'): ?>
                                                        <li class="mb-2">
                                                            <i class="mdi mdi-plus-circle text-success me-2"></i>
                                                            <small class="text-muted">Top-up anytime</small>
                                                        </li>
                                                    <?php endif; ?>
                                                </ul>

                                                <button class="btn btn-outline-primary w-100 rounded-pill"
                                                    onclick="openStartModal(<?php echo $prod->id; ?>, '<?php echo esc_js($prod->name); ?>', '<?php echo $prod->type; ?>', <?php echo $prod->amount; ?>)">
                                                    <i class="mdi mdi-rocket"></i> Start Saving Now
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div><!-- End Savings Container -->
        <?php
    }
}

class ng_verification extends Vtupress_services
{
    public string $id = 'ng_verification';
    public string $name = 'Verifications';
    public bool $status = true;
    public string $description = 'Verify NIN, BVN and others';
    public array $providers = ['nin', 'bvn'];
    public $has_verification = false;
    public bool $uses_api_setup = false;
    public $label_number = "NIN/BVN Number";
    public $label_provider = "Verification Type";
    public $label_service_type = "ID Card Type";
    public array $countries = ['all']; // Enable for all countries or specific? Default 'all'
    public $custom_id = 'bvn';

    public function __construct(wpdb $db, $service_type = false, ?int $user_id = null)
    {
        parent::__construct($db, $service_type, $user_id);
    }

    public function get_settings($cache = true, $global = true): array
    {
        global $wpdb;
        $raptor = $wpdb->get_row("SELECT * FROM {$wpdb->prefix}vtupress_raptor_settings LIMIT 1", ARRAY_A);
        if (!$raptor)
            return [];
        return $raptor;
    }

    public function form_fields(): array
    {
        return [];
    }


    public function render_frontend($user, $runcode = "")
    {
        $this->render_mutable_frontend($user, $runcode = "");
    }

    public function render_mutable_frontend($user, $runcode = "")
    {
        global $wpdb;
        $settings = $this->get_settings();
        $bvn_charge = $settings['custom_bvn_charge'] ?? 0;
        $nin_charge = $settings['custom_nin_charge'] ?? 0;
        ?>
        <style>
            .verification-card {
                background: #fff;
                border-radius: 12px;
                box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
                padding: 24px;
                border: 1px solid #eee;
            }

            .form-label {
                font-weight: 500;
                color: #374151;
                margin-bottom: 0.5rem;
            }

            .form-select,
            .form-control {
                border-radius: 8px;
                border: 1px solid #d1d5db;
                padding: 10px 14px;
                font-size: 14px;
                transition: border-color 0.15s ease-in-out, box-shadow 0.15s ease-in-out;
            }

            .form-select:focus,
            .form-control:focus {
                border-color: #6366f1;
                box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
                outline: none;
            }

            .btn-verify {
                background-color: #6366f1;
                border: none;
                color: white;
                padding: 12px;
                border-radius: 8px;
                font-weight: 600;
                width: 100%;
                transition: background-color 0.2s;
            }

            .btn-verify:hover {
                background-color: #4f46e5;
            }
        </style>

        <div class="container-md mt-4">
            <div class="row justify-content-center">
                <div class="col-md-8 col-lg-6">
                    <div class="verification-card">
                        <h4 class="mb-4 text-center">Identity Verification</h4>

                        <!-- Provider Select -->
                        <div class="mb-3">
                            <label class="form-label">Verification Type</label>
                            <select id="v_provider" class="form-select">
                                <option value="">Select Verification Type</option>
                                <option value="nin">NIN Verification</option>
                                <option value="bvn">BVN Verification</option>
                            </select>
                        </div>

                        <!-- Card Type -->
                        <div class="mb-3 d-none" id="card_type_group">
                            <label class="form-label">Card Design</label>
                            <select id="v_card_type" class="form-select">
                                <!-- Populated by JS -->
                            </select>
                        </div>

                        <!-- Number Input -->
                        <div class="mb-3">
                            <label class="form-label">Identity Number <span id="v_label"></span></label>
                            <input type="number" id="v_number" class="form-control" placeholder="Enter number here">
                            <div class="mt-1">
                                <small class="text-muted">Service Charge: <span id="v_charge"
                                        class="fw-bold">₦0.00</span></small>
                            </div>
                        </div>

                        <div class="mt-4">
                            <button id="btn_verify" class="btn-verify">Verify Identity</button>
                        </div>

                    </div>
                </div>
            </div>
        </div>

        <script>
            jQuery(document).ready(function ($) {
                const charges = {
                    'nin': <?php echo floatval($nin_charge); ?>,
                    'bvn': <?php echo floatval($bvn_charge); ?>
                };

                const cardTypes = {
                    'nin': [
                        { id: 'portrait_card', name: 'Portrait Card' },
                        { id: 'landscape_card', name: 'Landscape Card' },
                        { id: 'premium_card', name: 'Premium Card' },
                        // { id: 'standard', name: 'Standard Slip' },
                        // { id: 'basic_info', name: 'Basic Info Slip' }
                    ],
                    'bvn': [
                        { id: 'bvn_slip', name: 'BVN Slip' }
                    ]
                };

                $('#v_provider').on('change', function () {
                    const type = $(this).val();
                    const cardSelect = $('#v_card_type');

                    if (type && cardTypes[type]) {
                        $('#card_type_group').removeClass('d-none');
                        cardSelect.empty();
                        cardTypes[type].forEach(function (ct) {
                            cardSelect.append(new Option(ct.name, ct.id));
                        });

                        $('#v_charge').text('₦' + (charges[type] || 0));
                        $('#v_label').text('(' + type.toUpperCase() + ')');
                    } else {
                        $('#card_type_group').addClass('d-none');
                        $('#v_charge').text('₦0.00');
                        $('#v_label').text('');
                    }
                });

                $('#btn_verify').on('click', function () {
                    const provider = $('#v_provider').val();
                    const card = $('#v_card_type').val();
                    const number = $('#v_number').val();

                    if (!provider || !number) {
                        Swal.fire('Error', 'Please select a type and enter a number', 'error');
                        return;
                    }

                    Swal.fire({
                        title: 'Confirm Verification',
                        text: `You are about to verify ${provider.toUpperCase()} ${number} for ₦${charges[provider]}. Proceed?`,
                        icon: 'question',
                        showCancelButton: true,
                        confirmButtonText: 'Yes, Verify',
                        cancelButtonText: 'Cancel'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            performVerification(provider, card, number);
                        }
                    });
                });

                function performVerification(provider, card, number) {
                    const btn = $('#btn_verify');
                    const originalText = btn.text();
                    btn.prop('disabled', true).text('Verifying...');

                    $.post(vtupressobj.ajax_url, {
                        action: 'vtupress_ng_verification',
                        provider: provider,
                        type: card,
                        number: number
                    }, function (response) {
                        btn.prop('disabled', false).text(originalText);

                        try {
                            const res = JSON.parse(response);
                            if (res.status === 'success') {
                                Swal.fire('Success', res.message, 'success').then(() => {
                                    location.href = '?vend=history&page2=ng_verification';
                                });
                            } else {
                                Swal.fire('Failed', res.message, 'error');
                            }
                        } catch (e) {
                            Swal.fire('Error', 'Unexpected response from server', 'error');
                        }
                    });
                }
            });
        </script>
        <?php
    }

    public function process_frontend_request($request)
    {
        global $wpdb;
        $user_id = get_current_user_id();

        if (!$user_id) {
            return ['status' => 'error', 'message' => 'Please login to continue'];
        }

        $users = new Users();
        $user = $users->getUser($user_id, true, true);

        if (!$user) {
            return ['status' => 'error', 'message' => 'Users data not found in software tb'];
        }

        $user_id = $user["id"];
        $type = sanitize_text_field($request['provider'] ?? ''); // Provider is used as type (nin/bvn)
        $value = sanitize_text_field($request['number'] ?? ''); // Number
        $card_type = sanitize_text_field($request['type'] ?? 'portrait_card'); // Service Type as Card Template

        if (empty($type) || empty($value)) {
            return ['status' => 'error', 'message' => 'Type and Value are required'];
        }

        // Standardize type
        $type = strtolower($type);

        // Get User Info
        $fn = $user["first_name"];
        $ln = $user["last_name"];
        $phone = $user["phone"];
        $current_bal = $user["balance"];

        // Determine Charge
        $settings = $this->get_settings();
        $charge = 0;

        if ($type === 'bvn') {
            $charge = floatval($settings['custom_bvn_charge'] ?? 0);
        } elseif ($type === 'nin') {
            $charge = floatval($settings['custom_nin_charge'] ?? 0);
        } else {
            return ['status' => 'error', 'message' => 'Invalid Verification Type'];
        }

        if ($current_bal < $charge) {
            return ['status' => 'error', 'message' => 'Insufficient Balance'];
        }

        if (empty($settings['raptor_apiKey'])) {
            return ['status' => 'error', 'message' => 'Service Configuration Error (API Key)'];
        }

        $payload = [
            'verificationType' => $type,
            'firstName' => $fn,
            'lastName' => $ln,
            'phone' => $phone,
            'value' => $value,
            'includeCase' => true
        ];

        $headers = [
            'Authorization' => "Token " . ($settings['raptor_apiKey'] ?? ''),
            'cache-control' => "no-cache",
            'content-type' => "application/json",
            'connectionid' => $settings['raptor_connectionid'] ?? ''
        ];

        $url = "https://dashboard.raptor.ng/api/v1/verification/";

        $args = [
            'headers' => $headers,
            'timeout' => 120,
            'blocking' => true,
            'body' => json_encode($payload)
        ];

        // API Call
        $response = wp_remote_post($url, $args);

        if (is_wp_error($response)) {
            return ['status' => 'error', 'message' => $response->get_error_message()];
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (!$data || !isset($data['status'])) {
            return ['status' => 'error', 'message' => 'Invalid Response from Server'];
        }

        // Logic check: Raptor returns status:true for success
        if ($data['status'] !== true && $data['status'] !== "true") {
            return ['status' => 'error', 'message' => $data['message'] ?? 'Verification Failed'];
        }

        // Transaction & Debit
        // Re-check balance with lock? (Skipping complex lock for now as per bvn_verification.php precedent, but minimally check balance again)
        $current_bal = $user["balance"]; // Refetch
        if ($current_bal < $charge) {
            return ['status' => 'error', 'message' => 'Insufficient Balance'];
        }

        $new_bal = $current_bal - $charge;

        if ($charge > 0) {
            $debited_tnx_id = $users->debit($user_id, $charge, 'ng_verification', "Debited for $type [$value] verification", []);
            if (!$debited_tnx_id) {
                return ['status' => 'error', 'message' => 'Debit Failed'];
            }

            // // Log Transaction
            // $wpdb->insert($wpdb->prefix.'vtupress_wallet', [
            //     'name' => "$fn $ln",
            //     'type' => "Wallet",
            //     'description' => "Debited for " . strtoupper($type) . " verification",
            //     'fund_amount' => $charge,
            //     'before_amount' => $current_bal,
            //     'now_amount' => $new_bal,
            //     'user_id' => $user_id,
            //     'status' => "approved",
            //     'the_time' => current_time('mysql')
            // ]);
        }

        // Save Verification
        $inserted = $wpdb->insert($wpdb->prefix . 'vtupress_verifications', [
            'name' => "$fn $ln",
            'type' => strtoupper($type), // NIN / BVN
            'value' => $value,
            'fund_amount' => $charge,
            'card_type' => $card_type,
            'before_amount' => $current_bal,
            'now_amount' => $new_bal,
            'user_id' => $user_id,
            'vDatas' => str_replace("\/", "/", $body),
            'status' => "approved",
            'the_time' => current_time('mysql')
        ]);

        if ($inserted === false) {
            // Log error?
        }

        return ['status' => 'success', 'message' => 'Verification Successful!'];
    }
}

add_action('wp_ajax_vtupress_ng_verification', function () {
    $request = $_REQUEST;
    $service = Vtupress_Service_Container::get('ng_verification');
    if ($service && method_exists($service, 'process_frontend_request')) {
        $res = $service->process_frontend_request($request);
        echo json_encode($res);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Service not found or invalid']);
    }
    wp_die();
});
add_action('wp_ajax_nopriv_vtupress_ng_verification', function () {
    echo json_encode(['status' => 'error', 'message' => 'Login required']);
    wp_die();
});

add_action('wp_ajax_vtupress_view_verification', function () {
    global $wpdb;
    $id = intval($_REQUEST['id']);
    $table = $wpdb->prefix . 'vtupress_verifications';
    $row = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));

    if (!$row) {
        echo "Verification not found";
        wp_die();
    }

    $data = json_decode($row->vDatas, true);
    if (!$data || !isset($data['data'])) {
        echo "Invalid Data Format";
        wp_die();
    }

    $info = $data['data'];


    $photo = $info['photo'] ?? '';


    if(empty($photo) || strlen($photo) < 20){
        die("Image Not Found");
    }

    function formatPhoneNumber($number) {
    // Ensure the number is a string
    $number = strval($number);
    
    // Check the length of the number
    $length = strlen($number);
    
    if ($length == 11) {
        // Insert spaces into the string
        $formatted = substr($number, 0, 4) . ' ' . substr($number, 4, 3) . ' ' . substr($number, 7, 4);
        return $formatted;
    } else {
        // If the number is not 11 digits, return it as-is or handle as needed
        return $number;
    }
    }

    function getMonthNow($month){
        $month = intval($month);

        switch($month){
            case"1":
                $month = "JAN";
            break;
            case"2":
                $month = "FEB";
            break;
            case"3":
                $month = "MAR";
            break;
            case"4":
                $month = "APR";
            break;
            case"5":
                $month = "MAY";
            break;
            case"6":
                $month = "JUNE";
            break;
            case"7":
                $month = "JULY";
            break;
            case"8":
                $month = "AUG";
            break;
            case"9":
                $month = "SEPT";
            break;
            case"10":
                $month = "OCT";
            break;
            case"11":
                $month = "NOV";
            break;
            case"12":
                $month = "DEC";
            break;
        }
        return $month;
    }

    $rand = rand(1000,99999);



    // Map variables for template
    $fn = $info['firstName'] ?? '';
    $ln = $info['lastName'] ?? '';
    $mn = $info['middleName'] ?? '';

    $_dob = $info['birthdate'] ?? '';
    $original_dob = explode("-",$_dob);
    $o0 = $original_dob[0] ?? 1;
    $o1 = $original_dob[1] ?? 1;
    $o2 = $original_dob[2] ?? 1;
    $month = getMonthNow($o1);
    $dob = $o0." ".$month." ".$o2;
    $l_dob = $o2."-".$o1."-".$o0;

    $value = formatPhoneNumber($row->value); // Or $info['nin'] / $info['bvn']
    $lgar = $info['lgaOfResidence'] ?? 'N/A';
    $lgao = $info['lgaOfOrigin'] ?? 'N/A';
    $sor = $info['stateOfResidence'] ?? 'N/A';
    $gender = $info['gender'] ?? '';
    $phone = $info['phone'] ?? '';
    $email = $info['email'] ?? '';
    $ra = $info['residentialAddress'] ?? '';
    $date = explode("-",date("d-m-Y"));
    $trackingId = $info['trackingId'] ?? '--';
    $townOfResidence = $info['townOfResidence'] ?? '--';
    $title = $info['title'] ?? '--';
    $maritalStatus = $info['maritalStatus'] ?? '--';
    $enrollmentBank = $info['enrollmentBank'] ?? '--';
    $registrationDate = $info['registrationDate'] ?? '--';


    // Handle Photo (Base64)
    if (strpos($photo, 'data:image') === false) {
        $accountImage = 'data:image/jpeg;base64,' . $photo;
    } else {
        $accountImage = $photo;
    }

    // Card Type
    $card_type = $row->card_type;
    $template_path = VTUPRESS_PATH . '/images/id_cards/';
    $image_url = VTUPRESS_URL . '/images/id_cards/images/';
    $landscape_card = '';
    $premium_card = $image_url.'/image(1).png';

    //load css
    ?>
    <link rel="stylesheet"
        href="<?php echo WP_CONTENT_URL . "/plugins/vtupress-templates/templates/msorg/msorg_template/premiumcard.css?ver=769"; ?>">
    <?php


    // Determine Template
    $file = '';
    switch ($card_type) {
        case 'portrait_card':
            $file = 'portrait.html';
            break;
        case 'landscape_card':
            $file = 'landscape.html';
            break;
        case 'premium_card':
            $file = 'premiumcard.html';
            break;
        case 'slip':
            $file = 'standard.html';
            break; // 'slip' maps to standard.html usually? Or create slip.html? User said "Slip". portrait.html uses "nin_card" id.
        // Let's assume 'slip' -> 'standard.html' or 'basic_info.html'? 
        // Checking file list: bank_verification_slip.html, basic_info.html, bvn_slip.html, landscape.html, portrait.html, premiumcard.html, standard.html
        // NIN Slip usually refers to standard.html or basic_info.html. Let's try standard.html for now.
        case 'standard':
            $file = 'standard.html';
            break;
        case 'basic_info':
            $file = 'basic_info.html';
            break;
        case 'bvn_slip':
            $file = 'bvn_slip.html';
            break;
        default:
            $file = 'portrait.html';
            break;
    }

    if (file_exists($template_path . $file)) {
        include $template_path . $file;
        ?>
        <script>

            // document.addEventListener("DOMContentLoaded", function () {

                document.addEventListener('keydown', function (e) {
                    if (e.ctrlKey && (e.key === 'u' || e.key === 's' || e.key === 'i' || e.key === 'j' || e.key === 'p')) {
                        e.preventDefault();
                    }
                    if (e.key === 'F12' || e.key === 'F10') {
                        e.preventDefault();
                    }
                });

                document.addEventListener('contextmenu', function (event) {
                    event.preventDefault();
                });

                var text = "Fullname: " + jQuery(".nin_surname").text() + " " + jQuery(".nin_othername").text() + " | NIN: " + jQuery(".the_number").text();
                if (text) {
                    jQuery('.replace-qrcode').empty();


                    var canvas = document.createElement('canvas');
                    jQuery('.replace-qrcode').append(canvas);

                    qrVersion = 4;
                    var options = {
                        margin: 1
                    };

                    QRCode.toCanvas(canvas, text, options, function (error) {
                        if (error) {
                            console.log(error);
                        } else {
                            console.log('QR code generated!');
                        }
                    });
                } else {
                    console.log("Error with value");
                }



                jQuery(".download").on("click", function () {
                    var did = jQuery(this).attr("did");
                    var name = jQuery(this).attr("name");
                    var surname = jQuery(".nin_surname").text().toLowerCase();
                    var element = document.getElementById(did);

                    if (element) {
                        const opt = {
                            margin: 1,
                            filename: surname + '_' + name + '.pdf',
                            image: { type: 'jpeg', quality: 1.0 },
                            html2canvas: { scale: 2 },
                            jsPDF: { unit: 'mm', format: 'a4', orientation: 'portrait' }
                        };

                        // Generate and save the PDF
                        html2pdf(element, opt);
                    } else {
                        console.error('Element with id "areaID" not found.');
                    }

                });

            // });
        </script>


        <?php
    } else {
        echo "Template not found: $template_path . $file";
    }

    wp_die();
});

