<?php
if (!defined('ABSPATH')) {
    die("Not Allowed");
}

class Users
{
    private wpdb $db;
    private string $table;
    public function __construct()
    {
        global $wpdb;
        $this->db = $wpdb;
        $this->table = $wpdb->prefix . 'vtupress_users';
    }

    /* =========================================================
     * PIN VERIFICATION
     * ========================================================= */

    public function verifyPin(int $userId, int $pin): bool
    {
        $stored = $this->db->get_var(
            $this->db->prepare("SELECT pin FROM {$this->table} WHERE id=%d", $userId)
        );

        return ((int) $stored === (int) $pin);
    }

    public function verifyApi(string $apikey): int|false
    {
        $stored = $this->db->get_row(
            $this->db->prepare("SELECT id,apikey FROM {$this->table} WHERE apikey=%s", $apikey)
        );
        if (!$stored)
            return false;

        return ((string) $stored->apikey === (string) $apikey) ? $stored->id : false;
    }

    /* =========================================================
     * USER FETCH
     * ========================================================= */

    public function getUser(int $id, $array = false, $wp = false): mixed
    {
        if (Vtupress_option::get('vtupress_custom_active', 'no') != "yes")
            return [];
        
        $where = $wp ? "userId=%d" : "id=%d";
        return $this->db->get_row(
            $this->db->prepare("SELECT * FROM {$this->table} WHERE $where", $id),
            $array ? ARRAY_A : OBJECT
        );
    }

    /* =========================================================
     * BALANCE + AUTO LOGGING
     * ========================================================= */

    public function credit(
        int $userId,
        float|int $amount,
        string $type = 'fund',
        string $description = 'Wallet credit',
        array $details = []
    ): bool {
        do_action('vtupress_before_credit', $userId, $amount, $type);

        $before = $this->getBalance($userId);


        $updated = $this->db->query(
            $this->db->prepare(
                "UPDATE {$this->table}
                 SET balance = balance + %f
                 WHERE id = %d",
                $amount,
                $userId
            )
        );

        if (!$updated)
            return false;

        $after = $before + $amount;

        // $txn_id = $this->logWallet($userId, 'credit', $description, $amount, $before, $after);
        $this->logWallet($userId, 'credit', $description, $amount, $before, $after);
        // $details['type'] = 'credit';
        // $txn_id = $this->logTransaction($userId, $type, $amount, $before, $after, 'success', $details);

        do_action('vtupress_after_credit', $userId, $amount, $type);

        // if (!is_numeric($txn_id)) {
        //     return false;
        // }
        return true;
    }

    public function debit(
        int $userId,
        float|int $amount,
        string $type = 'purchase',
        string $description = 'Wallet debit',
        array $details = []
    ): int|bool {

        do_action('vtupress_before_debit', $userId, $amount, $type);


        $before = $this->getBalance($userId);

        if ($before < $amount)
            return false;

        $updated = $this->db->query(
            $this->db->prepare(
                "UPDATE {$this->table}
                 SET balance = balance - %f
                 WHERE id = %d",
                $amount,
                $userId
            )
        );

        if (!$updated)
            return false;

        $after = $before - $amount;

        $this->logWallet($userId, 'debit', $description, $amount, $before, $after);
        $details['type'] = 'debit';
        $txn_id = $this->logTransaction($userId, $type, $amount, $before, $after, 'success', $details);

        do_action('vtupress_after_debit', $userId, $amount, $type);
        if (!is_numeric($txn_id)) {
            return false;
        }

        return $txn_id;
    }

    private function logWallet($userId, $type, $name, $amount, $before, $after): int
    {
        $this->db->insert(
            $this->db->prefix . 'vtupress_wallet',
            [
                'type' => $type,
                'name' => $name,
                'fund_amount' => $amount,
                'before_amount' => $before,
                'now_amount' => $after,
                'user_id' => $userId,
                'the_time' => current_time('mysql'),
                'status' => 'success'
            ]
        );

        return $this->db->insert_id;

    }

    public function logTransaction($userId, $type, $amount, $before, $after, $status, $details = [], $recipient = 'system'): int
    {
        $this->db->insert(
            $this->db->prefix . 'vtupress_transactions',
            [
                'user_id' => $userId,
                'service' => $type,
                'amount' => $amount,
                'balance_before' => $before,
                'balance_now' => $after,
                'status' => $status,
                'created_at' => current_time('mysql'),
                'reference' => 'ref_' . uniqid(),
                'recipient' => $recipient,
                'details' => json_encode($details)
            ]
        );
        return $this->db->insert_id;
    }

    public function getBalance(int $userId): float
    {
        return (float) $this->db->get_var(
            $this->db->prepare("SELECT balance FROM {$this->table} WHERE id=%d", $userId)
        );
    }


    public function kycStatus(int $userId): string
    {
        return (string) $this->db->get_var(
            $this->db->prepare(
                "SELECT status FROM {$this->db->prefix}vtupress_kyc
                 WHERE user_id=%d ORDER BY id DESC LIMIT 1",
                $userId
            )
        ) ?: 'not_submitted';
    }

    public function kycRequiredForAmount(int $amount): bool
    {
        $settings = $this->db->get_row(
            "SELECT * FROM {$this->db->prefix}vtupress_kyc_settings WHERE id=1"
        );

        if (!$settings || $settings->enable !== 'yes')
            return false;

        return $amount >= intval($settings->kyc_limit);
    }



    public function update(int $userId, array $data): bool
    {
        if (empty($data)) {
            return false;
        }

        // Always update timestamp if column exists
        $data['updated_at'] = current_time('mysql');

        $updated = $this->db->update(
            $this->table,
            $data,
            ['id' => $userId]
        );

        return $updated !== false;
    }



    private function getReferralPercent(string $plan): float
    {
        $row = $this->db->get_row(
            $this->db->prepare(
                "SELECT referer_transaction_bonus
             FROM {$this->db->prefix}vtupress_levels
             WHERE name=%s AND status='active'",
                $plan
            )
        );

        return $row ? ((float) $row->referer_transaction_bonus / 100) : 0.0;
    }

    /* =========================================================
     * REFERRAL BONUS ENGINE
     * ========================================================= */


    public function applyReferralBonus(int $userId, int $amount): void
    {
        $user = $this->getUser($userId);
        if (!$user || empty($user->referrer_id))
            return;

        $referrer = $this->getUser((int) $user->referrer_id);
        if (!$referrer)
            return;

        $percent = $this->getReferralPercent($referrer->plan);
        if ($percent <= 0)
            return;

        $bonus = (int) round($amount * $percent);

        do_action('vtupress_before_referral_bonus', $userId, $referrer->id, $bonus);

        $this->credit(
            $referrer->id,
            $bonus,
            'referral',
            'Referral transaction bonus'
        );

        $this->db->query(
            $this->db->prepare(
                "UPDATE {$this->table}
             SET referral_commission = referral_commission + %d
             WHERE id = %d",
                $bonus,
                $referrer->id
            )
        );

        do_action('vtupress_after_referral_bonus', $userId, $referrer->id, $bonus);
    }



    /* =========================================================
     * KYC
     * ========================================================= */
    public function submitKyc(
        int $userId,
        string $method,
        string $selfie,
        string $proof
    ): bool {
        $this->db->insert(
            $this->db->prefix . 'vtupress_kyc',
            [
                'user_id' => $userId,
                'method' => $method,
                'selfie' => $selfie,
                'proof' => $proof,
                'status' => 'pending',
                'the_time' => current_time('mysql')
            ]
        );

        do_action('vtupress_kyc_submitted', $userId);

        return true;
    }

    public function maybeAutoApproveKyc(int $userId, int $amount): void
    {
        $settings = $this->db->get_row(
            "SELECT * FROM {$this->db->prefix}vtupress_kyc_settings WHERE id=1"
        );

        if (!$settings || $settings->enable !== 'yes')
            return;

        if ($amount < (int) $settings->kyc_limit) {
            $this->approveKyc($userId, 'auto');
        }
    }

    public function approveKyc(int $userId, string $source = 'manual'): void
    {
        $this->db->update(
            $this->db->prefix . 'vtupress_kyc',
            ['status' => 'approved'],
            ['user_id' => $userId]
        );

        do_action('vtupress_kyc_approved', $userId, $source);
    }

    public function refund(
        int $userId,
        int $amount,
        string $reason = 'Transaction reversal',
        string $originalRef = ''
    ): bool {
        $before = $this->getBalance($userId);
        $after = $before + $amount;

        do_action('vtupress_before_refund', $userId, $amount, $originalRef);

        $this->db->query(
            $this->db->prepare(
                "UPDATE {$this->table}
             SET balance = balance + %d
             WHERE id=%d",
                $amount,
                $userId
            )
        );

        $this->logWallet(
            $userId,
            'refund',
            $reason,
            $amount,
            $before,
            $after
        );

        $this->logTransaction(
            $userId,
            'refund',
            $amount,
            $before,
            $after,
            'success',
            ['type' => 'credit', 'reason' => $reason]
        );

        do_action('vtupress_after_refund', $userId, $amount, $originalRef);

        return true;
    }





/**
 * EXAMPLES
 */

    public function syncUsers($runSync = false)
    {
        global $wpdb;
        $vtupress_table = $wpdb->prefix . 'vtupress_users';
        $wp_users_table = $wpdb->users; // multisite-safe
        if ((isset($_GET['sync']) || $runSync) && current_user_can('manage_options')) {

            // 1️⃣ Get all WordPress user IDs
            $wp_user_ids = $wpdb->get_col("
            SELECT ID 
            FROM {$wp_users_table}
        ");

            if (empty($wp_user_ids)) {
                return;
            }

            // 2️⃣ Get already synced userIds
            $synced_user_ids = $wpdb->get_col("
            SELECT userId 
            FROM {$vtupress_table}
        ");

            // Normalize to strings for safe comparison
            $synced_user_ids = array_map('strval', $synced_user_ids);

            // 3️⃣ Find users not yet synced
            $missing_user_ids = array_diff(
                array_map('strval', $wp_user_ids),
                $synced_user_ids
            );

            if (empty($missing_user_ids)) {
                return;
            }

            // 4️⃣ Fetch missing users details
            $placeholders = implode(',', array_fill(0, count($missing_user_ids), '%d'));

            $users = $wpdb->get_results(
                $wpdb->prepare(
                    "
                SELECT ID, user_email, user_login, display_name
                FROM {$wp_users_table}
                WHERE ID IN ($placeholders)
                ",
                    $missing_user_ids
                ),
                ARRAY_A
            );

            // 5️⃣ Insert into vtupress_users
            foreach ($users as $user) {

                // Example usage
                $apiKey = function_exists('generateApiKey') ? generateApiKey() : wp_generate_password(32, false);
                $userD = get_user_meta($user['ID'], "vp_user_data", true);
                if (!empty($userD)) {
                    $userD = json_decode($userD, true);
                } else {
                    $userD = [];
                }
                $bal = $userD["vp_bal"] ?? 0;
                $inserted = $wpdb->insert(
                    $vtupress_table,
                    [
                        'userId' => $user['ID'],
                        'apikey' => $userD["apikey"] ?? $apiKey,
                        'balance' => is_numeric($bal) ? (int) $bal : 0,
                        'phone' => is_numeric($userD["phone"] ?? '') ? (int) $userD["phone"] ?? '' : '',
                        'pin' => is_numeric($userD["pin"] ?? '') ? (int) $userD["pin"] ?? 1234 : 1234,
                        'first_name' => $userD["first_name"] ?? '',
                        'last_name' => $userD["last_name"] ?? '',
                        'referrer_id' => $userD["vp_who_ref"] ?? '',
                        'referral_commission' => $userD["vp_tot_ref_earn"] ?? '',
                        'indirect_referral_commission' => $userD["vp_tot_in_ref_earn"] ?? '',
                        'other_referral_commission' => $userD["vp_tot_in_ref_earn3"] ?? '',
                        'email_verified' => !is_numeric($userD["email_verified"] ?? 'no') ? $userD["email_verified"] ?? 'no' : 'no',
                        'beneficiaries' => $userD["beneficiaries"] ?? '',
                        'kyc_status' => $userD["kyc_status"] ?? '',
                        'access_status' => $userD["access_status"] ?? 'blocked',
                        'bvn' => $userD["bvn"] ?? '',
                        'nin' => $userD["nin"] ?? ''
                    ],
                    [
                        '%d',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s',
                        '%s'
                    ]
                );

                if ($inserted) {
                    // delete_user_meta($user['ID'], "vp_user_data");
                }
            }
        }
    }

}
/*
$users = new Users();

if (!$users->verifyPin($userId, $pin)) {
    wp_die('Invalid PIN');
}

if ($users->kycRequiredForAmount($amount) && $users->kycStatus($userId) !== 'approved') {
    wp_die('KYC required');
}

$users->debit($userId, $amount, 'airtime', 'Airtime purchase');
$users->applyReferralBonus($userId, $amount);



$users = new Users();

/* Create *
$userId = $users->create([
    'phone' => '08012345678',
    'apikey' => wp_generate_password(32, false),
]);

/* Credit *
$users->credit($userId, 5000);

/* Debit *
$users->debit($userId, 1000);

/* Fetch *
$user = $users->getById($userId);


add_action('vtupress_before_debit', function ($userId, $amount) {
    $users = new Users();

    if ($users->kycRequiredForAmount($amount)
        && $users->kycStatus($userId) !== 'approved') {
        wp_die('KYC required for this transaction');
    }
}, 10, 2);


add_action('vtupress_before_referral_bonus', function ($from, $to, &$bonus) {
    if ($bonus > 5000) {
        $bonus = 5000; // cap bonus
    }
});


add_action('vtupress_after_refund', function ($userId, $amount) {
    error_log("Refunded ₦{$amount} to user {$userId}");
});

*/